
package com.ebmwebsourcing.easybox.impl;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.bind.JAXBElement;
import javax.xml.bind.annotation.XmlAnyAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementRef;
import javax.xml.bind.annotation.XmlElementRefs;
import javax.xml.bind.annotation.XmlElements;
import javax.xml.bind.annotation.XmlEnum;
import javax.xml.namespace.QName;

import com.ebmwebsourcing.easybox.api.ModelObject;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.easycommons.reflect.ReflectionHelper;

final class JaxbReflector {

    private JaxbReflector() {
    }

    static ModelObject[] collectJaxbChildren(ModelObject modelObject) {
        List<ModelObject> children = new ArrayList<ModelObject>();
        JaxbReflectionInfo reflectionInfo = JaxbReflectionInfo.getReflectionInfo(modelObject
                .getClass());
        for (JaxbReflectionFieldInfo fieldInfo : reflectionInfo.getChildrenFieldInfos()) {
            Object obj = ReflectionHelper.getFieldValue(modelObject, fieldInfo.getField());
            if (obj == null) {
                continue;
            } else if (obj instanceof Collection<?>) {
                for (Object o : ((Collection<?>) obj)) {
                    ModelObject childModelObject = ModelObjectFactory.createModelObject(
                            modelObject, o);
                    if (childModelObject == null)
                        continue;
                    children.add(childModelObject);
                }
            } else {
                ModelObject childModelObject = ModelObjectFactory.createModelObject(modelObject,
                        obj);
                if (childModelObject == null) {
                    continue;
                }
                children.add(childModelObject);
            }
        }
        return children.toArray(new ModelObject[children.size()]);
    }

    static void setJaxbChild(ModelObject parentModelObject, ModelObject childModelObject,
            QName elementQName) {
        assert parentModelObject != null;
        assert elementQName != null;
        JaxbReflectionInfo reflectionInfo = JaxbReflectionInfo.getReflectionInfo(parentModelObject
                .getClass());
        assert reflectionInfo != null;
        JaxbReflectionFieldInfo reflectionFieldInfo = reflectionInfo
                .getChildFieldInfoByName(elementQName.getLocalPart());
        Method setterMethod = reflectionFieldInfo.getSetterMethod();
        try {
            ReflectionHelper.invokeMethod(parentModelObject, setterMethod, childModelObject);
        } catch (InvocationTargetException e) {
            throw new UncheckedException("Error while setting JAXB child.", e);
        }
    }

    @SuppressWarnings( { "unchecked" })
    static Map<QName, Object> collectJaxbAttributes(ModelObject ModelObject) {
        Map<QName, Object> attributes = new HashMap<QName, Object>();
        for (JaxbReflectionFieldInfo fieldInfo : JaxbReflectionInfo.getReflectionInfo(
                ModelObject.getClass()).getAttributeFields()) {
            Object obj = getFieldValueOrDefaultValue(ModelObject, fieldInfo);
            if (obj == null)
                continue;
            if (fieldInfo.getField().isAnnotationPresent(XmlAnyAttribute.class)) {
                assert obj instanceof Map<?, ?>;
                Map<QName, String> anyAttributes = (Map<QName, String>) obj;
                attributes.putAll(anyAttributes);
            } else if (obj instanceof Collection<?>) {
                attributes.put(new QName(fieldInfo.getName()), obj);
            } else if (obj.getClass().isAnnotationPresent(XmlEnum.class)) {
                Method valueMethod = ReflectionHelper.getPublicMethod(obj.getClass(), "value");
                String attributeValue;
                try {
                    attributeValue = (String) ReflectionHelper.invokeMethod(obj, valueMethod);
                } catch (InvocationTargetException e) {
                    throw new UncheckedException(
                            String
                                    .format(
                                            "Cannot invoke reflectively '%s' method on class '%s' (InvocationTargetException).",
                                            valueMethod.getName(), obj.getClass().getSimpleName()),
                            e);
                }
                attributes.put(new QName(fieldInfo.getName()), attributeValue);
            } else {
                attributes.put(new QName(fieldInfo.getName()), obj);
            }
        }
        return Collections.unmodifiableMap(attributes);
    }

    private static Object getFieldValueOrDefaultValue(ModelObject modelObject,
            JaxbReflectionFieldInfo fieldInfo) {

        Object value = ReflectionHelper.getFieldValue(modelObject, fieldInfo.getField());
        final Method getterMethod = fieldInfo.getGetterMethod();
        if (value == null) {
            // try to get default value.
            try {
                value = ReflectionHelper.invokeMethod(modelObject, getterMethod);
            } catch (InvocationTargetException e) {
                if (e.getCause() instanceof NullPointerException) {
                    // CAUTION : we should not do anything here, but we must
                    // absorb JAXB bugs :
                    // NullPointerException can actually occur in a simple
                    // getter...
                    value = null;
                } else {
                    throw new UncheckedException(
                            String
                                    .format(
                                            "Cannot invoke reflectively '%s' method on class '%s' (InvocationTargetException).",
                                            getterMethod.getName(), modelObject.getClass()
                                                    .getSimpleName()), e);
                }
            }

            if (value != null) {
                // set back old value to prevent side effects of having asked
                // default value.
                ReflectionHelper.setFieldValue(modelObject, fieldInfo.getField(), null);
                return value;
            }
        }
        return value;
    }

    static Field findFieldContainingChild(ModelObject parent, Object child) {
        for (JaxbReflectionFieldInfo fieldInfo : JaxbReflectionInfo.getReflectionInfo(
                parent.getClass()).getChildrenFieldInfos()) {
            Object obj = ReflectionHelper.getFieldValue(parent, fieldInfo.getField());
            if (obj == null)
                continue;
            if (obj instanceof JAXBElement<?>) {
                obj = ((JAXBElement<?>) obj).getValue();
            }
            if (obj == child) {
                return fieldInfo.getField();
            } else if (obj instanceof Collection<?>) {
                // if obj contained in a Collection field, return it as well.
                Iterator<?> it = ((Collection<?>) obj).iterator();
                while (it.hasNext()) {
                    Object itObj = it.next();
                    if (itObj instanceof JAXBElement<?>) {
                        itObj = ((JAXBElement<?>) itObj).getValue();
                    }
                    if (itObj == child) {
                        return fieldInfo.getField();
                    }
                }
            }
        }
        return null;
    }

    private static QName guessQNameFromField(AbstractJaxbModelObject child, Field field) {
        String name = null;
        if (field.isAnnotationPresent(XmlElement.class)) {
            name = field.getAnnotation(XmlElement.class).name();
        } else if (field.isAnnotationPresent(XmlElementRef.class)) {
            name = field.getAnnotation(XmlElementRef.class).name();
        } else {
            name = field.getName();
        }

        if ("##default".equals(name)) {
            name = field.getName();
        }
        return new QName(child.getOriginatingNamespaceURI(), name);
    }

    private static QName guessQNameFromCollectionField(AbstractJaxbModelObject child, Field field) {
        String defaultName = null;
        if (field.isAnnotationPresent(XmlElementRefs.class)) {
            XmlElementRefs xmlElementRefs = field.getAnnotation(XmlElementRefs.class);
            assert xmlElementRefs != null;
            for (XmlElementRef xer : xmlElementRefs.value()) {
                if (xer.type().equals(child.getClass())) {
                    return new QName(child.getOriginatingNamespaceURI(), xer.name());
                } else if (XmlElement.DEFAULT.class.equals(xer.type())) {
                    // keep default name for later, when we will be sure that no
                    // subtype
                    // match more closely.
                    defaultName = xer.name();
                }
            }
            assert defaultName != null;
            return new QName(child.getOriginatingNamespaceURI(), defaultName);
        } else if (field.isAnnotationPresent(XmlElements.class)) {
            XmlElements xmlElements = field.getAnnotation(XmlElements.class);
            assert xmlElements != null;
            for (XmlElement xe : xmlElements.value()) {
                if (xe.type().equals(child.getClass())) {
                    return new QName(child.getOriginatingNamespaceURI(), xe.name());
                } else if (XmlElement.DEFAULT.class.equals(xe.type())) {
                    // keep default name for later, when we will be sure that no
                    // subtype
                    // match more closely.
                    defaultName = xe.name();
                }
            }
            assert defaultName != null;
            return new QName(child.getOriginatingNamespaceURI(), defaultName);
        } else if (isFieldCollectionOfJaxbElement(field)) {
            return guessQNameFromJaxbElementCollectionField(child, field);
        }

        return new QName(child.getOriginatingNamespaceURI(), field.getName());
    }

    static QName guessQNameFromParent(AbstractJaxbModelObject child, AbstractJaxbModelObject parent) {
        assert child != null;
        if (parent == null)
            return null;
        Field fieldContainingChild = findFieldContainingChild(parent, child);
        if (fieldContainingChild != null) {
            if (fieldContainingChild.getType().isAssignableFrom(child.getClass())) {
                return guessQNameFromField(child, fieldContainingChild);
            } else if (Collection.class.isAssignableFrom(fieldContainingChild.getType())) {
                return guessQNameFromCollectionField(child, fieldContainingChild);
            }
        }
        return null;
    }

    static boolean isFieldCollectionOfJaxbElement(Field field) {
        if (Collection.class.isAssignableFrom(field.getType())) {
            Type fieldType = field.getGenericType();
            Type[] parameterTypes = ((ParameterizedType) fieldType).getActualTypeArguments();
            Type parameterType = parameterTypes[0];
            return parameterType instanceof ParameterizedType
                    && ((ParameterizedType) parameterType).getRawType().equals(JAXBElement.class);
        }
        return false;
    }

    static QName guessQNameFromJaxbElementCollectionField(AbstractJaxbModelObject child, Field field) {
        if (field.isAnnotationPresent(XmlElementRef.class)) {
            XmlElementRef ref = field.getAnnotation(XmlElementRef.class);
            if (ref.type().equals(JAXBElement.class)) {
                Type fieldType = field.getGenericType();
                Type[] listTypes = ((ParameterizedType) fieldType).getActualTypeArguments();
                assert listTypes.length > 0;
                Type jaxbeltType = listTypes[0];
                if (jaxbeltType instanceof ParameterizedType
                        && ((ParameterizedType) jaxbeltType).getRawType().equals(JAXBElement.class)) {// JAXBElement
                    Type[] types = ((ParameterizedType) jaxbeltType).getActualTypeArguments();
                    assert types.length > 0;
                    if (types[0].equals(Object.class)) {
                        return new QName(ref.namespace(), ref.name());
                    } else {
                        return null;
                    }
                }
            }
        }
        return null;
    }

}
