package com.ebmwebsourcing.easybox.impl;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.xml.bind.Binder;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;

import org.w3c.dom.Node;

import com.ebmwebsourcing.easybox.api.XmlContext;
import com.ebmwebsourcing.easybox.api.XmlObjectBinding;
import com.ebmwebsourcing.easybox.api.XmlObjectFactory;
import com.ebmwebsourcing.easybox.api.XmlObjectReader;
import com.ebmwebsourcing.easybox.api.XmlObjectSchemaBinding;
import com.ebmwebsourcing.easybox.api.XmlObjectValidator;
import com.ebmwebsourcing.easybox.api.XmlObjectWriter;
import com.ebmwebsourcing.easybox.api.XmlObjectXPathEvaluator;
import com.ebmwebsourcing.easybox.api.XmlObjectXQueryEvaluator;
import com.ebmwebsourcing.easybox.api.XmlObjectXQueryValidator;
import com.ebmwebsourcing.easybox.api.analysis.Analyzer;
import com.ebmwebsourcing.easybox.api.analysis.AnalyzerException;
import com.ebmwebsourcing.easybox.api.analysis.ClassMetadata;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.easycommons.lang.reflect.ReflectionHelper;
import com.ebmwebsourcing.easycommons.xml.DefaultNamespaceContext;

public final class XmlContextImpl implements XmlContext {

    private static Logger LOG = Logger
            .getLogger(XmlContextImpl.class.getName());

    private Map<String, XmlObjectBinding> bindings;
    private XmlObjectFactory xmlObjectFactory;
    private JAXBContext jaxbContext;
    private ClassMetadata classMetadata;
    private DefaultNamespaceContext namespaceContext;

    public XmlContextImpl() {
        bindings = new HashMap<String, XmlObjectBinding>();
        xmlObjectFactory = null;
        jaxbContext = null;
        classMetadata = new ClassMetadata();
        namespaceContext = new DefaultNamespaceContext();
    }

    @Override
    public XmlObjectBinding[] getXmlObjectBindings() {
        return bindings.values().toArray(new XmlObjectBinding[bindings.size()]);
    }

    @Override
    public void addXmlObjectBindings(String... bindingNames) {
        clearXmlObjectBindings();

        List<Class<?>> jaxbFactoryClasses = new LinkedList<Class<?>>();

        ServiceLoader<XmlObjectBinding> sl = ServiceLoader
                .load(XmlObjectBinding.class);
        List<String> bindingNamesList = Arrays.asList(bindingNames);
        Iterator<XmlObjectBinding> it = sl.iterator();
        while (it.hasNext()) {
            XmlObjectBinding xob = it.next();
            if ((bindingNames.length == 0)
                    || (bindingNamesList.contains(xob.getName()))) {
                bindings.put(xob.getName(), xob);
                Analyzer analyzer = new Analyzer(xob, classMetadata);
                try {
                    analyzer.analyze();
                } catch (AnalyzerException e) {
                    throw new UncheckedException(e);
                }

                if (!(xob instanceof XmlObjectSchemaBinding))
                    continue;
                XmlObjectSchemaBinding xosb = (XmlObjectSchemaBinding) xob;
                namespaceContext.bindNamespace(
                        xosb.getOriginatingSchemaPreferredNamespacePrefix(),
                        xosb.getOriginatingSchemaNamespaceURI());

                String jaxbFactoryClassName = xosb.getModelObjectPackage()
                        .getName() + ".ObjectFactory";
                try {
                    Class<?> jaxbFactoryClass = Class
                            .forName(jaxbFactoryClassName);
                    // ensure that there is one and only one instance of
                    // ObjectFactory class in classpath.
                    String jaxbFactoryClassFile = jaxbFactoryClassName
                            .replaceAll("\\.", "/") + ".class";

                    List<?> jaxbFactoryClassFileUrls = Collections.EMPTY_LIST;
                    try {
                        jaxbFactoryClassFileUrls = Collections.list(getClass()
                                .getClassLoader().getResources(
                                        jaxbFactoryClassFile));
                    } catch (IOException ioe) {
                        LOG.log(Level.SEVERE,
                                String.format(
                                        "Several class files (%d) for '%s' are found in classpath, JAXB behaviour is unpredictable.",
                                        jaxbFactoryClassFileUrls.size(),
                                        jaxbFactoryClass.getName()), ioe);
                    }
                    jaxbFactoryClasses.add(jaxbFactoryClass);
                } catch (ClassNotFoundException cnfe) {
                    throw new UncheckedException(String.format(
                            "Cannot found JAXB factory class '%s'.",
                            jaxbFactoryClassName), cnfe);
                }

                LOG.config(String.format(
                        "Registered JAXB schema binding : '%s' => '%s'.", xosb
                                .getOriginatingSchemaNamespaceURI(), xosb
                                .getModelObjectPackage().getName()));

            }
        }
        try {
            jaxbContext = JAXBContext.newInstance(jaxbFactoryClasses
                    .toArray(new Class<?>[jaxbFactoryClasses.size()]));
        } catch (JAXBException je) {
            throw new UncheckedException(
                    "Problem while instantiating JAXB context.", je);
        }
        xmlObjectFactory = null;
    }

    @Override
    public void removeXmlObjectBindings(String... bindingNames) {
        for (String bindingName : bindingNames) {
            bindings.remove(bindingName);
        }
        xmlObjectFactory = null;
    }

    @Override
    public void clearXmlObjectBindings() {
        bindings.clear();
        classMetadata.clear();
        xmlObjectFactory = null;
        namespaceContext = new DefaultNamespaceContext();
    }

    @Override
    public XmlObjectFactory getXmlObjectFactory() {
        if (xmlObjectFactory == null) {
            xmlObjectFactory = new XmlObjectFactoryImpl(this);
        }
        return xmlObjectFactory;
    }

    JAXBContext getJaxbContext() {
        return jaxbContext;
    }

    Binder<Node> newBinder() {
        Binder<Node> binder = jaxbContext.createBinder();
        Method getUnmarshallerMethod = ReflectionHelper.getDeclaredMethod(binder.getClass(), "getUnmarshaller");
        getUnmarshallerMethod.setAccessible(true);
        try {
            Unmarshaller unmarshaller;
            unmarshaller = (Unmarshaller) ReflectionHelper.invokeMethod(binder,
                    getUnmarshallerMethod);
            unmarshaller.setListener(new JaxbAddExtraInfoUnmarshallListener(
                    binder));
        } catch (InvocationTargetException ite) {
            throw new UncheckedException(
                    "Cannot invoke 'getUnmarshaller' method on Binder class.",
                    ite);
        }
        return binder;
    }

    @Override
    public XmlObjectReader createReader() {
        return new XmlObjectReaderImpl(this);
    }

    @Override
    public XmlObjectValidator createValidator() {
        return new XmlObjectValidatorImpl(this);
    }

    @Override
    public XmlObjectWriter createWriter() {
        return new XmlObjectWriterImpl(this);
    }

    @Override
    public XmlObjectXPathEvaluator createXPathEvaluator() {
        return new XmlObjectXPathEvaluatorImpl(namespaceContext);
    }

    @Override
    public ClassMetadata getClassMetadata() {
        return classMetadata;
    }

    @Override
    public XmlObjectXQueryEvaluator createXQueryEvaluator() {
        return new XmlObjectXQueryEvaluatorImpl(namespaceContext);
    }

    @Override
    public XmlObjectXQueryValidator createXQueryValidator() {
        return new XmlObjectXQueryValidatorImpl(namespaceContext);
    }

}
