package com.ebmwebsourcing.easybox.impl;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;

import javax.xml.xpath.XPathExpressionException;

import com.ebmwebsourcing.easybox.api.Catalog;
import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectReadException;
import com.ebmwebsourcing.easybox.api.XmlObjectReader;
import com.ebmwebsourcing.easybox.api.XmlObjectXPathEvaluator;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.easycommons.uri.URIHelper;


final class XmlObjectAdoptionMediator {

	private final Map<String, XmlObject> adoptedChildrenCache;

	XmlObjectAdoptionMediator(XmlObject xmlObject) {
		this.adoptedChildrenCache = new HashMap<String, XmlObject>();
		URI baseURI = xmlObject.getXmlObjectBaseURI();
		if (baseURI != null) {
			adoptedChildrenCache.put(makeAdoptedChildKey(baseURI, ""), xmlObject.getXmlObjectBaseRoot());
		}
	}

	private String makeAdoptedChildKey(URI baseURI, String xpath) {
		assert baseURI != null;
		return baseURI.toString() + "#" + xpath; 
	}


	private XmlObject resolveXPath(XmlObject contextXmlObject, String xpath) {
		if (xpath.isEmpty()) return contextXmlObject;
		XmlObjectXPathEvaluator xpe = contextXmlObject.getXmlContext().createXPathEvaluator();
		try {
			XmlObject childToAdopt = xpe.selectSingleXmlObjectNode(contextXmlObject, xpath, XmlObject.class);
			return childToAdopt;
		} catch (XPathExpressionException xpee) {
			throw new UncheckedException(String.format("Invalid XPath expression '%s'.", xpath));
		}
	}



	private XmlObject adoptChild(XmlObject adoptiveParent, URI sourceUri, String xpath) throws InadequateChildAdoptionException {
		assert adoptiveParent != null;
		assert sourceUri != null;
		if (xpath == null) xpath = "";
		String key = makeAdoptedChildKey(sourceUri, xpath);
		if (adoptedChildrenCache.containsKey(key)) {
			return adoptedChildrenCache.get(key);
		}
		XmlObject childToAdopt;        
		URI baseURI = adoptiveParent.getXmlObjectBaseURI();
		// FIXME : we can allow baseURI == null if and only if sourceURI is absolute (it should have been asserted before).
		if (baseURI == null) {
			throw new InadequateChildAdoptionException("Cannot adopt child because adoptive parent has no base URI");
		}
		if (baseURI.equals(sourceUri)) {
			// internal adoption.
			childToAdopt = resolveXPath(adoptiveParent.getXmlObjectRoot(), xpath);
		} else {
			// external adoption.
			XmlObjectReader reader = adoptiveParent.getXmlContext().createReader();

			// created reader is set in lazy mode, otherwise we might fall in an infinite recursion.
			reader.setReadMode(XmlObjectReader.ReadMode.LAZY);
			try {
				XmlObject readXmlObject = reader.readDocument(sourceUri.toURL(), XmlObject.class);
				childToAdopt = resolveXPath(readXmlObject, xpath);
			} catch (MalformedURLException e) {
				throw new UncheckedException(
						String.format("Cannot adopt child from URI '%s' and XPath '%s' (MalformedURLException).", 
								sourceUri.toString(), xpath), e); 
			} catch (XmlObjectReadException e) {
				throw new UncheckedException(
						String.format("Cannot adopt child from URI '%s' and XPath '%s' (XmlObjectReadException).", 
								sourceUri.toString(), xpath), e); 
			}

		}
		assert childToAdopt != null;
		// System.err.println("ADD TO CACHE : " + key);
		adoptedChildrenCache.put(key, childToAdopt);
		return childToAdopt;
	}

	XmlObject adoptChild(XmlObject parentAdopter, String publicId,
			String systemId, Object object) throws InadequateChildAdoptionException {
		assert parentAdopter != null;
		URI resolvedUri = null;
		try {
			
			if (publicId != null) {
				resolvedUri = Catalog.getInstance().resolveUri(publicId);
			}
			if (resolvedUri == null) {
				if (systemId == null) {
					// in last resort, if systemId is null, and catalog does not
					// know about public id, try to use public id as system id.
					systemId = publicId;
				} 
				final URI baseURI = parentAdopter.getXmlObjectBaseURI();

				URI newUri = URIHelper.resolve(baseURI, systemId);

				// FIXME : newURI must be absolute at this level.
				
				resolvedUri = Catalog.getInstance().resolveUri(newUri);
				if (resolvedUri == null)
					resolvedUri = newUri;
			}
		} catch (MalformedURLException e) {
			throw new InadequateChildAdoptionException(e.getMessage());
		} catch (URISyntaxException e) {
			throw new InadequateChildAdoptionException(e.getMessage());
		}
		return adoptChild(parentAdopter, resolvedUri, null);
	}


}
