/**
* easy BPEL software - Copyright (c) 2009 PetalsLink, 
* http://www.petalslink.com/ 
*  
* This library is free software; you can redistribute it and/or modify it under 
* the terms of the GNU Lesser General Public License as published by the Free 
* Software Foundation; either version 2.1 of the License, or (at your option) 
* any later version. This library is distributed in the hope that it will be 
* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
* General Public License for more details. 
*  
* You should have received a copy of the GNU Lesser General Public License 
* along with this library; if not, write to the Free Software Foundation, Inc., 
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */ 
package com.ebmwebsourcing.easybpel.model.bpel.impl.registry;

import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import org.objectweb.fractal.fraclet.annotations.Interface;
import org.objectweb.fractal.fraclet.extensions.Membrane;
import org.ow2.easywsdl.extensions.wsdl4bpel.api.WSDL4BPELException;
import org.ow2.easywsdl.wsdl.api.Endpoint;
import org.ow2.easywsdl.wsdl.api.InterfaceType;
import org.ow2.easywsdl.wsdl.api.abstractItf.AbsItfInterfaceType;

import com.ebmwebsourcing.easybpel.model.bpel.api.BPELProcess;
import com.ebmwebsourcing.easybpel.model.bpel.api.registry.BPELRegistry;
import com.ebmwebsourcing.easybpel.model.bpel.impl.compiler.validation.StaticAnalysisImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.compiler.validation.StaticAnalysisPrinter;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Process;
import com.ebmwebsourcing.easyviper.core.api.model.registry.ProcessKey;
import com.ebmwebsourcing.easyviper.core.api.model.registry.definition.AbstractProcessDefinitionRegistry;
import com.ebmwebsourcing.easyviper.core.api.model.registry.definition.ProcessContextDefinition;
import com.ebmwebsourcing.easyviper.core.impl.engine.registry.MemoryProcessInstanceRegistryImpl;
import com.ebmwebsourcing.easyviper.core.impl.model.registry.ProcessKeyImpl;

/**
 * @author Nicolas Salatge - eBM WebSourcing
 */
@org.objectweb.fractal.fraclet.annotations.Component(provides = @Interface(name = "service", signature = BPELRegistry.class))
@Membrane(controller="primitive")
public class BPELMemoryRegistryImpl extends AbstractProcessDefinitionRegistry<BPELProcess> implements BPELRegistry  {

	/**
	 *
	 */
	private static final long serialVersionUID = 1L;

	//private static final int POOL_SIZE = 5;

	private Logger log = Logger.getLogger(BPELMemoryRegistryImpl.class.getName());

	Map<ProcessKey, BPELProcess> registry = new HashMap<ProcessKey, BPELProcess>();




	public BPELProcess getProcessDefinition(final ProcessKey name) {
		return this.registry.get(name);
	}

	public List<BPELProcess> getAllProcessDefinitions() {
		return new ArrayList<BPELProcess>(this.registry.values());
	}

	public BPELProcess removeProcessDefinition(final ProcessKey name, final boolean forceUninstall) throws CoreException {
		BPELProcess res = null;
		this.log.finest("Map of instance: " +((MemoryProcessInstanceRegistryImpl)this.getModel().getCore().getEngine().getProcessInstanceRegistry()).getMap());


		// Delete all instances if they are in state ended 
		Iterator<Process> it = this.getModel().getCore().getEngine().getProcessInstanceRegistry().getProcessInstances(name).iterator();
		final List<Process> undeleteProcess = new ArrayList<Process>();
		while(it.hasNext()) {
			final Process process = it.next();
			if(forceUninstall) {
				this.log.finest("remove process instance: " + process.getName());

				final Process p = this.getModel().getCore().getEngine().getProcessInstanceRegistry().removeProcessInstance(process, forceUninstall);
				if(p != null) {
					it = this.getModel().getCore().getEngine().getProcessInstanceRegistry().getProcessInstances(name).iterator();
				} else {
					this.log.warning("Impossible to delete instance: " + process.getName());
				}
			} else if(((process.getParentExecution() != null)&&(process.getParentExecution().getState() == Execution.State.ENDED))||
					(process.getParentExecution() == null)){
				this.log.finest("remove process instance: " + process.getName());

				final Process p = this.getModel().getCore().getEngine().getProcessInstanceRegistry().removeProcessInstance(process, forceUninstall);
				if(p != null) {
					it = this.getModel().getCore().getEngine().getProcessInstanceRegistry().getProcessInstances(name).iterator();
				} else {
					this.log.warning("Impossible to delete instance: " + process.getName());
				}
			} else {
				if(!undeleteProcess.contains(process)) {
					undeleteProcess.add(process);
				}
			}
		}


		if(undeleteProcess.size() > 0) {
			// impossible to delete all instance: so, create message and throw corresponding exception
			String errorMsg = "";
			for(final Process process: undeleteProcess) {
				errorMsg = errorMsg + "Impossible to remove this process instance" + process.getName() + " because it is not ended: state = " + process.getParentExecution().getState() + "\n";
			}
			throw new CoreException(errorMsg);
		} else {
			// Delete the definition
			res = this.registry.remove(name);
		}
		return res;
	}

	public synchronized void storeProcessDefinition(final URI uri, final ProcessContextDefinition	context) throws CoreException {

		BPELProcess bpelprocess = null;
		bpelprocess = (BPELProcess) this.getModel().getCompiler().validate(uri);
		if(StaticAnalysisImpl.getInstance().getErrors().size() > 0) {
			final String msg = StaticAnalysisPrinter.getInstance().printAll(StaticAnalysisImpl.getInstance().getErrors(), StaticAnalysisImpl.getInstance().getWarnings(), StaticAnalysisImpl.getInstance().getInfos());
			throw new CoreException("Compilation Error: " + msg);
		} else if(StaticAnalysisImpl.getInstance().getWarnings().size() > 0) {
			final String msg = StaticAnalysisPrinter.getInstance().printAll(StaticAnalysisImpl.getInstance().getErrors(), StaticAnalysisImpl.getInstance().getWarnings(), StaticAnalysisImpl.getInstance().getInfos());
			this.log.finest("Compilation warning: " + msg);
		} 
		// create Keys
		List<ProcessKey> keys = null;
		try {
			keys = this.createKeys(bpelprocess);
		} catch (final WSDL4BPELException e) {
			throw new CoreException(e);
		}
		if((keys == null) || (keys.size() == 0)) {
			throw new CoreException("The keys cannot be null");
		}
		String logMsg = "\nList of key :\n";
		for(final ProcessKey key: keys) {
			logMsg = logMsg + key.toString();
			this.registry.put(key, bpelprocess);
		}


		// Pool of instances
		int cpt=0;
		while(cpt<context.getPoolSize()) {
			cpt++;
			final Process processInstance = this.getModel().getCompiler().compile(bpelprocess);
			for(final ProcessKey key: keys){
				this.getModel().getCore().getEngine().getProcessInstanceRegistry().storeProcessInstance(key, processInstance);
			}
			processInstance.run();
		}

		this.log.finest(logMsg);
	}

	public void unStoreProcessDefinition(final URI uri, final boolean forceUninstall) throws CoreException {
		BPELProcess bpelprocess = null;

		bpelprocess = (BPELProcess) this.getModel().getCompiler().validate(uri);

		// create Keys
		List<ProcessKey> keys = null;
		try {
			keys = this.createKeys(bpelprocess);
		} catch (final WSDL4BPELException e) {
			throw new CoreException(e);
		}
		if(keys != null) {
			for(final ProcessKey key: keys) {
				this.removeProcessDefinition(key, forceUninstall);
			}
		}
	}

	private List<ProcessKey> createKeys(final BPELProcess bpelprocess) throws WSDL4BPELException {
		final List<ProcessKey> keys = new ArrayList<ProcessKey>();
		final List<AbsItfInterfaceType> itfs = bpelprocess.getImports().getProcessInterfaces();
		for(final AbsItfInterfaceType itf: itfs) {

			if(itf != null) {
				final List<Endpoint> endpoints = bpelprocess.getImports().findEndpointsImplementingInterface((InterfaceType) itf);
				if((endpoints != null)&&(endpoints.size() >= 1)) {
					for(final Endpoint endpoint: endpoints) {
						if(endpoint.getService() != null) {
							keys.add(new ProcessKeyImpl(itf.getQName(), endpoint.getService().getQName(), endpoint.getName()));
						} 
					}
				} else {
					keys.add(new ProcessKeyImpl(itf.getQName(), null, null));
				}
			} else {
				keys.add(new ProcessKeyImpl(bpelprocess.getQName(), null, null));
			}
		}
		return keys;
	}

	public void setLog(final Logger logger) {
		this.log = logger;
	}

}
