/**
 * easy BPEL software - Copyright (c) 2009 PetalsLink, 
 * http://www.petalslink.com/ 
 *  
 * This library is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2.1 of the License, or (at your option) 
 * any later version. This library is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
 * General Public License for more details. 
 *  
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */ 
package com.ebmwebsourcing.easybpel.xpath.exp.impl.analyzer;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Map.Entry;
import java.util.logging.Logger;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import javax.xml.namespace.QName;

import org.apache.commons.lang.NotImplementedException;
import org.jdom.Attribute;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.Namespace;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.ow2.easywsdl.schema.api.extensions.NamespaceMapperImpl;
import org.ow2.easywsdl.schema.impl.Constants;

import com.ebmwebsourcing.easybpel.model.bpel.api.BPELProcess;
import com.ebmwebsourcing.easybpel.model.bpel.api.util.MessageUtil;
import com.ebmwebsourcing.easybpel.model.bpel.api.variable.BPELVariable;
import com.ebmwebsourcing.easybpel.model.bpel.api.variable.BPELVariableImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.api.XPathExpressionException;
import com.ebmwebsourcing.easybpel.xpath.exp.api.verification.TypeVerifier;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.analyzer.ExpressionNode.Equality;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.analyzer.ExpressionNode.Inequality;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.analyzer.ExpressionNode.Multiplicative;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.CodepointsToStringFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.CompareFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.ConcatFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.ContainsFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.CountFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.CurrentDateFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.CurrentDateTimeFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.CurrentTimeFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.DateTimeFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.DayFromDateFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.DayFromDateTimeFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.DoXslTransformFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.EmptyFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.EscapeHTMLURIFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.ExistsFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.FalseFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.LowerCaseFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.NormalizeSpaceFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.NotFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.NumberFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.StartWithFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.StringFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.StringLengthFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.SubstringAfterFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.SubstringBeforeFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.SubstringFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.TranslateFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.TrueFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.UpperCaseFunctionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.function.XSLParam;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.util.ExpressionUtil;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.util.Util;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.verification.TypeVerifier2Impl;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Scope;
import com.ebmwebsourcing.easyviper.core.api.engine.variable.Variable;
import com.ebmwebsourcing.easyviper.core.api.factory.AbstractFactoryImpl;


/**
 * @author Nicolas Salatge - eBM WebSourcing
 */
public class ExpressionDumpVisitor implements ExpressionVisitor {


    private Logger log = Logger
    .getLogger(ExpressionDumpVisitor.class.getName());

    private int indent = 0;

    private final Scope scope;


    private final BPELProcess bpeldefinition;

    private final TypeVerifier typeVerifier;

    private final NamespaceMapperImpl nsLists = new NamespaceMapperImpl();

    public ExpressionDumpVisitor(final Scope scope, final BPELProcess bpeldefinition, final NamespaceMapperImpl nsLists) {
        this.scope = scope;
        this.bpeldefinition = bpeldefinition;
        this.typeVerifier = new TypeVerifier2Impl();

        // copy specific nsLists
        if(nsLists != null) {
            for(final Entry<String, String> entry: nsLists.getNamespaces().entrySet()) {
                if(!this.nsLists.getNamespaces().containsKey(entry.getKey())) {
                    this.nsLists.addNamespace(entry.getKey(), entry.getValue());
                }
            }
        }

        // copy bpel definition ns list
        if(bpeldefinition != null) {
            for(final Entry<String, String> entry: this.bpeldefinition.getNamespaceContext().getNamespaces().entrySet()) {
                if(!this.nsLists.getNamespaces().containsKey(entry.getKey())) {
                    this.nsLists.addNamespace(entry.getKey(), entry.getValue());
                }
            }
        }
    }

    public void setLog(final Logger log) {
        this.log = log;
    }

    private String indentString() {
        final StringBuffer sb = new StringBuffer();
        for (int i = 0; i < this.indent; ++i) {
            sb.append(' ');
        }
        return sb.toString();
    }

    private String getNameSpace(final String prefixMore2Points) throws XPathExpressionException {
        String res = null;
        if(prefixMore2Points != null) {
            final String prefix = prefixMore2Points.substring(0, prefixMore2Points.length()-1);
            res = this.nsLists.getNamespaceURI(prefix);
            if(res == null) {
                throw new XPathExpressionException("Impossible to find namespace corresponding to prefix: " + prefix);
            }
        }
        return res;
    }

    public Element visit(final SimpleNode node, Element data) throws XPathExpressionException {
        this.log.finest(this.indentString() + node
                + ": acceptor not unimplemented in subclass?");
        ++this.indent;
        data = (Element) node.childrenAccept(this, data);
        --this.indent;
        return data;
    }

    public Element visit(final ASTStart node, Element data) throws XPathExpressionException {
        this.log.finest(this.indentString() + node);
        ++this.indent;

        if (node.children != null) {
            if (node.children.length == 1) {
                data = node.children[0].jjtAccept(this, data);
            } else {
                this.log
                .severe("BPELError: impossible to analyze several expression in same time");
                throw new XPathExpressionException(
                "BPELError: impossible to analyze several expression in same time");
            }
        }

        --this.indent;
        return data;
    }

    public Element visit(final ASTVariable node, final Element data) throws XPathExpressionException {
        Element res = null;

        this.log.finest(this.indentString() + node + ": " + node.getImage());
        ++this.indent;
        try {
            final String variableName = BPELVariableImpl
            .getVariableNameInXPathExpression(node.getImage());
            final String partName = BPELVariableImpl
            .getPartNameInXPathExpression(node.getImage());
            if ((this.scope != null) && (this.bpeldefinition != null)) {
                // find variable in scope
                this.log.finest("find variable "
                        + Util.getLocalPartWithoutPrefix(variableName));
                final Variable v = this.scope.findVariable(new QName(Util
                        .getLocalPartWithoutPrefix(variableName)));
                if (v == null) {
                    throw new CoreException(
                            "BPELError: impossible to find the variable: "
                            + variableName);
                }
                final BPELVariable fromDefinition = (BPELVariable) v;

                // create variable message if not exist
                final boolean exist = MessageUtil.createMessageIfNotExist(v,
                        this.bpeldefinition.getImports(), AbstractFactoryImpl.getCore().getMessageFactory());
                if(exist) {
                    this.log.finest("message created: \n"
                            + new XMLOutputter(Format.getPrettyFormat())
                            .outputString((Element) v.getValue()
                                    .getContent()));
                } else {
                    this.log.finest("corresponding message: \n"
                            + new XMLOutputter(Format.getPrettyFormat())
                            .outputString((Element) v.getValue()
                                    .getContent()));
                }

                // replace variable by the element in xpath expression
                this.log.finest("with variable : "+node.getImage());
                final List<String> xpathExpressions = ExpressionUtil
                .createXPathExpressionByReplacingPartByElement(
                        partName, fromDefinition, node.getImage(),
                        this.bpeldefinition);
                this.log.finest("without variables : "+xpathExpressions);

                final XPathAnalyzer xpathAnalyzer = new XPathAnalyzer();
                res = xpathAnalyzer.evaluate(xpathExpressions, (Element) v
                        .getValue().getContent(), this.nsLists);

                --this.indent;

                if(res != null) {
                    this.log.finest("variable result: " + new XMLOutputter(Format.getPrettyFormat()).outputString(res));
                } else {
                    this.log.finest("variable result: " + "null");
                }
            }
        } catch (final Exception e) {
            this.log.severe("BPELError in evaluation expression: " + e.getMessage());
            e.printStackTrace(System.err);
            throw new XPathExpressionException(e);
        }
        --this.indent;
        return res;
    }

    public Element visit(final ASTInteger node, final Element data) throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:int");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;
        final Document doc = new Document(res);
        this.log.finest(this.indentString() + node);
        res.setText(String.valueOf(ExpressionNode.getInteger(node.getImage())));
        --this.indent;
        return doc.getRootElement();
    }

    public Element visit(final ASTOrExpression node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        boolean value = false;
        res.setAttribute("type", "xsd:boolean");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;
        if ((this.scope != null) && (this.bpeldefinition != null)) {
            // get left side element
            final Element left = this.getElementOrAttribute(0, node, data);
            this.log.finest("ASTOrExpression: left = \n"
                    + new XMLOutputter(Format.getPrettyFormat())
                    .outputString(left));

            // compare with others right side
            for(int i = 1; i < node.children.length; i++) {
                // get right side element
                final Element right = this.getElementOrAttribute(i, node, data);
                this.log.finest("ASTOrExpression: right = \n"
                        + new XMLOutputter(Format.getPrettyFormat())
                        .outputString(right));

                if(left.getName().equals("xpathExpression") || right.getName().equals("xpathExpression")) {
                    left.setName("xpathExpression");
                    left.getAttribute("type").setValue("xsd:string");
                    left.setText(left.getText() + " or " + right.getValue());
                } else if (left.getText().equals("true") || right.getText().equals("true")) {
                    value = true;
                } else {
                    value = false;
                    break;
                }
            }

            if(left.getName().equals("xpathExpression")) {
                res.setName("xpathExpression");
                res.getAttribute("type").setValue("xsd:string");
                res.setText(left.getText());
            }

        } else {
            for (final Node element : node.children) {
                element.jjtAccept(this, data);
            }
        }
        if(!res.getName().equals("xpathExpression")) {
            this.log.finest("ASTOrExpression: result = " + value);
            res.setText(String.valueOf(value));
        } else {
            this.log.finest("ASTOrExpression: result = " + res.getText());
        }
        --this.indent;
        return res;
    }

    public Element visit(final ASTAndExpression node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        boolean value = false;
        res.setAttribute("type", "xsd:boolean");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;
        if ((this.scope != null) && (this.bpeldefinition != null)) {
            // get left side element
            final Element left = this.getElementOrAttribute(0, node, data);
            this.log.finest("ASTAndExpression: left = \n"
                    + new XMLOutputter(Format.getPrettyFormat())
                    .outputString(left));

            // compare with others right side
            for(int i = 1; i < node.children.length; i++) {
                // get right side element
                final Element right = this.getElementOrAttribute(i, node, data);
                this.log.finest("ASTAndExpression: right = \n"
                        + new XMLOutputter(Format.getPrettyFormat())
                        .outputString(right));

                if(left.getName().equals("xpathExpression") || right.getName().equals("xpathExpression")) {
                    left.setName("xpathExpression");
                    left.getAttribute("type").setValue("xsd:string");
                    left.setText(left.getText() + " and " + right.getValue());
                } else if (left.getText().equals("true") && right.getText().equals("true")) {
                    value = true;
                } else {
                    value = false;
                    break;
                }
            }

            if(left.getName().equals("xpathExpression")) {
                res.setName("xpathExpression");
                res.getAttribute("type").setValue("xsd:string");
                res.setText(left.getText());
            }

        } else {
            for (final Node element : node.children) {
                element.jjtAccept(this, data);
            }
        }
        if(!res.getName().equals("xpathExpression")) {
            this.log.finest("ASTAndExpression: result = " + value);
            res.setText(String.valueOf(value));
        } else {
            this.log.finest("ASTAndExpression: result = " + res.getText());
        }
        --this.indent;
        return res;
    }

    public Element getElementOrAttribute(final int indice, final ExpressionNode node,
            final Element data) throws XPathExpressionException {
        Element elmt = null;
        elmt = node.children[indice].jjtAccept(this, data);
        if ((elmt == null) && (this.bpeldefinition != null)) {
            throw new XPathExpressionException("Impossible to find element from: "
                    + ((ExpressionNode) node.children[indice]).getImage());
        }

        if(elmt != null) {
            final Attribute workOnAttribute = elmt.getAttribute("workOnAttribute",
                    Namespace.getNamespace("meta_ebm",
                    "http://com.ebmwebsourcing.easybpel/metadata"));
            if ((workOnAttribute != null) && (this.bpeldefinition != null)) {
                final String attributeName = Util
                .getLocalPartWithoutPrefix(workOnAttribute.getValue());
                final String prefix = Util.getPrefix(workOnAttribute.getValue());
                Namespace nsAttr = null;
                if (prefix != null) {
                    final String ns = this.bpeldefinition.getNamespaceContext()
                    .getNamespaceURI(prefix);
                    nsAttr = Namespace.getNamespace(prefix, ns);
                }

                Attribute leftAttribute = null;
                if (nsAttr != null) {
                    leftAttribute = elmt.getAttribute(attributeName, nsAttr);
                } else {
                    leftAttribute = elmt.getAttribute(attributeName);
                }
                elmt = new Element(leftAttribute.getName(), leftAttribute
                        .getNamespace());
                elmt.setText(leftAttribute.getValue());

                // remove metadata attribute
                elmt.removeAttribute(workOnAttribute);
            }
        }
        return elmt;
    }

    public Element visit(final ASTEqualityExpression node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        boolean value = false;
        res.setAttribute("type", "xsd:boolean");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;

        // get left side element
        final Element left = this.getElementOrAttribute(0, node, data);
        if(left != null) {
            this.log.finest("ASTEqualityExpression: left = \n"
                    + new XMLOutputter(Format.getPrettyFormat())
                    .outputString(left));
        }
        // get right side element
        final Element right = this.getElementOrAttribute(1, node, data);
        if(right != null) {
            this.log.finest("ASTEqualityExpression: right = \n"
                    + new XMLOutputter(Format.getPrettyFormat())
                    .outputString(right));
        }
        if((left != null) && left.getName().equals("xpathExpression")) {
            res.setName("xpathExpression");
            res.getAttribute("type").setValue("xsd:string");
            if(right != null) {
                res.setText(left.getText() + "=" + "\"" + right.getValue() + "\"");
            } else {
                res.setText(left.getText() + "=" + "?");
            }
        } else if ((this.scope != null) && (this.bpeldefinition != null)) {


            // type verification
            Class<?> type = this.typeVerifier.compare(left, right);
            if (type == null) {
                type = this.typeVerifier.compare(left, right);
                //			    throw new XPathExpressionException(
                //				"Impossible to compare elements in equality expression");
            }

            // process
            if (ExpressionNode.getEquality(node.getImage()).equals(
                    Equality.EQUAL)) {
                if (type.equals(Integer.class)) {
                    final int leftValue = this.typeVerifier.getIntValue(left);
                    final int rightValue = this.typeVerifier.getIntValue(right);
                    this.log.finest("ASTEqualityExpression: left = right");
                    value = (leftValue == rightValue);
                }else if (type.equals(Long.class)) {
                    final long leftValue = this.typeVerifier.getLongValue(left);
                    final long rightValue = this.typeVerifier.getLongValue(right);
                    this.log.finest("ASTEqualityExpression: left = right");
                    value = (leftValue == rightValue);
                }else if (type.equals(Double.class)) {
                    final double leftValue = this.typeVerifier.getDoubleValue(left);
                    final double rightValue = this.typeVerifier.getDoubleValue(right);
                    this.log.finest("ASTEqualityExpression: left = right");
                    value = (leftValue == rightValue);
                } else if (type.equals(String.class)) {
                    final String leftValue = this.typeVerifier.getStringValue(left);
                    final String rightValue = this.typeVerifier.getStringValue(right);
                    this.log.finest("ASTEqualityExpression: left = right");
                    value = leftValue.equals(rightValue);
                } else if (type.equals(Boolean.class)) {
                    final boolean leftValue = this.typeVerifier.getBooleanValue(left);
                    final boolean rightValue = this.typeVerifier
                    .getBooleanValue(right);
                    this.log.finest("ASTEqualityExpression: left = right");
                    value = (leftValue == rightValue);
                } else if (type.equals(Date.class)) {
                    final Date leftValue = this.typeVerifier.getDateTimeValue(left);
                    final Date rightValue = this.typeVerifier.getDateTimeValue(right);
                    this.log.finest("ASTEqualityExpression: left = right");
                    value = leftValue.equals(rightValue);
                } else {
                    throw new XPathExpressionException(
                            "Impossible to realize this equality expression with this kind of type: "
                            + type);
                }
            } else if (ExpressionNode.getEquality(node.getImage()).equals(
                    Equality.DIFFERENT)) {
                if (type.equals(Integer.class)) {
                    final int leftValue = this.typeVerifier.getIntValue(left);
                    final int rightValue = this.typeVerifier.getIntValue(right);
                    this.log.finest("ASTEqualityExpression: left != right");
                    value = (leftValue != rightValue);
                } else if (type.equals(Long.class)) {
                    final long leftValue = this.typeVerifier.getLongValue(left);
                    final long rightValue = this.typeVerifier.getLongValue(right);
                    this.log.finest("ASTEqualityExpression: left != right");
                    value = (leftValue != rightValue);
                }else if (type.equals(Double.class)) {
                    final double leftValue = this.typeVerifier.getDoubleValue(left);
                    final double rightValue = this.typeVerifier.getDoubleValue(right);
                    this.log.finest("ASTEqualityExpression: left != right");
                    value = (leftValue != rightValue);
                }else if (type.equals(String.class)) {
                    final String leftValue = this.typeVerifier.getStringValue(left);
                    final String rightValue = this.typeVerifier.getStringValue(right);
                    this.log.finest("ASTEqualityExpression: left != right");
                    value = (!leftValue.equals(rightValue));
                } else if (type.equals(Boolean.class)) {
                    final boolean leftValue = this.typeVerifier.getBooleanValue(left);
                    final boolean rightValue = this.typeVerifier
                    .getBooleanValue(right);
                    this.log.finest("ASTEqualityExpression: left != right");
                    value = (leftValue != rightValue);
                } else if (type.equals(Date.class)) {
                    final Date leftValue = this.typeVerifier.getDateTimeValue(left);
                    final Date rightValue = this.typeVerifier.getDateTimeValue(right);
                    this.log.finest("ASTEqualityExpression: left != right");
                    value = !leftValue.equals(rightValue);
                } else {
                    throw new XPathExpressionException(
                            "Impossible to realize this equality expression with this kind of type: "
                            + type);
                }
            }
        }

        if((left != null) && (!left.getName().equals("xpathExpression"))) {
            this.log.finest("ASTEqualityExpression: result = " + value);
            res.setText(String.valueOf(value));
        }
        --this.indent;
        return res;
    }

    public Element visit(final ASTInequalityExpression node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        boolean value = false;
        res.setAttribute("type", "xsd:boolean");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;
        if ((this.scope != null) && (this.bpeldefinition != null)) {
            // get left side element
            final Element left = this.getElementOrAttribute(0, node, data);
            this.log.finest("ASTInequalityExpression: left = \n"
                    + new XMLOutputter(Format.getPrettyFormat())
                    .outputString(left));

            // get right side element
            final Element right = this.getElementOrAttribute(1, node, data);
            this.log.finest("ASTInequalityExpression: right = \n"
                    + new XMLOutputter(Format.getPrettyFormat())
                    .outputString(right));

            // type verification
            final Class<?> type = this.typeVerifier.compare(left, right);
            if (type == null) {
                throw new XPathExpressionException(
                "Impossible to compare elements in inequality expression");
            }

            // process
            if (ExpressionNode.getInequality(node.getImage()).equals(
                    Inequality.INFERIOR)) {
                if (type.equals(Integer.class)) {
                    final int leftValue = this.typeVerifier.getIntValue(left);
                    final int rightValue = this.typeVerifier.getIntValue(right);
                    this.log.finest("ASTInequalityExpression: left < right");
                    value = leftValue < rightValue;
                } else if (type.equals(Long.class)) {
                    final long leftValue = this.typeVerifier.getLongValue(left);
                    final long rightValue = this.typeVerifier.getLongValue(right);
                    this.log.finest("ASTEqualityExpression: left < right");
                    value = leftValue < rightValue;
                }else if (type.equals(Double.class)) {
                    final double leftValue = this.typeVerifier.getDoubleValue(left);
                    final double rightValue = this.typeVerifier.getDoubleValue(right);
                    this.log.finest("ASTEqualityExpression: left < right");
                    value = leftValue < rightValue;
                }else if (type.equals(Date.class)) {
                    final Date leftValue = this.typeVerifier.getDateTimeValue(left);
                    final Date rightValue = this.typeVerifier.getDateTimeValue(right);
                    this.log.finest("ASTInequalityExpression: left < right");
                    value = leftValue.before(rightValue);
                } else if (type.equals(String.class)) {
                    final String leftValue = this.typeVerifier.getStringValue(left);
                    final String rightValue = this.typeVerifier.getStringValue(right);
                    this.log.finest("ASTInequalityExpression: left < right");
                    value = leftValue.compareTo(rightValue) < 0;
                } else {
                    throw new XPathExpressionException(
                            "Impossible to realize this inequality expression with this kind of type: "
                            + type);
                }
            } else if (ExpressionNode.getInequality(node.getImage()).equals(
                    Inequality.SUPERIOR)) {
                if (type.equals(Integer.class)) {
                    final int leftValue = this.typeVerifier.getIntValue(left);
                    final int rightValue = this.typeVerifier.getIntValue(right);
                    this.log.finest("ASTInequalityExpression: left > right");
                    value = leftValue > rightValue;
                } else if (type.equals(Long.class)) {
                    final long leftValue = this.typeVerifier.getLongValue(left);
                    final long rightValue = this.typeVerifier.getLongValue(right);
                    this.log.finest("ASTEqualityExpression: left > right");
                    value = leftValue > rightValue;
                }else if (type.equals(Double.class)) {
                    final double leftValue = this.typeVerifier.getDoubleValue(left);
                    final double rightValue = this.typeVerifier.getDoubleValue(right);
                    this.log.finest("ASTEqualityExpression: left > right");
                    value = leftValue > rightValue;
                } else if (type.equals(Date.class)) {
                    final Date leftValue = this.typeVerifier.getDateTimeValue(left);
                    final Date rightValue = this.typeVerifier.getDateTimeValue(right);
                    this.log.finest("ASTInequalityExpression: left > right");
                    value = leftValue.after(rightValue);
                } else if (type.equals(String.class)) {
                    final String leftValue = this.typeVerifier.getStringValue(left);
                    final String rightValue = this.typeVerifier.getStringValue(right);
                    this.log.finest("ASTInequalityExpression: left > right");
                    value = leftValue.compareTo(rightValue) > 0;
                } else {
                    throw new XPathExpressionException(
                            "Impossible to realize this inequality expression with this kind of type: "
                            + type);
                }
            } else if (ExpressionNode.getInequality(node.getImage()).equals(
                    Inequality.INFERIOR_EQUALS)) {
                if (type.equals(Integer.class)) {
                    final int leftValue = this.typeVerifier.getIntValue(left);
                    final int rightValue = this.typeVerifier.getIntValue(right);
                    this.log.finest("ASTInequalityExpression: left <= right");
                    value = leftValue <= rightValue;
                }  else if (type.equals(Long.class)) {
                    final long leftValue = this.typeVerifier.getLongValue(left);
                    final long rightValue = this.typeVerifier.getLongValue(right);
                    this.log.finest("ASTEqualityExpression: left <= right");
                    value = leftValue <= rightValue;
                }else if (type.equals(Double.class)) {
                    final double leftValue = this.typeVerifier.getDoubleValue(left);
                    final double rightValue = this.typeVerifier.getDoubleValue(right);
                    this.log.finest("ASTEqualityExpression: left <= right");
                    value = leftValue <= rightValue;
                }else if (type.equals(Date.class)) {
                    final Date leftValue = this.typeVerifier.getDateTimeValue(left);
                    final Date rightValue = this.typeVerifier.getDateTimeValue(right);
                    this.log.finest("ASTInequalityExpression: left <= right");
                    value = leftValue.before(rightValue)
                    || leftValue.equals(rightValue);
                } else if (type.equals(String.class)) {
                    final String leftValue = this.typeVerifier.getStringValue(left);
                    final String rightValue = this.typeVerifier.getStringValue(right);
                    this.log.finest("ASTInequalityExpression: left <= right");
                    value = leftValue.compareTo(rightValue) <= 0;
                } else {
                    throw new XPathExpressionException(
                            "Impossible to realize this inequality expression with this kind of type: "
                            + type);
                }
            } else if (ExpressionNode.getInequality(node.getImage()).equals(
                    Inequality.SUPERIOR_EQUALS)) {
                if (type.equals(Integer.class)) {
                    final int leftValue = this.typeVerifier.getIntValue(left);
                    final int rightValue = this.typeVerifier.getIntValue(right);
                    this.log.finest("ASTInequalityExpression: left >= right");
                    value = leftValue >= rightValue;
                } else if (type.equals(Long.class)) {
                    final long leftValue = this.typeVerifier.getLongValue(left);
                    final long rightValue = this.typeVerifier.getLongValue(right);
                    this.log.finest("ASTEqualityExpression: left >= right");
                    value = leftValue >= rightValue;
                }else if (type.equals(Double.class)) {
                    final double leftValue = this.typeVerifier.getDoubleValue(left);
                    final double rightValue = this.typeVerifier.getDoubleValue(right);
                    this.log.finest("ASTEqualityExpression: left >= right");
                    value = leftValue >= rightValue;
                } else if (type.equals(Date.class)) {
                    final Date leftValue = this.typeVerifier.getDateTimeValue(left);
                    final Date rightValue = this.typeVerifier.getDateTimeValue(right);
                    this.log.finest("ASTInequalityExpression: left >= right");
                    value = leftValue.after(rightValue)
                    || leftValue.equals(rightValue);
                } else if (type.equals(String.class)) {
                    final String leftValue = this.typeVerifier.getStringValue(left);
                    final String rightValue = this.typeVerifier.getStringValue(right);
                    this.log.finest("ASTInequalityExpression: left >= right");
                    value = leftValue.compareTo(rightValue) >= 0;
                } else {
                    throw new XPathExpressionException(
                            "Impossible to realize this inequality expression with this kind of type: "
                            + type);
                }
            }
        }
        this.log.finest("ASTInequalityExpression: result = " + value);
        res.setText(String.valueOf(value));
        --this.indent;
        return res;
    }

    public Element visit(final ASTAdditiveExpression node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        Number value = null;

        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;
        if ((this.scope != null) && (this.bpeldefinition != null)) {
            // get left side element
            final Element left = this.getElementOrAttribute(0, node, data);
            this.log.finest("ASTAdditiveExpression: left = \n"
                    + new XMLOutputter(Format.getPrettyFormat())
                    .outputString(left));

            // get right side element
            final Element right = this.getElementOrAttribute(1, node, data);
            this.log.finest("ASTAdditiveExpression: right = \n"
                    + new XMLOutputter(Format.getPrettyFormat())
                    .outputString(right));

            // type verification
            /* final */ Class<?> type = this.typeVerifier.compare(left, right);
            if (type == null) {
                type = this.typeVerifier.compare(left, right);
                throw new XPathExpressionException(
                "Impossible to compare elements in additive expression");
            }


            Number leftValue = null;
            Number rightValue = null;

            if(type.equals(Integer.class)){
                leftValue = this.typeVerifier.getIntValue(left);

                rightValue = this.typeVerifier.getIntValue(right);

                res.setAttribute("type", "xsd:int");

            }else if(type.equals(Long.class)){

                leftValue = this.typeVerifier.getLongValue(left);

                rightValue = this.typeVerifier.getLongValue(right);

                res.setAttribute("type", "xsd:long");

            }else if(type.equals(Double.class)){

                leftValue = this.typeVerifier.getDoubleValue(left);

                rightValue = this.typeVerifier.getDoubleValue(right);

                res.setAttribute("type", "xsd:double");
            }

            //			final Integer leftValue = this.typeVerifier.getIntValue(left);
            if (leftValue == null) {
                throw new XPathExpressionException("Null value for left expression: "
                        + new XMLOutputter(Format.getPrettyFormat())
                        .outputString(left));
            }
            //			final Integer rightValue = this.typeVerifier.getIntValue(right);
            if (rightValue == null) {
                throw new XPathExpressionException("Null value for right expression: "
                        + new XMLOutputter(Format.getPrettyFormat())
                        .outputString(right));
            }

            // process
            if (ExpressionNode.isPlus(node.getImage())) {
                if (type.equals(Integer.class)) {
                    this.log.finest("ASTAdditiveExpression: left + right");
                    value = (Integer)leftValue + (Integer) rightValue;
                    res.setAttribute("type", "xsd:int");
                } else if(type.equals(Long.class)){
                    value = (Long)leftValue + (Long) rightValue;
                    res.setAttribute("type", "xsd:long");
                }else if(type.equals(Double.class)){
                    value = (Double)leftValue + (Double) rightValue;
                    res.setAttribute("type", "xsd:double");
                }else {
                    throw new XPathExpressionException(
                            "Impossible to realize this additive expression with this kind of type: "
                            + type);
                }
            } else {
                if (type.equals(Integer.class)) {
                    this.log.finest("ASTAdditiveExpression: left - right");
                    value = (Integer)leftValue - (Integer)rightValue;
                    res.setAttribute("type", "xsd:int");
                } else if(type.equals(Long.class)){
                    value = (Long)leftValue - (Long) rightValue;
                    res.setAttribute("type", "xsd:long");
                }else if(type.equals(Double.class)){
                    value = (Double)leftValue - (Double) rightValue;
                    res.setAttribute("type", "xsd:double");
                }else {
                    throw new XPathExpressionException(
                            "Impossible to realize this additive expression with this kind of type: "
                            + type);
                }
            }
        }
        this.log.finest("ASTAdditiveExpression: result = " + value);
        res.setText(String.valueOf(value));
        --this.indent;
        return res;
    }

    public Element visit(final ASTMultiplicativeExpression node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        Number value = null;

        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;
        if ((this.scope != null) && (this.bpeldefinition != null)) {
            // get left side element
            final Element left = this.getElementOrAttribute(0, node, data);
            this.log.finest("ASTMultiplicativeExpression: left = \n"
                    + new XMLOutputter(Format.getPrettyFormat())
                    .outputString(left));

            // get right side element
            final Element right = this.getElementOrAttribute(1, node, data);
            this.log.finest("ASTMultiplicativeExpression: right = \n"
                    + new XMLOutputter(Format.getPrettyFormat())
                    .outputString(right));

            // type verification
            final Class<?> type = this.typeVerifier.compare(left, right);
            if (type == null) {
                throw new XPathExpressionException(
                "Impossible to compare elements in inequality expression");
            }

            Number leftValue = null;
            Number rightValue = null;

            if(type.equals(Integer.class)){
                leftValue = this.typeVerifier.getIntValue(left);

                rightValue = this.typeVerifier.getIntValue(right);

                res.setAttribute("type", "xsd:int");

            }else if(type.equals(Long.class)){

                leftValue = this.typeVerifier.getLongValue(left);

                rightValue = this.typeVerifier.getLongValue(right);

                res.setAttribute("type", "xsd:long");

            }else if(type.equals(Double.class)){

                leftValue = this.typeVerifier.getDoubleValue(left);

                rightValue = this.typeVerifier.getDoubleValue(right);

                res.setAttribute("type", "xsd:double");
            }

            if (leftValue == null) {
                throw new XPathExpressionException("Null value for left expression: "
                        + new XMLOutputter(Format.getPrettyFormat())
                        .outputString(left));
            }

            if (rightValue == null) {
                throw new XPathExpressionException("Null value for right expression: "
                        + new XMLOutputter(Format.getPrettyFormat())
                        .outputString(right));
            }

            // process
            if (ExpressionNode.getMultiplicative(node.getImage()).equals(
                    Multiplicative.MULT)) {
                if (type.equals(Integer.class)) {
                    this.log.finest("ASTMultiplicativeExpression: left * right");
                    value = (Integer)leftValue * (Integer) rightValue;
                } else if(type.equals(Long.class)){
                    value = (Long)leftValue * (Long) rightValue;
                }else if(type.equals(Double.class)){
                    value = (Double)leftValue * (Double) rightValue;
                } else {
                    throw new XPathExpressionException(
                            "Impossible to realize this multiplicative expression with this kind of type: "
                            + type);
                }
            } else if (ExpressionNode.getMultiplicative(node.getImage())
                    .equals(Multiplicative.DIV)) {
                if (type.equals(Integer.class)) {
                    this.log.finest("ASTMultiplicativeExpression: left / right");
                    value = (Integer)leftValue / (Integer) rightValue;
                }  else if(type.equals(Long.class)){
                    value = (Long)leftValue / (Long) rightValue;
                }else if(type.equals(Double.class)){
                    value = (Double)leftValue / (Double) rightValue;
                } else {
                    throw new XPathExpressionException(
                            "Impossible to realize this multiplicative expression with this kind of type: "
                            + type);
                }
            } else if (ExpressionNode.getMultiplicative(node.getImage())
                    .equals(Multiplicative.MOD)) {
                if (type.equals(Integer.class)) {
                    this.log.finest("ASTMultiplicativeExpression: left % right");
                    value = (Integer)leftValue % (Integer)rightValue;
                }  else if(type.equals(Long.class)){
                    value = (Long)leftValue % (Long) rightValue;
                }else if(type.equals(Double.class)){
                    value = (Double)leftValue % (Double) rightValue;
                }else {
                    throw new XPathExpressionException(
                            "Impossible to realize this multiplicative expression with this kind of type: "
                            + type);
                }
            }
        }
        this.log.finest("ASTAdditiveExpression: result = " + value);
        res.setText(String.valueOf(value));
        --this.indent;
        return res;
    }



    public Element visit(final ASTUnaryExpression node, Element data)
    throws XPathExpressionException {
        this.log.finest(this.indentString() + node);
        ++this.indent;
        data = (Element) node.childrenAccept(this, data);
        --this.indent;
        return data;
    }

    public Element visit(final ASTPreIncrementExpression node, Element data)
    throws XPathExpressionException {
        this.log.finest(this.indentString() + node);
        ++this.indent;
        data = (Element) node.childrenAccept(this, data);
        --this.indent;
        return data;
    }

    public Element visit(final ASTPreDecrementExpression node, Element data)
    throws XPathExpressionException {
        this.log.finest(this.indentString() + node);
        ++this.indent;
        data = (Element) node.childrenAccept(this, data);
        --this.indent;
        return data;
    }

    public Element visit(final ASTPostfixExpression node, Element data)
    throws XPathExpressionException {
        this.log.finest(this.indentString() + node);
        ++this.indent;
        data = (Element) node.childrenAccept(this, data);
        --this.indent;
        return data;
    }

    public Element visit(final ASTPrimaryExpression node, Element data)
    throws XPathExpressionException {
        this.log.finest(this.indentString() + node);
        ++this.indent;
        data = (Element) node.childrenAccept(this, data);
        --this.indent;
        return data;
    }

    public Element visit(final ASTBooleanLiteral node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:boolean");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;
        final Document doc = new Document(res);
        this.log.finest(this.indentString() + node);
        res.setText(String.valueOf(ExpressionNode.getBoolean(node.getImage())));
        --this.indent;
        return doc.getRootElement();
    }

    public Element visit(final ASTNullLiteral node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:string");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;
        final Document doc = new Document(res);
        this.log.finest(this.indentString() + node);
        --this.indent;
        return doc.getRootElement();
    }

    public Element visit(final ASTFunction node, Element data) throws XPathExpressionException {
        this.log.finest(this.indentString() + node);
        ++this.indent;
        data = (Element) node.childrenAccept(this, data);
        --this.indent;
        return data;
    }

    public Element visit(final ASTFloat node, final Element data) throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:float");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;
        final Document doc = new Document(res);
        this.log.finest(this.indentString() + node);
        res.setText(String.valueOf(ExpressionNode.getFloat(node.getImage())));
        --this.indent;
        return doc.getRootElement();
    }

    public Element visit(final ASTCaracter node, final Element data) throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:string");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node + ": " + node.getImage());
        ++this.indent;
        final Document doc = new Document(res);
        this.log.finest(this.indentString() + node);
        res.setText(String
                .valueOf(ExpressionNode.getCharacter(node.getImage())));
        this.log.finest("res = "
                + new XMLOutputter(Format.getPrettyFormat()).outputString(res));

        --this.indent;
        return doc.getRootElement();
    }

    public Element visit(final ASTString node, final Element data) throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:string");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node + ": " + node.getImage());
        ++this.indent;
        final Document doc = new Document(res);
        this.log.finest(this.indentString() + node);
        res.setText(ExpressionNode.getString(node.getImage()));
        --this.indent;
        return doc.getRootElement();
    }

    public Element visit(final ASTDoXslTransform node, final Element data)
    throws XPathExpressionException {
        Element res = null;
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            if (node.jjtGetNumChildren() < 2) {
                this.log
                .severe("bad number of arguments for doXslTransform function");
                throw new XPathExpressionException(
                "bad number of arguments for doXslTransform function");
            } else {
                String templateURI = new String();
                templateURI = (node.children[0].jjtAccept(this, null))
                .getText();

                final Element nodeSet = (node.children[1].jjtAccept(this,
                        null));

                final List<XSLParam> xslParams = new ArrayList<XSLParam>();
                for (int i = 2; i < node.children.length - 1; i++) {
                    String paramName = new String();
                    paramName = (node.children[i].jjtAccept(this,
                            null)).getText();
                    String paramValue = new String();
                    paramValue = (node.children[i + 1].jjtAccept(
                            this, null)).getText();
                    final XSLParam param = new XSLParam(paramName, paramValue);
                    xslParams.add(param);
                }

                if ((this.scope != null) && (this.bpeldefinition != null)) {

                    // find url for style sheet
                    URL stylesheet = null;
                    try {
                        // test if url
                        stylesheet = new URL(templateURI);
                    } catch (final MalformedURLException e) {
                        // test if relatif path file
                        try {
                            String path = this.bpeldefinition.getDocumentBaseURI().toString();
                            if(!path.endsWith("/")) {
                                path = path + "/";
                            }
                            path = path + templateURI;
                            stylesheet = URI.create(path).toURL();
                        } catch (final MalformedURLException e1) {
                            // test if absolute path file
                            stylesheet = URI.create(templateURI).toURL();

                        }
                    }

                    final String namespaceFunction = this.getNameSpace(node.getImage());

                    res = new DoXslTransformFunctionImpl(namespaceFunction, stylesheet, nodeSet,
                            xslParams).process();
                }
            }

            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in DoXsltransform Function: " + e.getMessage());
            e.printStackTrace(System.err);
            throw new XPathExpressionException(e);
        }
        return res;
    }

    public Element visit(final ASTNotFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:boolean");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));
        final Document doc = new Document(res);
        try {
            this.log.finest(this.indentString() + node);


            if (node.jjtGetNumChildren() != 1) {
                this.log.severe("bad number of arguments for not function");
                throw new XPathExpressionException(
                "bad number of arguments for not function");
            } else {
                final Element elmt = (node.children[0].jjtAccept(this,
                        null));
                if(elmt != null) {
                    final String s_expression = elmt.getText();

                    final Boolean expression = Boolean.valueOf(s_expression); 

                    final String namespaceFunction = this.getNameSpace(node.getImage());

                    final Boolean notResult = new NotFunctionImpl(namespaceFunction, expression).process();


                    this.log.finest(this.indentString() + node);
                    ++this.indent;

                    this.log.finest(this.indentString() + node);
                    res.setText(notResult.toString());
                    --this.indent;
                }

            }

        } catch (final Exception e) {
            this.log.severe("BPELError in Not Function: " + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return doc.getRootElement();
    }

    public Element visit(final ASTTrueFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:boolean");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));
        this.log.finest(this.indentString() + node);
        ++this.indent;

        if ((node.children != null) && (node.children.length > 0)) {
            throw new XPathExpressionException(
            "Sorry, but true fonction has not parameters");
        }

        final Document doc = new Document(res);
        this.log.finest(this.indentString() + node);
        final String namespaceFunction = this.getNameSpace(node.getImage());

        res.setText(String.valueOf(new TrueFunctionImpl(namespaceFunction).process()));

        --this.indent;
        return doc.getRootElement();
    }

    public Element visit(final ASTFalseFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:boolean");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;

        if ((node.children != null) && (node.children.length > 0)) {
            throw new XPathExpressionException(
            "Sorry, but false fonction has not parameters");
        }

        final Document doc = new Document(res);
        this.log.finest(this.indentString() + node);
        final String namespaceFunction = this.getNameSpace(node.getImage());

        res.setText(String.valueOf(new FalseFunctionImpl(namespaceFunction).process()));

        --this.indent;
        return doc.getRootElement();
    }

    public Element visit(final ASTArrayExpression node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("xpathExpression");
        res.setAttribute("type", "xsd:string");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;

        if (node.children.length > 2) {
            throw new XPathExpressionException(
            "Sorry, but arrays with multiple dimensions are not implemented for the moment");
        }

        // get indice
        final Element indiceOrXpathElmt = node.children[1].jjtAccept(this,
                data);
        if(indiceOrXpathElmt != null) {
            res.setText((node.children[0].jjtAccept(this, data)).getText() + "[" + indiceOrXpathElmt.getText() + "]");
        } else {
            res.setText((node.children[0].jjtAccept(this, data)).getText() + "[" + "?" + "]");
        }

        --this.indent;
        return res;

    }

    public Element visit(final ASTConcatFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:string");
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            final List<String> params = new ArrayList<String>();
            for (final Node element : node.children) {
                Element elmt = (element.jjtAccept(this, null));
                String param = new StringFunctionImpl(com.ebmwebsourcing.easybpel.xpath.exp.api.Constants.XPATH_FUNCTIONS_NAMESPACE, elmt).process();
                params.add(param);
            }

            // call function
            final String namespaceFunction = this.getNameSpace(node.getImage());

            final String value = new ConcatFunctionImpl(namespaceFunction, params).process();
            res.setText(value);

            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in DoXsltransform Function: " + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res;
    }

    public Element visit(final ASTNumberFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:int");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;
        final Document doc = new Document(res);
        this.log.finest(this.indentString() + node);

        if (node.jjtGetNumChildren() != 1) {
            this.log.severe("bad number of arguments for number function");
            throw new XPathExpressionException(
            "bad number of arguments for number function");
        }

        final String namespaceFunction = this.getNameSpace(node.getImage());

        final Integer val = new NumberFunctionImpl(namespaceFunction, node.children[0]
                                                                                    .jjtAccept(this, data)).process();

        res.setText(String.valueOf(val));
        --this.indent;
        return doc.getRootElement();
    }

    public Element visit(final ASTStringFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:string");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;
        final Document doc = new Document(res);
        this.log.finest(this.indentString() + node);

        if (node.jjtGetNumChildren() != 1) {
            this.log.severe("bad number of arguments for string function");
            throw new XPathExpressionException(
            "bad number of arguments for string function");
        }

        final String namespaceFunction = this.getNameSpace(node.getImage());

        final String val = new StringFunctionImpl(namespaceFunction, node.children[0]
                                                                                   .jjtAccept(this, data)).process();

        res.setText(String.valueOf(val));
        --this.indent;
        return doc.getRootElement();
    }

    public Element visit(final ASTCountFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:int");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;
        final Document doc = new Document(res);
        this.log.finest(this.indentString() + node);

        if (node.jjtGetNumChildren() != 1) {
            this.log.severe("bad number of arguments for count function");
            throw new XPathExpressionException(
            "bad number of arguments for count function");
        }

        final String namespaceFunction = this.getNameSpace(node.getImage());

        final Integer val = new CountFunctionImpl(namespaceFunction, node.children[0]
                                                                                   .jjtAccept(this, data)).process();
        this.log.finest("value returned by count function: " + val);

        res.setText(String.valueOf(val));
        --this.indent;
        return doc.getRootElement();
    }


    public Element visit(final ASTCurrentDateFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:date");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));
        final Document doc = new Document(res);
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;
            this.log.finest(this.indentString() + node);


            final String namespaceFunction = this.getNameSpace(node.getImage());

            if (node.jjtGetNumChildren() > 0) {
                this.log.severe("bad number of arguments for number function");
                throw new XPathExpressionException(
                "bad number of arguments for number function");
            }

            final GregorianCalendar gCalendar = new GregorianCalendar();
            gCalendar.setTime(new CurrentDateFunctionImpl(namespaceFunction).process());

            final XMLGregorianCalendar xmlTZ = DatatypeFactory
            .newInstance().newXMLGregorianCalendar();


            final XMLGregorianCalendar xmlCalendar = DatatypeFactory
            .newInstance().newXMLGregorianCalendarDate(gCalendar.get(Calendar.YEAR), gCalendar.get(Calendar.MONTH)+1, gCalendar.get(Calendar.DAY_OF_MONTH), xmlTZ.getTimezone());

            this.log.finest("current date: " + xmlCalendar.toXMLFormat());
            res.setText(xmlCalendar.toXMLFormat());
            --this.indent;
        } catch (final DatatypeConfigurationException e) {
            throw new XPathExpressionException(e);
        }
        return doc.getRootElement();
    }

    public Element visit(final ASTCurrentDateTimeFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:dateTime");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));
        final Document doc = new Document(res);
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;
            this.log.finest(this.indentString() + node);


            final String namespaceFunction = this.getNameSpace(node.getImage());

            if (node.jjtGetNumChildren() > 0) {
                this.log.severe("bad number of arguments for number function");
                throw new XPathExpressionException(
                "bad number of arguments for number function");
            }

            final GregorianCalendar gCalendar = new GregorianCalendar();
            gCalendar.setTime(new CurrentDateTimeFunctionImpl(namespaceFunction).process());
            final XMLGregorianCalendar xmlCalendar = DatatypeFactory
            .newInstance()
            .newXMLGregorianCalendar(gCalendar);

            this.log.finest("current dateTime: " + xmlCalendar.toXMLFormat());
            res.setText(xmlCalendar.toXMLFormat());
            --this.indent;
        } catch (final DatatypeConfigurationException e) {
            throw new XPathExpressionException(e);
        }
        return doc.getRootElement();
    }


    public Element visit(final ASTCurrentTimeFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:time");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));
        final Document doc = new Document(res);
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;
            this.log.finest(this.indentString() + node);


            final String namespaceFunction = this.getNameSpace(node.getImage());

            if (node.jjtGetNumChildren() > 0) {
                this.log.severe("bad number of arguments for number function");
                throw new XPathExpressionException(
                "bad number of arguments for number function");
            }

            final GregorianCalendar gCalendar = new GregorianCalendar();
            gCalendar.setTime(new CurrentTimeFunctionImpl(namespaceFunction).process());

            final XMLGregorianCalendar xmlTZ = DatatypeFactory
            .newInstance().newXMLGregorianCalendar();
            final XMLGregorianCalendar xmlCalendar = DatatypeFactory
            .newInstance().newXMLGregorianCalendarTime(gCalendar.get(Calendar.HOUR), gCalendar.get(Calendar.MINUTE), gCalendar.get(Calendar.SECOND), xmlTZ.getTimezone());

            this.log.finest("current time: " + xmlCalendar.toXMLFormat());
            res.setText(xmlCalendar.toXMLFormat());
            --this.indent;
        } catch (final DatatypeConfigurationException e) {
            throw new XPathExpressionException(e);
        }
        return doc.getRootElement();
    }


    public Element visit(final ASTAttributeLitteral node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("xpathExpression");
        res.setAttribute("type", "xsd:string");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node + ": " + node.getImage());
        ++this.indent;
        final Document doc = new Document(res);
        res.setText(node.getImage());

        this.log.finest(this.indentString() + node);


        --this.indent;
        return doc.getRootElement();
    }

    public Element visit(final ASTSubstringBeforeFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:string");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            final List<String> params = new ArrayList<String>();
            for (final Node element : node.children) {
                String param = new StringFunctionImpl(com.ebmwebsourcing.easybpel.xpath.exp.api.Constants.XPATH_FUNCTIONS_NAMESPACE, (element.jjtAccept(this, null))).process();
                params.add(param);
            }

            // call function
            final String namespaceFunction = this.getNameSpace(node.getImage());

            final String value = new SubstringBeforeFunctionImpl(namespaceFunction, params).process();
            res.setText(value);



            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in SubstringBeforeFunctionImpl Function: "
                    + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res;
    }

    public Element visit(final ASTSubstringAfterFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:string");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            final List<String> params = new ArrayList<String>();
            for (final Node element : node.children) {
                String param = new StringFunctionImpl(com.ebmwebsourcing.easybpel.xpath.exp.api.Constants.XPATH_FUNCTIONS_NAMESPACE, (element.jjtAccept(this, null))).process();
                params.add(param);
            }

            // call function
            final String namespaceFunction = this.getNameSpace(node.getImage());

            final String value = new SubstringAfterFunctionImpl(namespaceFunction, params).process();

            res.setText(value);

            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in SubstringAfterFunctionImpl Function: "
                    + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res;
    }

    public Element visit(final ASTTranslateFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:string");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            final List<String> params = new ArrayList<String>();
            for (final Node element : node.children) {
                String param = new StringFunctionImpl(com.ebmwebsourcing.easybpel.xpath.exp.api.Constants.XPATH_FUNCTIONS_NAMESPACE, (element.jjtAccept(this, null))).process();
                params.add(param);
            }

            // call function
            final String namespaceFunction = this.getNameSpace(node.getImage());

            final String value = new TranslateFunctionImpl(namespaceFunction, params).process();
            res.setText(value);

            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in TranslateFunctionImpl Function: "
                    + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res;
    }

    public Element visit(final ASTChildNodeXpathFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("xpathExpression");
        res.setAttribute("type", "xsd:string");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node);
        ++this.indent;
        final Document doc = new Document(res);

        String functionName = (node.children[0].jjtAccept(this, null)).getText();
        if(functionName.equals("node")) {
            functionName = functionName + "()";
        }

        res.setText("child::" + functionName);

        res.setText(res.getText() + "[" + (node.children[1].jjtAccept(this, null)).getText() + "]" );


        this.log.finest(this.indentString() + node);


        --this.indent;
        return doc.getRootElement();
    }

    public Element visit(final ASTTextXpathFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("xpathExpression");
        res.setAttribute("type", "xsd:string");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node + ": " + node.getImage());
        ++this.indent;
        final Document doc = new Document(res);
        res.setText("text()");

        this.log.finest(this.indentString() + node);


        --this.indent;
        return doc.getRootElement();
    }


    public Element visit(final ASTXpathExpression node, final Element data)
    throws XPathExpressionException {
        Element res = null;
        this.log.finest(this.indentString() + node + ": " + node.getImage());
        ++this.indent;
        try {

            final Element elmt = (node.children[0].jjtAccept(this, null));
            final List<Attribute> attrs = new ArrayList<Attribute>();

            final String variableName = BPELVariableImpl
            .getVariableNameInXPathExpression(((ExpressionNode)node.children[0]).getImage());
            final String partName = BPELVariableImpl
            .getPartNameInXPathExpression(((ExpressionNode)node.children[0]).getImage());

            // get xpath expression
            String xpathExpression = ((ExpressionNode)node.children[0]).getImage();
            Element item = null;
            if(node.children.length >= 2) {
                for(int i = 1; i < node.children.length; i++) {
                    item = node.children[i].jjtAccept(this, null);
                    if(node.children[i] instanceof ASTAttributeLitteral) {
                        // Add metadata information to inform that we work on the attribute
                        // and not the element
                        final String attributeName = item.getText().replace("/@", "");
                        attrs.add(new Attribute("workOnAttribute", attributeName, Namespace
                                .getNamespace("meta_ebm",
                                "http://com.ebmwebsourcing.easybpel/metadata")));

                        if(elmt != null) {
                            res = elmt;
                            this.log.finest("Attribute Expression: res = \n"
                                    + new XMLOutputter(Format.getPrettyFormat())
                                    .outputString(res));
                        }
                    } else if(((item != null) && (item.getName().equals("xpathExpression")))||(item.getName().equals("value"))) {
                        xpathExpression = xpathExpression + item.getText();
                    } else {
                        throw new XPathExpressionException("This item is not an xpath expresssion: "  + item);
                    }

                }
            }


            if ((this.scope != null) && (this.bpeldefinition != null)) {
                // find variable in scope
                this.log.finest("find variable "
                        + Util.getLocalPartWithoutPrefix(variableName));
                final Variable v = this.scope.findVariable(new QName(Util
                        .getLocalPartWithoutPrefix(variableName)));
                if (v == null) {
                    throw new CoreException(
                            "BPELError: impossible to find the variable: "
                            + variableName);
                }
                final BPELVariable fromDefinition = (BPELVariable) v;

                // replace variable by the element in xpath expression
                final List<String> xpathExpressions = ExpressionUtil
                .createXPathExpressionByReplacingPartByElement(
                        partName, fromDefinition, xpathExpression,
                        this.bpeldefinition);


                final Element temp = (Element) v.getValue().getContent();
                final XPathAnalyzer xpathAnalyzer = new XPathAnalyzer();

                res = xpathAnalyzer.evaluate(xpathExpressions, temp, this.nsLists);

                --this.indent;

                if(res != null) {
                    // add metadata attribute
                    for(final Attribute attr: attrs) {
                        res.setAttribute(attr.detach());
                    }

                    this.log.finest("xpathExpression result: " + new XMLOutputter(Format.getPrettyFormat()).outputString(res));
                } else {

                    this.log.finest("xpathExpression result: " + "null");
                }
            }
        } catch (final Exception e) {
            this.log.severe("BPELError in evaluation expression: " + e.getMessage());
            e.printStackTrace(System.err);
            throw new XPathExpressionException(e);
        }
        return res;
    }

    public Element visit(final ASTChildLitteral node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("xpathExpression");
        res.setAttribute("type", "xsd:string");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node + ": " + node.getImage());
        ++this.indent;
        final Document doc = new Document(res);
        res.setText(node.getImage());

        this.log.finest(this.indentString() + node);


        --this.indent;
        return doc.getRootElement();
    }

    public Element visit(final ASTChildNodeNameXpathFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("xpathExpression");
        res.setAttribute("type", "xsd:string");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        this.log.finest(this.indentString() + node + ": " + node.getImage());
        ++this.indent;
        final Document doc = new Document(res);
        res.setText(node.getImage());

        this.log.finest(this.indentString() + node);


        --this.indent;
        return doc.getRootElement();
    }

    public Element visit(final ASTDateTimeFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:dateTime");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));

        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            if (node.jjtGetNumChildren() != 2) {
                this.log
                .severe("bad number of arguments for dateTime function");
                throw new XPathExpressionException(
                "bad number of arguments for dateTime function");
            } else {
                Date date = null;
                Date time = null;

                final Element dateE = node.children[0].jjtAccept(this, null);
                if((dateE != null) && (dateE.getText() != null) && (dateE.getText().trim().length() > 0)) {
                    date = DatatypeFactory.newInstance().newXMLGregorianCalendar(dateE.getText()).toGregorianCalendar().getTime();
                }


                final Element timeE = node.children[1].jjtAccept(this, null);
                if((timeE != null) && (timeE.getText() != null) && (timeE.getText().trim().length() > 0)) {
                    time = DatatypeFactory.newInstance().newXMLGregorianCalendar(timeE.getText()).toGregorianCalendar().getTime();
                }


                final String namespaceFunction = this.getNameSpace(node.getImage());

                final Date dateTime = new DateTimeFunctionImpl(namespaceFunction, date, time).process();

                final GregorianCalendar gCalendar = new GregorianCalendar();
                gCalendar.setTime(dateTime);
                final XMLGregorianCalendar xmlCalendar = DatatypeFactory
                .newInstance()
                .newXMLGregorianCalendar(gCalendar);

                this.log.finest("dateTime: " + xmlCalendar.toXMLFormat());
                res.setText(xmlCalendar.toXMLFormat());

            }
            this.log.finest(this.indentString() + node);


            --this.indent;
        } catch (final XPathExpressionException e) {
            this.log.severe("BPELError in dateTime Function: " + e.getMessage());
            throw new XPathExpressionException(e);
        } catch (final DatatypeConfigurationException e) {
            this.log.severe("BPELError in dateTime Function: " + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res;
    }

    public Element visit(final ASTDayFromDateFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:int");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));
        final Document doc = new Document(res);

        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            this.log.finest(this.indentString() + node);

            if (node.jjtGetNumChildren() == 1) {
                final Element dateE = node.children[0].jjtAccept(this, data);
                if((dateE.getText() != null) && (dateE.getText().trim().length() > 0)) {
                    final Date date = DatatypeFactory.newInstance().newXMLGregorianCalendar(dateE.getText()).toGregorianCalendar().getTime();

                    final String namespaceFunction = this.getNameSpace(node.getImage());

                    final int value = new DayFromDateFunctionImpl(namespaceFunction, date).process();

                    res.setText(String.valueOf(value));
                    this.log.finest("value returned by dayFromDate function: " + value);
                }
            }

            --this.indent;
        } catch (final DatatypeConfigurationException e) {
            throw new XPathExpressionException(e);
        }
        return doc.getRootElement();
    }

    public Element visit(final ASTDayFromDateTimeFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:int");
        res.addNamespaceDeclaration(Namespace.getNamespace("xsd",
                Constants.SCHEMA_NAMESPACE));
        final Document doc = new Document(res);

        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            this.log.finest(this.indentString() + node);

            if (node.jjtGetNumChildren() == 1) {
                final Element dateE = node.children[0].jjtAccept(this, data);
                if((dateE.getText() != null) && (dateE.getText().trim().length() > 0)) {
                    final Date date = DatatypeFactory.newInstance().newXMLGregorianCalendar(dateE.getText()).toGregorianCalendar().getTime();

                    final String namespaceFunction = this.getNameSpace(node.getImage());

                    final int value = new DayFromDateTimeFunctionImpl(namespaceFunction, date).process();

                    res.setText(String.valueOf(value));
                    this.log.finest("value returned by dayFromDateTime function: " + value);
                }
            }

            --this.indent;
        } catch (final DatatypeConfigurationException e) {
            throw new XPathExpressionException(e);
        }
        return doc.getRootElement();
    }

    public Element visit(final ASTAbsFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");

    }

    public Element visit(final ASTAdjustDateToTimezoneFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTAdjustDateTimeToTimezoneFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTAdjustTimeToTimezoneFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTBaseURIFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTBooleanFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTCeilingFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTCodepointEqualFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTCodepointsToStringFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:string");
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            final List<Integer> params = new ArrayList<Integer>();
            for (final Node element : node.children) {
                String param = new StringFunctionImpl(com.ebmwebsourcing.easybpel.xpath.exp.api.Constants.XPATH_FUNCTIONS_NAMESPACE, (element.jjtAccept(this, null))).process();
                params.add(new Integer(param));
            }

            // call function
            final String namespaceFunction = this.getNameSpace(node.getImage());
            String value;
            value = new CodepointsToStringFunctionImpl(namespaceFunction, params).process();

            res.setText(value);

            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in DoXsltransform Function: " + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res;

    }

    public Element visit(final ASTCollectionFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTCompareFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:int");
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            final List<String> params = new ArrayList<String>();
            for (final Node element : node.children) {
                String param = new StringFunctionImpl(com.ebmwebsourcing.easybpel.xpath.exp.api.Constants.XPATH_FUNCTIONS_NAMESPACE, (element.jjtAccept(this, null))).process();
                params.add(param);
            }

            // call function
            final String namespaceFunction = this.getNameSpace(node.getImage());
            int value;
            if(node.children.length >2){
                value = new CompareFunctionImpl(namespaceFunction, params.get(0), params.get(1), params.get(2)).process();
            }else{
                value = new CompareFunctionImpl(namespaceFunction, params.get(0), params.get(1)).process();
            }
            res.setText(Integer.toString(value));

            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in DoXsltransform Function: " + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res;
    }

    public Element visit(final ASTContainsFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:boolean");
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            final List<String> params = new ArrayList<String>();
            for (final Node element : node.children) {
                String param = new StringFunctionImpl(com.ebmwebsourcing.easybpel.xpath.exp.api.Constants.XPATH_FUNCTIONS_NAMESPACE, (element.jjtAccept(this, null))).process();
                params.add(param);
            }

            // call function
            final String namespaceFunction = this.getNameSpace(node.getImage());
            boolean value;
            if(node.children.length != 2){
                throw new XPathExpressionException("Wrong number of arguments in fn:contains(string, string)");
            }else{
                value = new ContainsFunctionImpl(namespaceFunction, params.get(0), params.get(1)).process();
            }
            res.setText(Boolean.toString(value));

            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in DoXsltransform Function: " + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res;

    }

    public Element visit(final ASTDataFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTDaysFromDurationFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTDeepEqualFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTDistinctValuesFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTDocFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTDocAvailableFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTDocumentURIFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTEmptyFunction node, final Element data)
    throws XPathExpressionException {

        final Element res = new Element("value");
        res.setAttribute("type", "xsd:boolean");
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            if (node.jjtGetNumChildren() < 1) {
                this.log
                .severe("bad number of arguments for empty function");
                throw new XPathExpressionException(
                "bad number of arguments for empty function");
            } else {
                final String namespaceFunction = this.getNameSpace(node.getImage());
                final Element nodeSet = (node.children[0].jjtAccept(this,
                        null));

                Boolean value = new EmptyFunctionImpl(namespaceFunction, nodeSet).process();
                res.setText(Boolean.toString(value));

                return res;

            }
        }catch(XPathExpressionException e){
            throw new XPathExpressionException(e);

        }
    }

    public Element visit(final ASTEncodeForURIFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTEndsWithFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTErrorFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTEscapeHTMLURIFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:string");
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            final List<String> params = new ArrayList<String>();
            for (final Node element : node.children) {
                String param = new StringFunctionImpl(com.ebmwebsourcing.easybpel.xpath.exp.api.Constants.XPATH_FUNCTIONS_NAMESPACE, (element.jjtAccept(this, null))).process();
                params.add(param);
            }

            // call function
            final String namespaceFunction = this.getNameSpace(node.getImage());
            String value;
            if(node.children.length > 1){
                throw new XPathExpressionException("Wrong number of arguments in fn:escape-html-uri(string)");
            }else{
                value = new EscapeHTMLURIFunctionImpl(namespaceFunction, params.get(0)).process();
            }
            res.setText(value);

            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in DoXsltransform Function: " + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res;
    }

    public Element visit(final ASTExactlyOneFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTExistsFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:boolean");
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            if (node.jjtGetNumChildren() < 1) {
                this.log
                .severe("bad number of arguments for empty function");
                throw new XPathExpressionException(
                "bad number of arguments for empty function");
            } else {
                final String namespaceFunction = this.getNameSpace(node.getImage());
                final Element nodeSet = (node.children[0].jjtAccept(this,
                        null));

                Boolean value = new ExistsFunctionImpl(namespaceFunction, nodeSet).process();
                res.setText(Boolean.toString(value));

                return res;

            }
        }catch(XPathExpressionException e){
            throw new XPathExpressionException(e);

        }
    }

    public Element visit(final ASTHoursFromDateTimeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTHoursFromDurationFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTHoursFromTimeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTIdFunction node, final Element data) throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTIdRefFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTImplicitTimezoneFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTInScopePrefixesFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTIndexOfFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTInsertBeforeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTIRIToURIFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTLangFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTLastFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTLocalNameFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTLocalNameFromQNameFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTLowerCaseFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:string");
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            final List<String> params = new ArrayList<String>();
            for (final Node element : node.children) {
                String param = new StringFunctionImpl(com.ebmwebsourcing.easybpel.xpath.exp.api.Constants.XPATH_FUNCTIONS_NAMESPACE, (element.jjtAccept(this, null))).process();
                params.add(param);
            }

            // call function
            final String namespaceFunction = this.getNameSpace(node.getImage());
            String value;
            if(node.children.length > 1){
                throw new XPathExpressionException("Wrong number of arguments in fn:upper(string)");
            }else{
                value = new LowerCaseFunctionImpl(namespaceFunction, params.get(0)).process();
            }
            res.setText(value);

            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in DoXsltransform Function: " + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res;
    }

    public Element visit(final ASTMatchesFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTMaxFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTMinutesFromDateTimeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTMinutesFromDurationFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTMinutesFromTimeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTMonthFromDateFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTMonthFromDateTimeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTMonthFromDurationFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTNamespaceURIFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTNamespaceURIForPrefixFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTNamespaceURIFromQNameFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTNilledFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTNodeNameFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTNormalizeSpaceFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:string");
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            final List<String> params = new ArrayList<String>();
            for (final Node element : node.children) {
                String param = new String();
                param = (element.jjtAccept(this, null))
                .getText();
                params.add(param);
            }

            // call function
            final String namespaceFunction = this.getNameSpace(node.getImage());
            String value;
            if(node.children.length > 1){
                throw new XPathExpressionException("Wrong number of arguments in fn:string-length(string?)");
            }else if (node.children.length==1){
                value = new NormalizeSpaceFunctionImpl(namespaceFunction, params.get(0)).process();
            }else{
                value = new NormalizeSpaceFunctionImpl(namespaceFunction).process();
            }
            res.setText(value);

            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in DoXsltransform Function: " + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res;	
    }

    public Element visit(final ASTNormalizeUnicodeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTOneOrMoreFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTPositionFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTPrefixFromQNameFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTQNameFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTRemoveFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTReplaceFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTResolveQNameFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTResolveURIFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTReverseFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTRootFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTRoundFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTRoundHalfToEvenFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTSecondsFromDateTimeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTSecondsFromDurationFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTSecondsFromTimeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTStaticBaseURIFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTStringJoinFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTStringLengthFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:int");
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            final List<String> params = new ArrayList<String>();
            for (final Node element : node.children) {
                String param = new String();
                param = new StringFunctionImpl(com.ebmwebsourcing.easybpel.xpath.exp.api.Constants.XPATH_FUNCTIONS_NAMESPACE, (element.jjtAccept(this, null))).process();
                params.add(param);
            }

            // call function
            final String namespaceFunction = this.getNameSpace(node.getImage());
            int value;
            if(node.children.length > 1){
                throw new XPathExpressionException("Wrong number of arguments in fn:string-length(string?)");
            }else if (node.children.length==1){
                value = new StringLengthFunctionImpl(namespaceFunction, params.get(0)).process();
            }else{
                value = new StringLengthFunctionImpl(namespaceFunction).process();
            }
            res.setText(new Integer(value).toString());

            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in DoXsltransform Function: " + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res;	

    }

    public Element visit(final ASTStringToCodepointsFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTSubsequenceFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTSubstringFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:string");
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            final List<String> params = new ArrayList<String>();
            for (final Node element : node.children) {
                String param = new StringFunctionImpl(com.ebmwebsourcing.easybpel.xpath.exp.api.Constants.XPATH_FUNCTIONS_NAMESPACE, (element.jjtAccept(this, null))).process();
                params.add(param);
            }

            // call function
            final String namespaceFunction = this.getNameSpace(node.getImage());
            String value;
            if((node.children.length > 3) || (node.children.length<2)){
                throw new XPathExpressionException("Wrong number of arguments in fn:substring(string, int, int?)");
            }else
                if(node.children.length==2){
                    value = new SubstringFunctionImpl(namespaceFunction, params.get(0), new Integer(params.get(1))).process();
                }else{
                    value = new SubstringFunctionImpl(namespaceFunction, params.get(0), new Integer(params.get(1)), new Integer(params.get(2))).process();
                }
            res.setText(value);

            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in DoXsltransform Function: " + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res;	
    }

    public Element visit(final ASTStartWithFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:boolean");
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            final List<String> params = new ArrayList<String>();
            for (final Node element : node.children) {
                String param = new StringFunctionImpl(com.ebmwebsourcing.easybpel.xpath.exp.api.Constants.XPATH_FUNCTIONS_NAMESPACE, (element.jjtAccept(this, null))).process();
                params.add(param);
            }

            // call function
            final String namespaceFunction = this.getNameSpace(node.getImage());
            Boolean value;
            if((node.children.length > 2) || (node.children.length < 2)){
                throw new XPathExpressionException("Wrong number of arguments in fn:start-with(string, string)");
            }else {

                value = new StartWithFunctionImpl(namespaceFunction, params.get(0), params.get(1)).process();
            }
            res.setText(value.toString());

            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in DoXsltransform Function: " + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res; 
    }


    public Element visit(final ASTSumFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTTimeZoneFromDateFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTTimeZoneFromDateTimeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTTimeZoneFromTimeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTTokenizeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTTraceFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTUnorderedFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTUpperCaseFunction node, final Element data)
    throws XPathExpressionException {
        final Element res = new Element("value");
        res.setAttribute("type", "xsd:string");
        try {
            this.log.finest(this.indentString() + node);
            ++this.indent;

            final List<String> params = new ArrayList<String>();
            for (final Node element : node.children) {
                String param = new StringFunctionImpl(com.ebmwebsourcing.easybpel.xpath.exp.api.Constants.XPATH_FUNCTIONS_NAMESPACE, (element.jjtAccept(this, null))).process();
                params.add(param);
            }

            // call function
            final String namespaceFunction = this.getNameSpace(node.getImage());
            String value;
            if(node.children.length > 1){
                throw new XPathExpressionException("Wrong number of arguments in fn:upper(string)");
            }else{
                value = new UpperCaseFunctionImpl(namespaceFunction, params.get(0)).process();
            }
            res.setText(value);

            --this.indent;
        } catch (final Exception e) {
            this.log.severe("BPELError in DoXsltransform Function: " + e.getMessage());
            throw new XPathExpressionException(e);
        }
        return res;

    }

    public Element visit(final ASTYearFromDateFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTYearFromDateTimeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTYearFromDurationFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTZeroOrOneFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTNameFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTCurrentFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTCurrentGroupFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTCurrentGroupingKeyFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTDocumentFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTElementAvailableFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTFormatDateFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTFormatDateTimeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTFormatNumberFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTFormatTimeFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTFunctionAvailableFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTGenerateIdFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTKeyFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTRegexGroupFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTSystemPropertyFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTTypeAvailableFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTUnparsedEntityPublicIdFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTUnparsedEntityURIFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTUnparsedTextFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

    public Element visit(final ASTUnparsedTextAvailableFunction node, final Element data)
    throws XPathExpressionException {
        // TODO Auto-generated method stub
        this.log.warning("Not yet implemented ...");
        throw new NotImplementedException(node.getClass().getSimpleName() + " not yet implemented");
    }

}
