/**
 * geasy-graph - A project to deal with graph problematics: pathfinding etc... - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.geasytools.geasygraph.impl.alphastar;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashSet;

import com.ebmwebsourcing.geasytools.geasygraph.api.IGraph;
import com.ebmwebsourcing.geasytools.geasygraph.api.INode;
import com.ebmwebsourcing.geasytools.geasygraph.api.alphastar.IAlphaNode;
import com.ebmwebsourcing.geasytools.geasygraph.api.alphastar.IAlphaStarPathFinder;
import com.ebmwebsourcing.geasytools.geasygraph.impl.PathFinder;

/**
 * Based on resource found at:
 * http://www.policyalmanac.org/games/aStarTutorial.htm
 * 
 * @author nfleury
 * 
 */
public class AlphaStarPathFinder extends PathFinder implements
		IAlphaStarPathFinder {

	private LinkedHashSet<IAlphaNode> openList;

	private LinkedHashSet<IAlphaNode> closedList;

	public AlphaStarPathFinder(IGraph graph, INode source, INode target) {
		super(graph, source, target);

		this.openList = new LinkedHashSet<IAlphaNode>();
		this.closedList = new LinkedHashSet<IAlphaNode>();

	}

	/**
	 * Gets lowest cost node in open list
	 * 
	 * @return
	 */
	public IAlphaNode getLowestCostNode() {
		
		//case where open list only contains the source node 
		if (openList.size() == 1 && openList.contains(this.getSource()))
			return (IAlphaNode) this.getSource();

		Iterator<IAlphaNode> openListIt = openList.iterator();

		IAlphaNode lowestCostNode = null;

		// search the lowest costNode
		while (openListIt.hasNext()) {

			IAlphaNode currentNode = openListIt.next();

			// put one of the open list has lowest cost
			if (lowestCostNode == null)
				lowestCostNode = currentNode;

			if (currentNode.getFCost() < lowestCostNode.getFCost()) {
				lowestCostNode = currentNode;
			}

		}

		return lowestCostNode;
	}

	public LinkedHashSet<INode> getShortestPath() {

		LinkedHashSet<INode> path = new LinkedHashSet<INode>();

		// 1- Add the starting node to the open list
		this.addNodeToOpenseList((IAlphaNode) this.getSource());
		//System.out.println("openList:"+openList);
		// 2- Repeat until target node have been added to closed list

		while (this.closedList.contains(this.getTarget()) == false) {

			// a- Look for the lowest F cost node on the open list
			IAlphaNode currentNode = this.getLowestCostNode();

			// b- add currentNode to closed list
			this.addNodeToClosedList(currentNode);
			this.openList.remove(currentNode);
			//System.out.println("closedList:"+closedList);
			
			// c- for each adjacent node of currentNode
			Iterator<INode> currentNodeIt = currentNode.getAdjacentNodes()
					.iterator();

			while (currentNodeIt.hasNext()) {

				INode currentAdjacentNode = currentNodeIt.next();

				// if adjacent node is walkable and is not in the closed list
				if (this.getNonWalkableNodes().contains(currentAdjacentNode) == false
						&& closedList.contains(currentAdjacentNode) == false) {

					// add adjacent node to open list
					this.addNodeToOpenseList((IAlphaNode) currentAdjacentNode);

					IAlphaNode currAdjacentAlphaNode = (IAlphaNode) currentAdjacentNode;

					// make current node parent of current adjacent node
					currAdjacentAlphaNode.setParentNode(currentNode);

					// process G,H,F costs
					double gCost = this.getGCost(currAdjacentAlphaNode,
							currentNode);
					double hCost = this.getHCost(currAdjacentAlphaNode);
					double fCost = gCost + hCost;

					currAdjacentAlphaNode.setGCost(gCost);
					currAdjacentAlphaNode.setHCost(hCost);
					currAdjacentAlphaNode.setFCost(fCost);

					// if current adjacent node is on the open list already
					// check to see if this path to that node is better
					// using G cost
					if (this.openList.contains(currAdjacentAlphaNode)) {

						// Calculate G cost
						double possibleBetterGCost = this.getGCost(
								currAdjacentAlphaNode, currentNode)
								+ currentNode.getGCost();

						if (possibleBetterGCost < currAdjacentAlphaNode
								.getGCost()) {

							currAdjacentAlphaNode.setParentNode(currentNode);
							currAdjacentAlphaNode.setGCost(possibleBetterGCost);
							double nhCost = this
									.getHCost(currAdjacentAlphaNode);
							currAdjacentAlphaNode.setHCost(nhCost);
							currAdjacentAlphaNode.setFCost(possibleBetterGCost
									+ nhCost);

						}

					}

				}
				

			}
			//System.out.println("openList:"+openList);
		}

		
		//process path: start from the target node until we reach the source node
		
		ArrayList<INode> tmpPath = new ArrayList<INode>();
		
		INode targetNode = this.getTarget();
		
		tmpPath.add(targetNode);
		
		IAlphaNode parent = (IAlphaNode) targetNode;
		

		
		while(parent!=null){
			
			parent = (IAlphaNode) parent.getParentNode();
				
			//System.out.println(parent);
			if (parent!=null) tmpPath.add(parent);
		}
		
		tmpPath.add(this.getSource());


		for(int i=tmpPath.size()-1;i>=0;i--){
			
			path.add(tmpPath.get(i));
			
		}
		
		
		return path;
	}

	public double getGCost(IAlphaNode node1, IAlphaNode node2) {

		double diagonalCost = 0;
		double horizontalCost = 0;
		double verticalCost = 0;

		if (nodesAreOthogonal(node1, node2) == false) {

			// get x,y delta
			float xDelta = Math.abs(node1.getX() - node2.getX());
			float yDelta = Math.abs(node1.getY() - node2.getY());

			if (xDelta == yDelta) {

				diagonalCost = xDelta * 14;

			} else {

				// TODO: consider nonWalkable nodes
				diagonalCost = Math.round(Math.sqrt((Math.pow(xDelta, 2) + Math
						.pow(yDelta, 2))));

			}

		} else {

			horizontalCost = (Math.abs(node1.getX() - node2.getX())) * 10;
			verticalCost = (Math.abs(node1.getY() - node2.getY())) * 10;

		}

		double cost = diagonalCost + verticalCost + horizontalCost;

		return cost;
	}

	/**
	 * Manhattan (only orthogonal movements) distance between a node and the
	 * target destination
	 * 
	 * @param node
	 * @return
	 */
	public double getHCost(IAlphaNode node) {

		double cost = 0;

		cost = Math.abs(node.getX() - this.getTarget().getX())
				+ Math.abs(node.getY() - this.getTarget().getY());
		
		cost = cost * 10;
		
		return cost;
	}

	public boolean nodesAreOthogonal(INode node1, INode node2) {

		// orthogonal means node 1 has same x or y than node 2
		if (node1.getX() == node2.getX() || node1.getY() == node2.getY())
			return true;

		return false;
	}

	public void addNodeToOpenseList(IAlphaNode node) {
		this.openList.add(node);
	}

	public void addNodeToClosedList(IAlphaNode node) {
		this.closedList.add(node);
	}

	public LinkedHashSet<IAlphaNode> getOpenList() {
		return openList;
	}

	public LinkedHashSet<IAlphaNode> getClosedList() {
		return closedList;
	}

}
