/**
 * model-manager - Handles models on client side for stuffs like undo/redo, methods observers, uibinding ... - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.uibinder;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;

import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.HasModelManager;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.ModelProxy;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.ModelRegistry;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.uibinder.uifield.AbstractUIFieldWidget;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.uibinder.uifield.TextBoxUIField;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.uibinder.uifield.UIFieldCheckBox;
import com.google.gwt.user.client.ui.Widget;

@SuppressWarnings(value = "unchecked")
public class UIBindingManager<T> {

	private static UIBindingManager instance;

	private HashMap<Class<?>, FieldMappingPolicy> typeMappingPolicies;
	private HashMap<Class<?>, List<EnclosedFieldMappingPolicy>> enclosedFieldMappingPolicies;
	private HashMap<Class<?>, List<EnclosedFieldMappingPolicy>> ignoreMappingPolicies;

	
	private HashMap<Class<? extends AbstractUITemplate>,IInstantiationHandler<?>> templateInstantiationHandlers;
	private HashMap<Class<?>,IInstantiationHandler<?>> uiFieldsInstantiationHandlers;
	
	//store uifields widgets that have already been created previously
	//A Type have multiple instances which themselves have multiple ui fields 
	private HashMap<ModelProxy,HashMap<String,UIFieldWidget<?>>> singleModelUIFieldRegistry;
	private HashMap<HashSet<ModelProxy>,HashMap<String,UIFieldWidget<?>>> multipleModelUIFieldRegistry;
	
	
	static {
		instance = new UIBindingManager();
	}

	private UIBindingManager() {

		this.typeMappingPolicies 	= new HashMap<Class<?>, FieldMappingPolicy>();
		this.enclosedFieldMappingPolicies 	= new HashMap<Class<?>, List<EnclosedFieldMappingPolicy>>();
		this.ignoreMappingPolicies 	= new HashMap<Class<?>, List<EnclosedFieldMappingPolicy>>();


		this.singleModelUIFieldRegistry 		= new HashMap<ModelProxy,HashMap<String,UIFieldWidget<?>>>();
		this.multipleModelUIFieldRegistry		= new HashMap<HashSet<ModelProxy>, HashMap<String,UIFieldWidget<?>>>();
		
		this.templateInstantiationHandlers = new HashMap<Class<? extends AbstractUITemplate>, IInstantiationHandler<?>>();
		this.uiFieldsInstantiationHandlers = new HashMap<Class<?>, IInstantiationHandler<?>>();
		
		this.defaultTypeMappingPolicy();

	}
	

	
	public void registerUIFieldInstantiationHandler(Class<?> uifieldtype,IInstantiationHandler<?> handler){
		this.uiFieldsInstantiationHandlers.put(uifieldtype, handler);
	}
	
	public IInstantiationHandler<?> getUIFieldInstantiationHandler(Class<?> uifieldType){
		return this.uiFieldsInstantiationHandlers.get(uifieldType);
	}


	
	public void registerUIField(ModelProxy instance,String fieldName,UIFieldWidget<?> uifield){
		
		HashMap<String, UIFieldWidget<?>> instanceUIFields = singleModelUIFieldRegistry.get(instance);
		
		if (instanceUIFields==null){
			
			instanceUIFields = new HashMap<String, UIFieldWidget<?>>();
			
		}
		
		instanceUIFields.put(fieldName, uifield);
		
		singleModelUIFieldRegistry.put(instance, instanceUIFields);
	}
	
	
	public void registerUIField(HashSet<ModelProxy> instances,String fieldName,UIFieldWidget<?> uifield){
		
		HashMap<String, UIFieldWidget<?>> instanceUIFields = multipleModelUIFieldRegistry.get(instances);
		
		if (instanceUIFields==null){
			
			instanceUIFields = new HashMap<String, UIFieldWidget<?>>();
			
		}
		
		instanceUIFields.put(fieldName, uifield);
		
		multipleModelUIFieldRegistry.put(instances, instanceUIFields);
	}
	
	
	/**
	 * Return the widget considering all mapping policies
	 * @param field
	 * @return
	 */
	public UIFieldWidget<?> getWidget(String fieldName,Class<?> fieldType,ModelProxy enclosingObject){
		
		Field field = new Field(fieldName, null, enclosingObject.getClass());
		
		//First check in registry if a widget for this enclosing object and field name haven't been created yet
		UIFieldWidget<?> uiwidget = this.getFieldWidgetFromSingleModelRegistry(fieldName, enclosingObject);
		
		if (uiwidget!=null){
			return uiwidget;
		}
		
		////////Otherwise create the widget considering the actual policies and put it into registry
		///////so that we don't do this anymore
		
		//First check if actual field is not to be ignored so that we don't go further
		if (this.isIgnored(field)) return null;
		
		//Then check if current field is in one of an EnclosedFieldMapping policies
		Class<?> realType = ModelRegistry.getInstance().getModelProxyRealType(field.getEnclosingType());
		if (enclosedFieldMappingPolicies.get(realType)!=null){
			for(EnclosedFieldMappingPolicy efmp:enclosedFieldMappingPolicies.get(realType)){
				
				if (efmp.getEnclosingType()==realType && efmp.getFieldName().equals(field.getName())){
					
					//If yes than instantiate the widget with actual policy widget type
					IInstantiationHandler<?> iHandler = efmp.getInstantiationHandler();
					uiwidget = (UIFieldWidget<?>) iHandler.instantiate();
				
				}
				
			}
		}
		//if uiwidget is still null get the ui widget type provided for actual fields type
		if (uiwidget==null){
			
			for(FieldMappingPolicy fmp:typeMappingPolicies.values()){
				
				if (fmp.getFieldType()==fieldType){
					
					IInstantiationHandler<UIFieldWidget<?>> hh = (IInstantiationHandler<UIFieldWidget<?>>) this.getUIFieldInstantiationHandler(fmp.getUImappingType());
					
					if (hh==null){
						throw new IllegalStateException("No instantiation handler registered for type:"+fmp.getUImappingType());
					}
					
					uiwidget = hh.instantiate();					
					
				}
				
			}
			
			//uiwidget shouldnt be null ! didnt found the Field Mapping policy
			if (uiwidget==null) throw new IllegalStateException("No Field Type Mapping Policy registered for type:"+fieldType);
			
		}
		
		
		//register the widget
		this.registerUIField(enclosingObject, fieldName, uiwidget);
		
		
		
		return uiwidget;
	}
	
	
	public UIFieldWidget<?> getWidget(String fieldName,Class<?> fieldType,HashSet<ModelProxy> enclosingObjects){

		//First check in registry if a widget for those enclosing objects and field name haven't been created yet
		UIFieldWidget<?> uiwidget = this.getFieldWidgetFromMultipleModelRegistry(fieldName, enclosingObjects);
		
		if (uiwidget!=null){
			return uiwidget;
		}
		
		////////Otherwise create the widget considering mapping policies and put it into registry
		///////so that we don't do this anymore
		
		//First check if we are dealing with more than one enclosingObject type
			//if there is only one type we can consider EnclosedTypeMapping policy
			//if not we have to ensure that each of the enclosingObject type (if there is one) has a the same mapping policy for actual field
				//if it is not true we cannot return the widget
			Class<? extends UIFieldWidget<?>> commonEnclosedUIFieldType = (Class<? extends UIFieldWidget<?>>) this.getCommonUIFieldTypeFromEnclosedMapingPolicies(enclosingObjects,fieldName);
			
			if (commonEnclosedUIFieldType!=null){
				
				//If yes than instantiate the widget with actual policy widget type
				IInstantiationHandler<UIFieldWidget<?>> hh = (IInstantiationHandler<UIFieldWidget<?>>) this.getUIFieldInstantiationHandler(commonEnclosedUIFieldType);
				
				if (hh==null){
					throw new IllegalStateException("No instantiation handler registered for type:"+commonEnclosedUIFieldType);
				}
				
				uiwidget = hh.instantiate();
			
				
				
			}
		
			//if uiwidget is still null
			//if uiwidget is still null get the ui widget type provided for actual fields type
			if (uiwidget==null){
				
				for(FieldMappingPolicy fmp:typeMappingPolicies.values()){
					
					if (fmp.getFieldType()==fieldType){
						
						IInstantiationHandler<UIFieldWidget<?>> hh = (IInstantiationHandler<UIFieldWidget<?>>) this.getUIFieldInstantiationHandler(fmp.getUImappingType());
						
						if (hh==null){
							throw new IllegalStateException("No instantiation handler registered for type:"+fmp.getUImappingType());
						}
						
						uiwidget = hh.instantiate();					
						
					}
					
				}
				
				//uiwidget shouldnt be null ! didnt found the Field Mapping policy
				if (uiwidget==null) throw new IllegalStateException("No Field Type Mapping Policy registered for type:"+fieldType);
				
			}
			
			
		//register widget
		this.registerUIField(enclosingObjects, fieldName, uiwidget);	
		
		return uiwidget;
	}
	
	/**
	 * Check for a list of enclosing object if there is a common ui field type
	 * if not null is returned
	 * @param enclosingObjects
	 * @return
	 */
	private Class<?> getCommonUIFieldTypeFromEnclosedMapingPolicies(HashSet<ModelProxy> enclosingObjects,String fieldName){
		
		//first get the enclosedObject types involved
		HashSet<Class<?>> enclosingObjectsTypes = this.getEnclosingObjectsTypes(enclosingObjects);
		
		Class<?> commonUIFieldType = null;
		
		//for each enclosing type check if there is a enclosed mapping policy
		//if yes keep track of the commounUIfieldType
		//if each enclosing type has a policy registered and that the return field type is always the same return it
		//if only one of the type doesnt have any enclosed mapping policy return null
		//if each enclosing type has a policy registered but all the mapping type are not the same => return null
		for(Class<?> type:enclosingObjectsTypes){
			
			Class<?> realType = ModelRegistry.getInstance().getModelProxyRealType(type);
			
			if (enclosedFieldMappingPolicies.get(realType)==null) return null;

			for(EnclosedFieldMappingPolicy efmp:enclosedFieldMappingPolicies.get(realType)){
				
				if (efmp.getEnclosingType()==realType && efmp.getFieldName().equals(fieldName)){
					
					if (commonUIFieldType==null){
						
						commonUIFieldType = efmp.getInstantiationHandler().getClass();
						
					}else{
						
						if (commonUIFieldType!=efmp.getInstantiationHandler().getClass()){
							
							return null;
						
						}
						
					}
					
					break;
				}
				
			}
			
		}
		
		return commonUIFieldType;
	}
	
	/**
	 * Returns a list of enclosing object types for a list of enclosing objects
	 * @param fields
	 * @return
	 */
	public HashSet<Class<?>> getEnclosingObjectsTypes(HashSet<ModelProxy> enclosingObjects){
		
		HashSet<Class<?>> result = new HashSet<Class<?>>();
		
		for(ModelProxy m:enclosingObjects){
			
			result.add(m.getClass());
		}
		
		return result;
	}
	
	/**
	 * Checks if the specified field should be ignored as specified in
	 * UIBindingManager mapping policy
	 * 
	 * @param field
	 * @return boolean
	 */
	public boolean isIgnored(Field field) {
		
		Class<?> modelRealType = ModelRegistry.getInstance()
		.getModelProxyRealType(field.getEnclosingType());

		if (ignoreMappingPolicies.get(modelRealType)==null) return false;
		
		for (EnclosedFieldMappingPolicy fmp : ignoreMappingPolicies.get(field.getEnclosingType())) {

			if (fmp.getFieldName().equals(field.getName())
					&& fmp.getEnclosingType().toString().equals(
							field.getEnclosingType().toString())) {
				return true;
			}

		}

		return false;
	}
	
	
	/**
	 * Get a single uifield by a specific model and the field name from registry
	 * @param fieldName
	 * @param instance
	 * @return
	 */
	public UIFieldWidget getFieldWidgetFromSingleModelRegistry(String fieldName,ModelProxy instance){
		HashMap<String, UIFieldWidget<?>> instanceUIFields = singleModelUIFieldRegistry.get(instance);
		
		if (instanceUIFields==null){
			return null;
		}
		
		return instanceUIFields.get(fieldName);
	}
	
	/**
	 * Get a widget for a set of models ant the field name from registry
	 * @param fieldName
	 * @param instances
	 * @return
	 */
	public UIFieldWidget getFieldWidgetFromMultipleModelRegistry(String fieldName,HashSet<ModelProxy> instances){
		HashMap<String, UIFieldWidget<?>> instanceUIFields = multipleModelUIFieldRegistry.get(instances);
		
		if (instanceUIFields==null){
			return null;
		}
		
		return instanceUIFields.get(fieldName);
	}
 
	
	private void defaultTypeMappingPolicy() {

		this.addTypeMappingPolicy(new FieldMappingPolicy(String.class,
				TextBoxUIField.class));
		
		this.addTypeMappingPolicy(new FieldMappingPolicy(Boolean.class, UIFieldCheckBox.class));
		
		this.addTypeMappingPolicy(new FieldMappingPolicy(Integer.class, TextBoxUIField.class));
		
		this.registerUIFieldInstantiationHandler((Class<?>) TextBoxUIField.class, new IInstantiationHandler<TextBoxUIField<String>>() {

			@Override
			public TextBoxUIField<String> instantiate() {
				
				TextBoxUIField<String> txt = new TextBoxUIField<String>();
				
				return txt;
			}
		});
		
		
		this.registerUIFieldInstantiationHandler(UIFieldCheckBox.class, new IInstantiationHandler<UIFieldCheckBox>() {

			@Override
			public UIFieldCheckBox instantiate() {
				
				UIFieldCheckBox chk = new UIFieldCheckBox(false);
				
				return chk;
			}
		
		});

		
		
		
	}

	public static UIBindingManager getInstance() {
		return instance;
	}

	public void addTypeMappingPolicy(FieldMappingPolicy policy) {
		this.typeMappingPolicies.put(policy.getFieldType(), policy);
	}

	public void addClassMappingPolicy(EnclosedFieldMappingPolicy policy) {

		List<EnclosedFieldMappingPolicy> efmp = this.enclosedFieldMappingPolicies
				.get(policy.getEnclosingType());

		if (efmp == null) {

			efmp = new ArrayList<EnclosedFieldMappingPolicy>();

		}

		efmp.add(policy);

		this.enclosedFieldMappingPolicies.put(policy.getEnclosingType(), efmp);

	}

	public void addIgnoreMappingPolicy(EnclosedFieldMappingPolicy policy) {

		List<EnclosedFieldMappingPolicy> efmp = this.ignoreMappingPolicies
				.get(policy.getEnclosingType());

		if (efmp == null) {

			efmp = new ArrayList<EnclosedFieldMappingPolicy>();

		}

		efmp.add(policy);

		this.ignoreMappingPolicies.put(policy.getEnclosingType(), efmp);

	}

	// public void addUITemplate(AbstractUITemplate uiTemplate){
	// this.templates.put(uiTemplate.getMappedModelType(), uiTemplate);
	// }
	//	

//	public void registerUITemplate(Class<?> clazz, AbstractUITemplate template) {
//		this.templates.put(clazz, template);
//	}
	
	
	
	
	
	public HashMap<Class<?>, FieldMappingPolicy> getTypeMappingPolicies() {
		return typeMappingPolicies;
	}

	public HashMap<Class<?>, List<EnclosedFieldMappingPolicy>> getClassMappingPolicies() {
		return enclosedFieldMappingPolicies;
	}

	public HashMap<Class<?>, List<EnclosedFieldMappingPolicy>> getIgnoreMappingPolicies() {
		return ignoreMappingPolicies;
	}

	public void clearIgnoreMappingPolicies() {
		this.ignoreMappingPolicies.clear();
	}

	public void clearClassMappingPolicies() {
		this.enclosedFieldMappingPolicies.clear();
	}


	
	public Widget bind(T model,AbstractUITemplate tmpl){
		
		ArrayList<T> models = new ArrayList<T>();
		
		models.add(model);
		
		return this.bind(models, tmpl);
	}
	
	
	public Widget bind(List<? extends T> models,AbstractUITemplate tmpl) {

			if (tmpl==null){

				throw new IllegalStateException("Template type cannot be null");
				
			}else if (models.size()==0){
				
				throw new IllegalStateException("No models provided");
				
			}
			
			//first check if the given models implements HasModelManager
			//and if every models does have some fields 
			for(T m:models){
				
				if (m instanceof HasModelManager == false){
					
					throw new IllegalStateException("Model " + m
							+ " does not implement " + HasModelManager.class);

				}
				
				ModelProxy mp = (ModelProxy) m;
				
				if (mp.getFields().size()==0){

					throw new IllegalStateException("Model " + m
							+ " doesn't contain any fields");
				}
				
			}
		
			//Get the fields that are going to be available in template
			///////
			
			HashMap<String, List<Field>> fields = new HashMap<String, List<Field>>();
			

			//if just one model was given => get every common fields
			if (models.size()==1){
				
				HashMap<String, Field> fields2 = ((ModelProxy)models.get(0)).getFields();
				
				for(Field f:fields2.values()){
					
					ArrayList<Field> list = new ArrayList<Field>();
					
					list.add(f);
					
					fields.put(f.getName(), list);
					
				}
				
				
			}else{
				
				fields = this.getCommonFieldsFromModels(models);
			
			}

			
 
			
			// as we cannot dynamically instantiate a class with params ...
			tmpl.setFields(fields);


			return tmpl.getTemplate();
	}

	/**
	 * Returns fields that have same names and types from multiple models
	 * 
	 * @param models
	 * @return
	 */
	protected HashMap<String, List<Field>> getCommonFieldsFromModels(
			List<? extends T> models) {

		List<ModelProxy> modelsproxyList = (List<ModelProxy>) models;

		HashMap<String, List<Field>> commonFields = new HashMap<String, List<Field>>();

		// get the model which has the most fields
		ModelProxy maxModel = modelsproxyList.get(0);

		int maxSize = modelsproxyList.get(0).getFields().size();

		for (ModelProxy m : (List<ModelProxy>) models) {

			if (m.getFields().size() > maxSize) {

				maxModel = m;

			}

		}

		// using those fields compare with other models
		for (Field pf : maxModel.getFields().values()) {

			for (ModelProxy m : modelsproxyList) {

				for (Field cf : m.getFields().values()) {

					if (pf.getName().equals(cf.getName())
							&& pf.getType() == cf.getType()) {

						List<Field> fields = commonFields.get(pf.getName());

						if (fields == null) {
							ArrayList<Field> fields2 = new ArrayList<Field>();
							fields2.add(pf);
							fields2.add(cf);
							commonFields.put(pf.getName(), fields2);
						} else {

							fields.add(cf);

						}

					}

				}

			}

		}

		return commonFields;
	}

}
