/**
 * model-manager - Handles models on client side for stuffs like undo/redo, methods observers, uibinding ... - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client;

import java.util.ArrayList;
import java.util.List;

import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.HistoryManager.ICopyHandler;

/**
 * Handles historization on multiple models 
 * 
 * @author nfleury
 *
 */
public class UndoRedoSession {
	
	//last in first out
	private List<ModelProxy> history;
	
	private int pointer;

	private List<IUndoRedoSessionHandler> handlers;
	
	public UndoRedoSession() {

		this.history = new ArrayList<ModelProxy>();

		this.handlers = new ArrayList<IUndoRedoSessionHandler>();
		this.pointer  = -1; 
	
	}
	
	@SuppressWarnings("unchecked")
	public void registerModel(Object model){
		
		//undos.add((ModelProxy) model);
		
		if (model instanceof ModelProxy){
				
			//every time model changes (a method is called or copy is called) we add it to undo list
			ObservableManager.getInstance().addMethodObserver(model, new MethodObserver<ModelProxy>() {
			
				@Override
				public void afterMethodCalled(ModelProxy model,
						Object... parameters) {
					
					pointer++;
					history.add(pointer,model);

					for(IUndoRedoSessionHandler h:handlers){
						h.onAddModelToHistory(model);
					}
					
					
					//The last model added reflects the actual session state
					//so we have to remove every p > p+1 in history 
					if (history.size()-1>pointer){
						
						for(int i = pointer;i<=history.size()-1;i++){
							
							history.remove(i);
							
						}
						
					}
					
				}
				
				@Override
				public void beforeMethodCalled(ModelProxy model,
						Object... parameters) {
				
					
				}
			});
			
			
//			HistoryManager.getInstance().addCopyHandler((ModelProxy) model, new ICopyHandler(){
//
//				@Override
//				public void onCopy(ModelProxy modelProxy) {
//					
//					undos.add(modelProxy);
//					System.out.println("A copy just have been done");
//					System.out.println("Undo Size:"+undos.size());
//				}
//				
//			});
			
			
			
		}else{
			
			throw new IllegalStateException("Object " +model+" is not instance of"+ModelProxy.class);
			
		}
		
	}
	
	
	public void undo(){
		//System.out.println("Session history size:"+history.size());
		if (hasMoreUndo()){
			
			ModelProxy previousModel = history.get(pointer);
			//System.out.println("pointer:"+pointer);
			
			for(IUndoRedoSessionHandler h:handlers){
				h.onBeforeUndo(previousModel);
			}
			
			HistoryManager.getInstance().undo(previousModel);
		
			for(IUndoRedoSessionHandler h:handlers){
				h.onAfterUndo(previousModel);
			}
			
			this.pointer--;
		}
		
	} 
	
	public void redo(){
		//System.out.println("Session history size:"+history.size());
		if (hasMoreRedo()){
			this.pointer++;
			//System.out.println("pointer:"+pointer);
			ModelProxy nextModel = history.get(pointer);
			
			for(IUndoRedoSessionHandler h:handlers){
				h.onBeforeRedo(nextModel);
			}
			
			HistoryManager.getInstance().redo(nextModel);
			
			for(IUndoRedoSessionHandler h:handlers){
				h.onAfterRedo(nextModel);
			}

		}
		
	}
	
	
	public void addHandler(IUndoRedoSessionHandler handler){
		this.handlers.add(handler);
	}
	
	public boolean hasMoreUndo(){

		if ((pointer)>=0){
			return true;
		}
		
		return false;
	}
	
	public boolean hasMoreRedo(){
		
		if (pointer+1<=history.size()-1){
			return true;
		}

		return false;
	}
	
	
	
	
	public interface IUndoRedoSessionHandler{
		
		void onBeforeUndo(ModelProxy undoModel);
		
		void onBeforeRedo(ModelProxy redoModel);
		
		void onAfterUndo(ModelProxy undoModel);
		
		void onAfterRedo(ModelProxy redoModel);
		
		void onAddModelToHistory(ModelProxy addeModel);
	}
	
	
	
}
