/**
 * webeditor-impl - Petalslink is an open source editor providing tools for Service Oriented Architecture. - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.geasytools.webeditor.impl.client.core;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;

import com.ebmwebsourcing.geasytools.webeditor.api.components.IComponent;
import com.ebmwebsourcing.geasytools.webeditor.api.components.IContentPanelComponent;
import com.ebmwebsourcing.geasytools.webeditor.api.components.IQualifiedName;
import com.ebmwebsourcing.geasytools.webeditor.api.core.IEditorRegistry;
import com.ebmwebsourcing.geasytools.webeditor.api.plugin.IPlugin;
import com.ebmwebsourcing.geasytools.webeditor.api.plugin.IProjectPlugin;
import com.ebmwebsourcing.geasytools.webeditor.api.project.configuration.view.IEditConfigurationPanel;
import com.ebmwebsourcing.geasytools.webeditor.impl.client.plugin.events.PluginRegisteredEvent;
import com.ebmwebsourcing.geasytools.webeditor.impl.client.plugin.events.PluginRegistrationCompleteEvent;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstance;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectType;

public class EditorRegistry implements IEditorRegistry{
	
	private HashMap<IQualifiedName, IComponent> registeredComponents;
	private HashMap<IProjectType, IProjectPlugin> projectHandlerPlugins;
	private HashMap<IContentPanelComponent,HashSet<IComponent>> contentPanelComponents;
	private List<IProjectType> availableProjectTypes;
	private HashMap<IProjectInstance, List<IEditConfigurationPanel>> editConfigurationPanelsByType;
	
	private EditorController controller;
	
	public EditorRegistry(EditorController controller) {
		
		this.registeredComponents 			= new HashMap<IQualifiedName, IComponent>();
		this.projectHandlerPlugins 			= new HashMap<IProjectType, IProjectPlugin>();
		this.contentPanelComponents 		= new HashMap<IContentPanelComponent, HashSet<IComponent>>();
		this.editConfigurationPanelsByType 	= new HashMap<IProjectInstance, List<IEditConfigurationPanel>>();
		
		this.availableProjectTypes			= new ArrayList<IProjectType>();
		this.controller 					= controller;
	
	}
	
	@Override
	public IComponent getComponentByQualifiedName(IQualifiedName qname) {
		return this.registeredComponents.get(qname);
	}

	@Override
	public HashMap<IQualifiedName, IComponent> getRegisteredComponents() {
		return registeredComponents;
	}

	@Override
	public void registerComponent(IComponent component) {
		this.registeredComponents.put(component.getQualifiedName(), component);		
	}

	@Override
	public IProjectPlugin getProjectPlugin(
			IProjectType projectType) {
		return projectHandlerPlugins.get(projectType);
	}

	@Override
	public void registerPlugins(HashSet<IPlugin> plugins) {
		
		for(IPlugin plugin:plugins){
		
			if (plugin instanceof IProjectPlugin){
				//add available type
				IProjectPlugin projectPlugin = (IProjectPlugin) plugin; 
				
				availableProjectTypes.add(projectPlugin.getProjectType());
				
				//assign event bus to plugin
				plugin.setEventBus(controller.getEventBus());
				
				
				//
				this.projectHandlerPlugins.put(((IProjectPlugin) plugin).getProjectType(), (IProjectPlugin) plugin);
				controller.getEventBus().fireEvent(new PluginRegisteredEvent(plugin));
			}
		}
		
		//notify that plugin registration is complete
		controller.getEventBus().fireEvent(new PluginRegistrationCompleteEvent());
		
	}

	@Override
	public HashSet<IComponent> getAssociatedComponents(
			IContentPanelComponent component) {
		return contentPanelComponents.get(component);
	}


	@Override
	public void registerComponent(IContentPanelComponent component,
			HashSet<IComponent> components) {
		this.contentPanelComponents.put(component, components);
	}

	@Override
	public List<IProjectType> getAvailableProjectsType() {
		return availableProjectTypes;
	}

	@Override
	public List<IEditConfigurationPanel> getEditConfigurationPanelsByProjectInstance(IProjectInstance instance) {
		return editConfigurationPanelsByType.get(instance);
	}

	@Override
	public void registerEditConfigurationPanels(IProjectInstance instance, List<IEditConfigurationPanel> panels) {
		
		editConfigurationPanelsByType.put(instance, panels);
		
	}

	@Override
	public IContentPanelComponent getContentPanelComponent(
			IProjectInstance instance) {
		
		for(IContentPanelComponent component:contentPanelComponents.keySet()){
			
			if (component.getProjectInstance()==instance){
				return component;
			}
			
		}
		
		return null;
	}

	@Override
	public HashSet<IComponent> getAllContentPanelsComponents() {
		
		HashSet<IComponent> allComponents = new HashSet<IComponent>();
		
		for(IContentPanelComponent cpc : contentPanelComponents.keySet()){
			
			for(IComponent c:contentPanelComponents.get(cpc)){
				
				allComponents.add(c);
				
			}
			
		}
		
		return allComponents;
	}


}
