/**
 * Petals View - Functional Supervision.
 * Copyright (c) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * NotificationProcessorImpl.java
 * -------------------------------------------------------------------------
 */
package com.ebmwebsourcing.petalsview.service.notification;

import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import javax.annotation.Resource;

import org.springframework.stereotype.Service;

import com.ebmwebsourcing.petalsview.persistence.dao.flow.FlowDAO;
import com.ebmwebsourcing.petalsview.persistence.dao.flow.FlowStepDAO;
import com.ebmwebsourcing.petalsview.persistence.model.flow.Flow;
import com.ebmwebsourcing.petalsview.persistence.model.flow.FlowStep;

/**
 * This class is in charge of processing a {@link Notification} instance to
 * create/update appropriated objects ( {@link Flow} and {@link FlowStep} ) in
 * database
 * 
 * @author ofabre
 */
@Service("notificationProcessor")
public class NotificationProcessorImpl implements NotificationProcessor {

	@Resource(name="flowDAO")
	private FlowDAO flowDAO;

	@Resource(name="flowStepDAO")
	private FlowStepDAO flowStepDAO;

	private Map<String, Flow> flowMap = null;

	private Map<String, FlowStep> flowStepMap = null;

	public NotificationProcessorImpl() {
		super();
		this.flowMap = new ConcurrentHashMap<String, Flow>();
		this.flowStepMap = new ConcurrentHashMap<String, FlowStep>();
	}

	public void process(final List<Notification> notificationList)
			throws NotificationProcessingException {
		if (notificationList != null) {
			final Iterator<Notification> iterator = notificationList.iterator();
			while (iterator.hasNext()) {
				final Notification notification = iterator.next();
				if (notification != null && notification.isValid()) {
					final Flow flow = this.createOrRetrieveFlow(notification);
					this.createOrRetrieveFlowStep(notification, flow);
				}
			}
			this.save();
		} else {
			throw new NotificationProcessingException(
					"Can't process a null notification list");
		}
	}

	/**
	 * Retrieve if exists and create if not exists a flow step
	 * 
	 * @param notification
	 *            The notification
	 * @param flow
	 *            The flow
	 * @return The flow step
	 * @throws NotificationProcessingException
	 */
	private void createOrRetrieveFlowStep(final Notification notification,
			final Flow flow) throws NotificationProcessingException {
		final FlowStep flowStep = this.retrieveFlowStep(notification);
		if (flowStep == null) {
			this.createFlowStep(notification, flow);
		}
		this.updateFlowStep(notification);
	}

	/**
	 * Retrieve if exists and create if not exists a flow
	 * 
	 * @param notification
	 *            The notification
	 * @return the flow
	 * @throws Throwable
	 */
	private Flow createOrRetrieveFlow(final Notification notification) {
		Flow flow = null;
		flow = this.retrieveFlow(notification);
		if (flow == null) {
			flow = this.createFlow(notification);
		}
		return flow;
	}

	/**
	 * Save the flows and flowSteps
	 * 
	 * @param notification
	 *            The notification
	 * @return the flow
	 * @throws Throwable
	 */
	private void save() {
		final Collection<Flow> flowCollection = this.flowMap.values();
		for (Flow flow : flowCollection) {
			this.flowDAO.save(flow);
		}
		final Collection<FlowStep> flowStepCollection = this.flowStepMap
				.values();
		for (FlowStep flowStep : flowStepCollection) {
			this.flowStepDAO.save(flowStep);
		}
		this.flowMap.clear();
		this.flowStepMap.clear();
	}

	private FlowStep retrieveFlowStep(final Notification notification) {
		FlowStep flowStep = null;
		flowStep = this.flowStepMap.get(notification.getMeUUID());
		if (flowStep == null) {
			flowStep = this.flowStepDAO.loadByMeUUID(notification.getMeUUID());
			if (flowStep != null) {
				this.flowStepMap.put(flowStep.getMeUUID(), flowStep);
			}
		}
		return flowStep;
	}

	private Flow retrieveFlow(final Notification notification) {
		Flow flow = null;

		flow = this.flowMap.get(notification.getProcessId());
		if (flow == null) {
			flow = this.flowDAO.loadByPetalsId(notification.getProcessId());
			if (flow != null) {
				this.flowMap.put(flow.getIdpetals(), flow);
			}
		}
		return flow;
	}

	private Flow createFlow(final Notification notification) {
		final Flow flow = new Flow();
		// We admit that the process Id is generated before
		// notification emit and is unique
		flow.setIdpetals(notification.getProcessId());
		flow.setType(notification.getProcessType());
		this.flowMap.put(flow.getIdpetals(), flow);
		return flow;
	}

	private FlowStep createFlowStep(final Notification notification,
			final Flow flow) throws NotificationProcessingException {
		final FlowStep flowStep = new FlowStep();
		flowStep.setEndpointName(notification.getEndpointName());
		flowStep.setInterfaceName(notification.getInterfaceName());
		flowStep.setServiceName(notification.getServiceName());
		flowStep.setMeUUID(notification.getMeUUID());
		flowStep.setStatus(Flow.IN_PROGRESS_STATUS);
		flowStep.setFlow(flow);
		this.flowStepMap.put(flowStep.getMeUUID(), flowStep);
		return flowStep;
	}

	/**
	 * @param notification
	 * @param flowStep
	 * @throws NotificationProcessingException
	 */
	private FlowStep updateFlowStep(final Notification notification)
			throws NotificationProcessingException {
		final FlowStep flowStep = this.flowStepMap
				.get(notification.getMeUUID());
		if (flowStep != null) {
			switch (notification.getStatus()) {
			case Flow.IN_PROGRESS_STATUS:
				Date date = notification.getNotifDate();
				long startDate = date.getTime();
				flowStep.setStartDate(startDate);
				final List<String> paramList = new CopyOnWriteArrayList<String>();
				if (notification.getParams() != null
						&& !notification.getParams().isEmpty()) {
					paramList.addAll(notification.getParams());
				}
				if (flowStep.getParameters() != null
						&& !flowStep.getParameters().isEmpty()) {
					paramList.addAll(flowStep.getParameters());
				}
				flowStep.setParameters(paramList);
				break;
			default:
				if (notification.getStatus() < -1) {
					throw new NotificationProcessingException(
							"Try to process an unsupported notif status: "
									+ notification.getStatus());
				}
				flowStep.setStatus(notification.getStatus());
				date = notification.getNotifDate();
				long endDate = date.getTime();
				flowStep.setEndDate(endDate);
				break;
			}
			this.flowStepMap.put(flowStep.getMeUUID(), flowStep);
		}
		return flowStep;
	}
}
