/**
 * bpmn-plugins - Geasy BPMN Editor Plugins - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.server.plugin.bpmn.descriptive;

import java.io.File;

import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;

import com.ebmwebsourcing.easybox.api.XmlContextFactory;
import com.ebmwebsourcing.easybox.api.XmlObjectWriter;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Definitions;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNDiagram;
import com.ebmwebsourcing.easycommons.xml.XMLPrettyPrinter;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.IDefinitionsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.DefinitionsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNDiagram;
import com.ebmwebsourcing.petalsbpm.client.plugin.bpmn.BPMNExportFormat;
import com.ebmwebsourcing.petalsbpm.client.plugin.bpmn.BPMNProjectInstance;
import com.ebmwebsourcing.petalsbpm.server.plugin.bpmn.BPMNFileHelper;
import com.ebmwebsourcing.petalsbpm.server.service.bpmn2.clientToServer.ClientToServer;
import com.ebmwebsourcing.petalsbpm.server.service.bpmndi.clientToServer.BPMNDiagramGenerator;
import com.ebmwebsourcing.petalsbpm.utils.server.FileHelper;
import com.ebmwebsourcing.petalsbpm.utils.server.ZipHelper;
import com.ebmwebsourcing.petalsbpm.xpdl.XPDLAdapter;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstance;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstanceFormat;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstanceMetaData;
import com.ebmwebsourcing.webeditor.impl.domain.exception.ServiceException;
import com.ebmwebsourcing.webeditor.server.impl.service.user.UserServiceImpl;
import com.ebmwebsourcing.webeditor.server.impl.service.webeditor.WebEditorService;
import com.ebmwebsourcing.webeditor.shared.server.api.service.project.IProjectInstanceWriter;


public class DescriptiveBPMNProjectInstanceWriter implements IProjectInstanceWriter{

	
	@Override
	public File writeProjectFile(IProjectInstance projectInstance, File projectOutputDirectory) throws ServiceException {
		IDefinitionsBean defs = ((BPMNProjectInstance) projectInstance).getDefinitions();
		IBPMNDiagram idiagram = ((BPMNProjectInstance) projectInstance).getBpmndiagram();
		
		String directory = projectOutputDirectory.getAbsolutePath();
		File dir = new File(directory);
		if(!dir.exists()) {
		    dir.mkdir();
		}
        String path = getTempPathFromFile(dir);
		
        String bpmnFileAbsolutePath = path + defs.getId()+".bpmn"; 
        
        try {
            //cleans the directory that is going to be zipped
            File zipDir = new File(path);
        	FileHelper.cleanDirectory(zipDir);
        	
			Definitions def = ClientToServer.adapt((DefinitionsBean)defs);
			
			BPMNDiagram diagram = BPMNDiagramGenerator.generateBPMNDiagram(idiagram);
			def.addBPMNDiagram(diagram);
			
			XmlObjectWriter xmlWriter = new XmlContextFactory().newContext().createWriter();
			Document docDefs = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
			xmlWriter.writeDocument(def, docDefs);
		
			FileHelper.writeFile(bpmnFileAbsolutePath, XMLPrettyPrinter.prettyPrint(docDefs));

			//copy the imports into the temporary directory
			File attachedFilesDir = WebEditorService.getInstance().getAttachedFilesDirectory(UserServiceImpl.getInstance().getLoggedUser(), projectInstance);
			File[] importFiles = attachedFilesDir.listFiles();
			if(importFiles!=null && importFiles.length>0) {
			    FileHelper.copyFile(attachedFilesDir, zipDir);
			    File zipFile = ZipHelper.getInstance().createZipFromFolder(path,directory+File.separator+defs.getId()+".zip");
			    return zipFile;
			}
            else {
                return new File(bpmnFileAbsolutePath);
            }
		}catch (Throwable e){
			e.printStackTrace();
			throw new ServiceException("An error occured during the file writing :"+e.getMessage(),e);
		}
	}
	
	
	@Override
	public File writeProjectFile(File projectFile, IProjectInstanceMetaData exportData,
			IProjectInstanceFormat format, File directory) throws ServiceException{
		return writeProjectFile(projectFile,format,directory);
	}

	
	@Override
	public File writeProjectFile(File projectInstanceFile, IProjectInstanceFormat formatToWriteIn,
			File temporaryOutputDirectory) throws ServiceException {
		try{
			switch((BPMNExportFormat)formatToWriteIn){

			case BPMN:
				return projectInstanceFile;

			case XPDL:
				return getXPDLFromBPMN2(projectInstanceFile, temporaryOutputDirectory);
				
			default:
				throw new ServiceException("This format ("+formatToWriteIn+") is not supported.");
			}
		}catch(Exception e){
			throw new ServiceException("An error occured during the file writing :"+e.getMessage(),e); 
		}
	}

	
	private File getXPDLFromBPMN2(File projectInstanceFile, File temporaryOutputDirectory) throws Exception{
		XPDLAdapter adapter = new XPDLAdapter();
		if(ZipHelper.getInstance().isZipFile(projectInstanceFile)) {
		    ZipHelper.getInstance().unzipFile(projectInstanceFile, temporaryOutputDirectory);
		    return adapter.getXPDL2FromBPMN2(BPMNFileHelper.getBPMNFileInFolder(temporaryOutputDirectory).getAbsolutePath());
		}
		else {
		    return adapter.getXPDL2FromBPMN2(projectInstanceFile.getAbsolutePath());
		}
	}
	
	
	protected String getTempPathFromFile(File dir) {
		String result = FileHelper.getUsablePath(dir);
		result = result + BPMNFileHelper.UnzipDirectoryName + File.separator;
		File f = new File(result);
		if(!f.exists()){
			f.mkdir();
		}
		return result;
	}
	
}
