/**
 * Addressing Descriptor - SOA Tools Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * EndpointReferenceType.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.addressing.descriptor;


import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import javax.xml.XMLConstants;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.apache.neethi.Policy;
import org.apache.neethi.PolicyEngine;
import org.w3._2005._08.addressing.AttributedURIType;
import org.w3._2005._08.addressing.EndpointReferenceType;
import org.w3._2005._08.addressing.ObjectFactory;
import org.w3._2005._08.addressing.ReferenceParametersType;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

public class Endpoint extends EndpointReferenceType {

	/**
	 * WS-Addressing schema name
	 */
	public static final String XSD_ADDRESSING = "addressing.xsd";

	/**
	 * WS-Policy schema name
	 */
	public static final String XSD_POLICY = "policy.xsd";

	/**
	 * The JAXB unique unmarshaller.
	 */
	private static Unmarshaller unmarshaller = null;

	/**
	 * The exception raised during JAXB unique unmarshaller creation.
	 */
	private static AddressingException unmarshallerCreationEx = null;

	/**
	 * The JAXB unique marshaller.
	 */
	private static Marshaller marshaller = null;

	/**
	 * The exception raised during JAXB unique marshaller creation.
	 */
	private static AddressingException marshallerCreationEx = null;


	private ObjectFactory factory;

	/*
	 * Private object initializations
	 */
	static {
		final SchemaFactory factory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
		try {
			// The EndpointReference schema resource is in the Jar where the class is
			// loaded
			final URL schemaUrl = Endpoint.class.getResource("/"
					+ XSD_ADDRESSING);
			final URL schemaPolicyUrl = Endpoint.class.getResource("/"
					+ XSD_POLICY);
			final Schema schema = factory
			.newSchema(new StreamSource[] {
					new StreamSource(schemaUrl.openStream()), new StreamSource(schemaPolicyUrl.openStream()) });

			// Unmarshaller initialization
			try {
				final JAXBContext jaxbContext = JAXBContext
				.newInstance(new Class[] { EndpointReferenceType.class, Policy.class });
				unmarshaller = jaxbContext.createUnmarshaller();
				//		unmarshaller.setSchema(schema);
			} catch (JAXBException e) {
				unmarshallerCreationEx = new AddressingException(
						"Failed to create the JAXB unmarshaller", e);
			}

			// Marshaller initialization
			try {
				final JAXBContext jaxbContext = JAXBContext
				.newInstance(new Class[] { EndpointReferenceType.class, Policy.class  });
				marshaller = jaxbContext.createMarshaller();
				//		marshaller.setSchema(schema);
			} catch (JAXBException e) {
				marshallerCreationEx = new AddressingException(
						"Failed to create the JAXB marshaller", e);
			}

		} catch (final IOException e) {
			final AddressingException schemaCreationEx = new AddressingException(
					"Failed to get resource '" + XSD_ADDRESSING
					+ "'from the current class-path", e);
			unmarshallerCreationEx = new AddressingException(
					"Failed to create the JAXB unmarshaller",
					schemaCreationEx);

			marshallerCreationEx = new AddressingException(
					"Failed to create the JAXB marshaller",
					schemaCreationEx);
		} catch (final SAXException e) {
			final AddressingException schemaCreationEx = new AddressingException(
					"Failed to parse resource '" + XSD_ADDRESSING + "'", e);
			unmarshallerCreationEx = new AddressingException(
					"Failed to create the JAXB unmarshaller",
					schemaCreationEx);

			marshallerCreationEx = new AddressingException(
					"Failed to create the JAXB marshaller",
					schemaCreationEx);
		}
	}




	public Endpoint(String endpoint) throws URISyntaxException, AddressingException {
		this(new URI(endpoint), null, null, null, null);
	}

	public Endpoint(String endpoint, QName serviceName, QName interfaceName, List<Policy> policies) throws URISyntaxException, AddressingException {
		this(new URI(endpoint), null, serviceName, interfaceName, policies);
	}


	public Endpoint(URI endpoint,
			ReferenceParametersType referenceParameters,
			QName interfaceName,
			QName serviceName,
			List<Policy> policies) throws AddressingException {


		this.factory = new ObjectFactory();

		// set address
		AttributedURIType adr = new AttributedURIType();
		adr.setValue(endpoint.toString());
		this.setAddress(adr);

		if(serviceName != null) {
			this.getAddress().getOtherAttributes().put(new QName("service"), serviceName.toString());
		}
		if(interfaceName != null) {
			this.getAddress().getOtherAttributes().put(new QName("interface"), interfaceName.toString());
		}


		// set reference parameters
		this.setReferenceParameters(referenceParameters);


		// set policies
		this.setPolicy(policies);

	}


	private Endpoint(EndpointReferenceType epr) {
		this.address = epr.getAddress();
		this.any = epr.getAny();
		this.metadata = epr.getMetadata();
		this.referenceParameters = epr.getReferenceParameters();
	}

	private JAXBElement<EndpointReferenceType> getEndpointReference() {
		return this.factory.createEndpointReference(this);
	}


	private List<Element> convertPolicies2Elements(List<Policy> policies) throws AddressingException {
		List<Element> res = new ArrayList<Element>();
		for (Policy policy : policies) {
			res.add(convertPolicy2Element(policy));
		}
		return res;
	}

	private Element convertPolicy2Element(Policy p) throws AddressingException {
		Element res = null;
		try {
			ByteArrayOutputStream out = new ByteArrayOutputStream();

			XMLOutputFactory factory = XMLOutputFactory.newInstance();
			XMLStreamWriter writer = factory.createXMLStreamWriter(out);
			p.serialize(writer);
			out.close();

			String policy = out.toString() + "</wsp:Policy>";
			InputStream input = new ByteArrayInputStream(policy.getBytes());

			Document doc = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(input);

			res = doc.getDocumentElement();

		} catch (XMLStreamException e) {
			throw new AddressingException(e);
		} catch (IOException e) {
			throw new AddressingException(e);
		} catch (SAXException e) {
			throw new AddressingException(e);
		} catch (ParserConfigurationException e) {
			throw new AddressingException(e);
		}
		return res;
	}

	public String marshall2String() throws AddressingException {
		try {
			final StringWriter stringWriter = new StringWriter();
			// TODO : Check if it is a Thread safe method
			getMarshaller().marshal(this.getEndpointReference(), stringWriter);
			return stringWriter.toString();
		} catch (JAXBException e) {
			throw new AddressingException(
					"Failed to build XML binding from EndpointReference descriptor Java classes", e);
		}
	}


	public Element marshall2Element() throws AddressingException {
		Document doc = null;
		try {
			doc = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();

			// TODO : Check if it is a Thread safe method
			getMarshaller().marshal(this.getEndpointReference(), doc);


		} catch (JAXBException ex) {
			throw new AddressingException(
					"Failed to build XML binding from EndpointReference descriptor Java classes", ex);
		} catch (ParserConfigurationException ex) {
			throw new AddressingException(
					"Failed to build XML binding from EndpointReference descriptor Java classes", ex);

		}
		return doc.getDocumentElement();
	}


	public static Endpoint unmarshall(final InputStream EndpointReferenceDescriptorStream)
	throws AddressingException {

		try {
			// TODO : Check if it is a Thread safe method
			final JAXBElement<EndpointReferenceType> EndpointReferenceBinding = getUnMarshaller().unmarshal(
					new StreamSource(EndpointReferenceDescriptorStream), EndpointReferenceType.class);

			return new Endpoint(EndpointReferenceBinding.getValue());

		} catch (JAXBException e) {
			throw new AddressingException(
					"Failed to build Java bindings from EndpointReference descriptor XML document", e);
		}
	}

	/**
	 * Get the unmarshaller instance.
	 *
	 * @return the unmarshaller instance
	 * @throws AddressingException
	 *             The exception raised during the unmarshaller creation.
	 */
	public static Unmarshaller getUnMarshaller() throws AddressingException {

		if (unmarshallerCreationEx != null) {
			throw unmarshallerCreationEx;
		}

		return unmarshaller;
	}

	/**
	 * Get the marshaller instance.
	 *
	 * @return the marshaller instance
	 * @throws AddressingException
	 *             The exception raised during the marshaller creation.
	 */
	public static Marshaller getMarshaller() throws AddressingException {

		if (marshallerCreationEx != null) {
			throw marshallerCreationEx;
		}

		return marshaller;
	}


	public List<Policy> getPolicies() {
		List<Policy> policies = new ArrayList<Policy>();
		if(this.any != null) {
			for (Object elmt : this.any) {
				if(elmt instanceof Element) {
					try {
						Policy p = PolicyEngine.getPolicy(XmlUtils.toOM((Element) elmt));
						policies.add(p);
					} catch (Exception e) {
						// do nothing
					}
				}
			}
		}
		return policies;
	}


	public void setPolicy(List<Policy> policies) throws AddressingException {
		if(policies != null) {
			if(this.any == null) {
				this.any = new ArrayList<Object>();
			}
			for (Element e : convertPolicies2Elements(policies)) {
				this.any.add(e);
			}
		}
	}
}
