/**
 * Web commons : persistence.
 * Copyright (c) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * GenericHibernateDAOImpl.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.webcommons.persistence.dao;

import java.io.Serializable;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.queryParser.MultiFieldQueryParser;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.util.Version;
import org.hibernate.NonUniqueResultException;
import org.hibernate.SessionFactory;
import org.hibernate.search.FullTextSession;
import org.springframework.orm.hibernate3.LocalSessionFactoryBean;
import org.springframework.orm.hibernate3.support.HibernateDaoSupport;

import com.ebmwebsourcing.webcommons.persistence.bo.BaseObject;
import com.ebmwebsourcing.webcommons.persistence.dao.util.HibernateQueryHelper;
import com.ebmwebsourcing.webcommons.persistence.dao.util.RequestOptions;
import com.trg.search.IMutableSearch;
import com.trg.search.ISearch;
import com.trg.search.Search;
import com.trg.search.SearchResult;
import com.trg.search.hibernate.HibernateSearchProcessor;

/**
 * This class serves as the Base class for all other DAOs - namely to hold
 * common CRUD methods that they might all use. You should only need to extend
 * this class when your require custom CRUD logic.
 * 
 * <p>
 * To register this class in your Spring context file, use the following XML.
 * 
 * <pre>
 *      &lt;bean id=&quot;fooDao&quot; class=&quot;org.ow2.dragon.persistence.dao.GenericDaoHibernate&quot;&gt;
 *          &lt;property name=&quot;type&quot; value=&quot;org.ow2.dragon.persistence.bo.Foo&quot;/&gt;
 *          &lt;property name=&quot;sessionFactory&quot; ref=&quot;sessionFactory&quot;/&gt;
 *      &lt;/bean&gt;
 * </pre>
 * 
 * @author ofabre - eBM WebSourcing
 * @param <T>
 *            a type variable
 * @param <PK>
 *            the primary key for that type
 * 
 */
public class GenericHibernateDAOImpl<T extends BaseObject, PK extends Serializable> extends
        HibernateDaoSupport implements GenericORMDAO<T, PK> {

    private final Logger logger = Logger.getLogger(this.getClass());

    private final Analyzer defaultAnalyser = new StandardAnalyzer(Version.LUCENE_29);

    private LocalSessionFactoryBean localSessionFactoryBean;

    public LocalSessionFactoryBean getLocalSessionFactoryBean() {
        return localSessionFactoryBean;
    }

    public void setLocalSessionFactoryBean(LocalSessionFactoryBean localSessionFactoryBean) {
        this.localSessionFactoryBean = localSessionFactoryBean;
    }

    public GenericHibernateDAOImpl(SessionFactory factory) {
        setSessionFactory(factory);
        setType();
    }

    public Class<T> type;

    @SuppressWarnings("unchecked")
    public void setType() {

        final Type thisType = getClass().getGenericInterfaces()[0];
        final Type genericType;
        if (thisType instanceof ParameterizedType) {
            genericType = ((ParameterizedType) thisType).getActualTypeArguments()[0];
        } else if (thisType instanceof Class) {
            genericType = ((ParameterizedType) ((Class) thisType).getGenericInterfaces()[0])
                    .getActualTypeArguments()[0];
        } else {
            throw new IllegalArgumentException("Problem handling type construction for "
                    + getClass());
        }

        // TODO Find a better way to do this !!!!!!
        if (genericType.getClass().getName().contains("java.lang.Class")) {
            logger.debug("###### Create dao for entity :"
                    + ((Class<T>) genericType).getCanonicalName());
            this.type = (Class<T>) genericType;
        }
        /*
         * if (genericType instanceof Class) { this.type = (Class<T>)
         * genericType; } else if (genericType instanceof ParameterizedType) {
         * this.type = (Class<T>) ((ParameterizedType)
         * genericType).getRawType(); } else { throw new
         * IllegalArgumentException
         * ("Problem determining the class of the generic for " + getClass()); }
         */
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public List<T> getAll(final RequestOptions requestOptions) {

        HibernateSearchProcessor searchProcessor = HibernateSearchProcessor
                .getInstanceForSessionFactory(getSessionFactory());

        ISearch search = HibernateQueryHelper.createSearchContext(this.type, null, null,
                requestOptions, null);

        return searchProcessor.search(getSession(), search);
    }

    /**
     * {@inheritDoc}
     */
    public List<T> getAll() {
        return super.getHibernateTemplate().loadAll(this.type);
    }

    public List<T> getAll(List<PK> ids) {
        return this.getAll(ids, null);
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public List<T> getAll(List<PK> ids, RequestOptions requestOptions) {
        HibernateSearchProcessor searchProcessor = HibernateSearchProcessor
                .getInstanceForSessionFactory(getSessionFactory());

        Search search = (Search) HibernateQueryHelper.createSearchContext(this.type, null, null,
                requestOptions, null);
        search.addFilterIn("id", ids);

        return searchProcessor.search(getSession(), search);
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public List<T> getAllDistinct() {
        Collection result = new LinkedHashSet(getAll());
        return new ArrayList(result);
    }

    /**
     * {@inheritDoc}
     */
    public T get(PK id) {
        T entity = super.getHibernateTemplate().get(this.type, id);

        return entity;
    }

    /**
     * {@inheritDoc}
     */
    public boolean exists(PK id) {
        T entity = super.getHibernateTemplate().get(this.type, id);
        return entity != null;
    }

    /**
     * {@inheritDoc}
     */
    public T save(T object) {
        super.getHibernateTemplate().saveOrUpdate(object);
        return object;
    }

    /**
     * {@inheritDoc}
     */
    public void remove(PK id) {
        T object = this.get(id);
        // if object is null, skip removal
        if (object != null) {
            super.getHibernateTemplate().delete(object);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void remove(T object) {
        // if object is null, skip removal
        if (object != null) {
            super.getHibernateTemplate().delete(object);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void removeAll(List<T> objects) {
        // if objects is null, skip removal
        if (objects != null) {
            for (T object : objects) {
                super.getHibernateTemplate().delete(object);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public List<T> findByNamedQuery(String queryName, Map<String, Object> queryParams) {
        String[] params = new String[queryParams.size()];
        Object[] values = new Object[queryParams.size()];
        int index = 0;
        Iterator<String> i = queryParams.keySet().iterator();
        while (i.hasNext()) {
            String key = i.next();
            params[index] = key;
            values[index++] = queryParams.get(key);
        }
        return getHibernateTemplate().findByNamedQueryAndNamedParam(queryName, params, values);
    }

    /**
     * {@inheritDoc}
     */
    public Class<T> getManipulatedType() {
        return this.type;
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public List<T> search(IMutableSearch search) {
        HibernateSearchProcessor searchProcessor = HibernateSearchProcessor
                .getInstanceForSessionFactory(getSessionFactory());
        search.setSearchClass(this.type);
        return searchProcessor.search(getSession(), search);
    }

    /**
     * {@inheritDoc}
     */
    public int count(IMutableSearch search) {
        HibernateSearchProcessor searchProcessor = HibernateSearchProcessor
                .getInstanceForSessionFactory(getSessionFactory());
        search.setSearchClass(this.type);
        return searchProcessor.count(getSession(), search);
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public SearchResult<T> searchAndCount(IMutableSearch search) {
        HibernateSearchProcessor searchProcessor = HibernateSearchProcessor
                .getInstanceForSessionFactory(getSessionFactory());
        search.setSearchClass(this.type);
        return searchProcessor.searchAndCount(getSession(), search);
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public T searchUnique(IMutableSearch search) throws NonUniqueResultException {
        HibernateSearchProcessor searchProcessor = HibernateSearchProcessor
                .getInstanceForSessionFactory(getSessionFactory());
        search.setSearchClass(this.type);
        return (T) searchProcessor.searchUnique(getSession(), search);
    }

    /**
     * {@inheritDoc}
     */
    public List<T> searchEquals(final String[] criteria, final String[] properties,
            final RequestOptions requestOptions) {
        return this.search(criteria, properties, requestOptions,
                HibernateQueryHelper.PREDICATE_EQUALS);
    }

    /**
     * {@inheritDoc}
     */
    public List<T> searchLike(final String[] criteria, final String[] properties,
            final RequestOptions requestOptions) {
        return this.search(criteria, properties, requestOptions,
                HibernateQueryHelper.PREDICATE_LIKE);
    }

    @SuppressWarnings("unchecked")
    private List<T> search(final String[] criteria, final String[] properties,
            final RequestOptions requestOptions, final String predicate) {
        HibernateSearchProcessor searchProcessor = HibernateSearchProcessor
                .getInstanceForSessionFactory(getSessionFactory());

        ISearch search = HibernateQueryHelper.createSearchContext(this.type, criteria, properties,
                requestOptions, predicate);

        return searchProcessor.search(getSession(), search);
    }

    /**
     * {@inheritDoc}
     * 
     * @throws DAOLayerException
     */
    public List<T> searchLucene(String[] criteria, String[] searchedProperties)
            throws DAOLayerException {
        if (type == null) {
            setType();
        }
        return this.searchLucene(criteria, searchedProperties, null);
    }

    /**
     * {@inheritDoc}
     * 
     * @throws DAOLayerException
     */
    @SuppressWarnings("unchecked")
    public List<T> searchLucene(String[] criteria, String[] searchedProperties,
            RequestOptions requestOptions) throws DAOLayerException {
        if (criteria.length == 0 || searchedProperties.length == 0) {
            throw new DAOLayerException("Search criterii and properties can't be empty.");
        }
        FullTextSession fullTextSession = getFullTextSession();
        org.apache.lucene.queryParser.QueryParser parser = new MultiFieldQueryParser(
                Version.LUCENE_29, searchedProperties, defaultAnalyser);
        String critString = createCritString(criteria);
        org.apache.lucene.search.Query luceneQuery;
        try {
            parser.setDefaultOperator(QueryParser.OR_OPERATOR);
            luceneQuery = parser.parse(critString);
        } catch (ParseException e) {
            throw new DAOLayerException("Can't parse lucene query", e);
        }
        logger.debug("Lucene query is: " + luceneQuery.toString());
        org.hibernate.search.FullTextQuery fullTextQuery = fullTextSession.createFullTextQuery(
                luceneQuery, this.getManipulatedType());
        // Sort on tokenized fields isn't allowed so if a sort option is
        // enabled, search on lucene and then retrieve orm objects from database
        // with sort options.
        List<T> result = null;
        if (requestOptions != null) {
            if (!requestOptions.hasSortOption()) {
                HibernateQueryHelper.applyRequestOptions(fullTextQuery, requestOptions, this
                        .getManipulatedType(), getSession());
                result = fullTextQuery.list();
            } else {
                result = fullTextQuery.list();
                List<PK> ids = createIdList(result);
                result = getAll(ids, requestOptions);
            }
        } else {
            result = fullTextQuery.list();
        }
        return result;
    }

    @SuppressWarnings("unchecked")
    private List<PK> createIdList(List<T> result) {
        List<PK> ids = new ArrayList<PK>();
        for (T t : result) {
            ids.add((PK) t.getId());
        }
        return ids;
    }

    private String createCritString(String[] criteria) {
        String result = "";
        for (String string : criteria) {
            result = result + " AND " + string;
        }
        return "(" + result.substring(4) + ")";
    }

    public FullTextSession getFullTextSession() {
        return org.hibernate.search.Search.getFullTextSession(getSession());
    }

}
