/**
 * Web commons : persistence.
 * Copyright (c) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * HibernateQueryHelper.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.webcommons.persistence.dao.util;

import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import org.apache.lucene.search.SortField;
import org.hibernate.Criteria;
import org.hibernate.FetchMode;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Order;
import org.springframework.orm.hibernate3.HibernateTemplate;

import com.trg.search.Filter;
import com.trg.search.ISearch;
import com.trg.search.Search;
import com.trg.search.Sort;

/**
 * A JPA Query Builder. It helps building complex JPA Query by abstracting some
 * query constructs like grouped OR or AND, order by, paging...
 * 
 * @author ofabre - ebmwebsourcing
 * 
 */
public class HibernateQueryHelper {

    public static final String CLAUSE_SELECT = "select";

    public static final String OPERATOR_DISTINCT = "distinct";

    public static final String CLAUSE_FROM = "from";

    public static final String CLAUSE_WHERE = "where";

    public static final String CLAUSE_GROUPBY = "group by";

    public static final String CLAUSE_ORDERBY = "order by";

    public static final String CLAUSE_HAVING = "having";

    public static final String OPERATOR_OR = "or";

    public static final String OPERATOR_AND = "and";

    public static final String PREDICATE_EQUALS = "=";

    public static final String PREDICATE_NOTEQUALS = "!=";

    public static final String PREDICATE_LIKE = "like";

    public static final String PREDICATE_IN = "in";

    public static final String SORT_ASC = "asc";

    public static final String SORT_DESC = "desc";

    public static final String WILDCARD = "%";

    public static final String INNER_JOIN = "inner join";

    public static final String LEFT_OUTER_JOIN = "left outer join";

    public static final String OPERATOR_UPPER = "UPPER";

    /**
     * A vector of parameter values
     */
    private Vector<Object> values = null;

    /**
     * The SQL query buffer
     */
    private StringBuffer sql = null;

    /**
     * Creates a new builder with an empty params vector and an empty query
     */
    public HibernateQueryHelper() {
        this.values = new Vector<Object>();
        this.sql = new StringBuffer();
    }

    /**
     * Creates a new builder with an empty params vector and a query buffer
     * initialized with the given sql string
     * 
     * @param sql
     *            an sql string
     */
    public HibernateQueryHelper(String sql) {
        this.values = new Vector<Object>();
        this.sql = new StringBuffer(sql);
    }

    /**
     * Append the given sql string to the inner sql buffer
     * 
     * @param sql
     *            an sql string
     * @return this query builder
     */
    public HibernateQueryHelper append(String sql) {
        this.sql.append(sql);
        return this;
    }

    /**
     * Append a blank character (" ") to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper pad() {
        this.sql.append(" ");
        return this;
    }

    /**
     * Append a dot character (".") to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper dot() {
        this.sql.append(".");
        return this;
    }

    /**
     * Append an open paren character ("(") to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper openParen() {
        this.sql.append("(");
        return this;
    }

    /**
     * Append an close paren character (")") to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper closeParen() {
        this.sql.append(")");
        return this;
    }

    /**
     * Append a question mark character ("?") to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper param() {
        this.sql.append("?");
        return this;
    }

    /**
     * Append a comma character (",") to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper comma() {
        this.sql.append(",");
        return this;
    }

    /**
     * Append an sql "select" clause to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper SELECT() {
        this.sql.append(CLAUSE_SELECT);
        return this;
    }

    /**
     * Append an sql "distinct" operator to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper DISTINCT() {
        this.sql.append(OPERATOR_DISTINCT);
        return this;
    }

    /**
     * Append an sql "from" clause to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper FROM() {
        this.sql.append(CLAUSE_FROM);
        return this;
    }

    /**
     * Append an sql "and" operator to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper AND() {
        this.sql.append(OPERATOR_AND);
        return this;
    }

    /**
     * Append an sql "or" operator to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper OR() {
        this.sql.append(OPERATOR_OR);
        return this;
    }

    /**
     * Append an sql "UPPER" operator to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper UPPER() {
        this.sql.append(OPERATOR_UPPER);
        return this;
    }

    /**
     * Append an sql "where" operator to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper WHERE() {
        this.sql.append(CLAUSE_WHERE);
        return this;
    }

    /**
     * Append an sql "in" operator to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper IN() {
        this.sql.append(PREDICATE_IN);
        return this;
    }

    /**
     * Append an sql "group by" operator to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper GROUPBY() {
        this.sql.append(CLAUSE_GROUPBY);
        return this;
    }

    /**
     * Append an sql "order by" operator to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper ORDERBY() {
        this.sql.append(CLAUSE_ORDERBY);
        return this;
    }

    /**
     * Append an sql "having" operator to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper HAVING() {
        this.sql.append(CLAUSE_HAVING);
        return this;
    }

    /**
     * Append an sql "inner join" operator to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper INNER_JOIN() {
        this.sql.append(INNER_JOIN);
        return this;
    }

    /**
     * Append an sql "left outer join" operator to the inner sql buffer
     * 
     * @return this query builder
     */
    public HibernateQueryHelper LEFT_OUTER_JOIN() {
        this.sql.append(LEFT_OUTER_JOIN);
        return this;
    }

    /**
     * See {@link #appendCondition(String, Parameter...)}. The operator used is
     * "and"
     * 
     * @param params
     *            a list of parameters to link with the operator. Nothing
     *            appended if null or empty
     * @return this query builder
     */
    public HibernateQueryHelper appendGroupedAnd(Parameter... params) {
        return appendCondition(OPERATOR_AND, params);
    }

    /**
     * See {@link #appendCondition(String, Parameter...)}. The operator used is
     * "or"
     * 
     * @param params
     *            a list of parameters to link with the operator. Nothing
     *            appended if null or empty
     * @return this query builder
     */
    public HibernateQueryHelper appendGroupedOr(Parameter... params) {
        return appendCondition(OPERATOR_OR, params);
    }

    /**
     * Appends a list of conditions on parameter values, linked by the specified
     * operator. For exemple, if the given operator is "or" and two params
     * (name:foo1, value:bar1, predicate:like) (name:foo2, value:bar2,
     * predicate:like), then the appended string is : (foo1 like ? or foo2 like
     * ?) ; in the same time two values are added to the builder inner values
     * vector (bar1, bar2)
     * 
     * @param operator
     *            an operator ("or", "and",...). Nothing appended if null or
     *            empty
     * @param params
     *            a list of parameters to link with the operator. Nothing
     *            appended if null or empty
     * @return this query builder
     */
    public HibernateQueryHelper appendCondition(String operator, Parameter... params) {
        if (params == null || operator == null)
            return this;

        openParen();
        for (int i = 0; i < params.length; i++) {
            if (params[i] != null) {
                append(params[i].getName()).pad().append(params[i].getPredicate()).pad().param();
                addValue(params[i].getValue());

                if (i + 1 < params.length) {
                    pad().append(operator).pad();
                }
            }
        }
        closeParen().pad();
        return this;

    }

    /**
     * Append a "where" clause to the sql query.
     * 
     * @param criterii
     *            an array of criterii ( {@link String} )
     * @param properties
     *            the properties to search on
     * @param queryString
     *            the {@link StringBuffer} representing the hibernate query
     * @param clazzAlias
     *            the simple class name used as alias
     * @param predicate
     *            type of search (strict equality, similarity...)
     * @param caseSensitive
     *            true if the search must be done with case sensitivity enabled
     */
    public HibernateQueryHelper appendSearchCriteriaInQuery(final String[] criterii,
            final String[] properties, final String clazzAlias, final String predicate,
            final boolean caseSensitive) {
        // Cleanup properties
        // String[] cleanProperties = cleanUpProperties(properties);

        // Create where clause
        WHERE().pad();
        for (int i = 0; i < criterii.length; i++) {
            openParen();
            List<Parameter> parameters = createSearchParameters(properties, criterii[i],
                    clazzAlias, predicate, caseSensitive);
            appendGroupedOr(parameters.toArray(new Parameter[0]));
            closeParen();
            if (i + 1 < criterii.length) {
                pad().AND().pad();
            }
        }
        pad();
        return this;
    }

    private List<Parameter> createSearchParameters(String[] properties, String criteria,
            String clazzAlias, String predicate, boolean caseSensitive) {
        List<Parameter> parameters = new ArrayList<Parameter>();
        for (final String p : properties) {
            Parameter parameter = new Parameter(
                    createSearchedProperty(clazzAlias, p, caseSensitive), createSearchedValue(
                            criteria, caseSensitive, predicate), predicate);
            parameters.add(parameter);
        }
        return parameters;
    }

    private String createSearchedValue(String criteria, boolean caseSensitive, String predicate) {
        String value;
        if (PREDICATE_EQUALS.equals(predicate)) {
            value = (caseSensitive ? criteria : criteria.toUpperCase());
        } else {
            value = WILDCARD + (caseSensitive ? criteria : criteria.toUpperCase()) + WILDCARD;
        }
        return value;
    }

    /**
     * Create a searched property.
     * 
     * @param clazzAlias
     *            the simple class name used as alias
     * @param p
     *            a property (can be a simple or dotted property)
     * @param caseSensitive
     *            true if the search must be done with case sensitivity enabled
     */
    public String createSearchedProperty(final String clazzAlias, final String p,
            final boolean caseSensitive) {
        StringBuffer property = new StringBuffer("");
        if (!caseSensitive) {
            property.append(OPERATOR_UPPER).append("(");
        }
        if (!p.contains(".")) {
            // a simple property (deepness 1 level): simple class name used as
            // alias
            property.append(clazzAlias).append(".").append(p);
        } else {
            // a dotted property (deepness multi level): "last-1" field name
            // used as alias (thanks to join statement)
            final String[] pSplit = p.split("\\.");
            final int length = pSplit.length;
            property.append(pSplit[length - 2]).append(".").append(pSplit[length - 1]);
        }
        if (!caseSensitive) {
            property.append(") ");
        }
        return property.toString();
    }

    /**
     * Append an inner join statement for a given field.
     * 
     * @param parentAlias
     *            the alias of the parent part of the join statement. Nothing
     *            appended if null or empty
     * @param fieldName
     *            the field name to join. Nothing appended if null or empty
     * @param joinedAlias
     *            an alias for the joined field. Nothing appended if null or
     *            empty
     * @return this query builder
     */
    public HibernateQueryHelper appendInnerJoinStatement(final String parentAlias,
            final String fieldName, final String joinedAlias) {
        if (parentAlias == null || fieldName == null || joinedAlias == null)
            return this;

        INNER_JOIN().pad().append(parentAlias).dot().append(fieldName).pad().append(joinedAlias)
                .pad();
        return this;
    }

    /**
     * Append an left outer join statement for a given field.
     * 
     * @param parentAlias
     *            the alias of the parent part of the join statement. Nothing
     *            appended if null or empty
     * @param fieldName
     *            the field name to join. Nothing appended if null or empty
     * @param joinedAlias
     *            an alias for the joined field. Nothing appended if null or
     *            empty
     * @return this query builder
     */
    public HibernateQueryHelper appendLeftOuterJoinStatement(final String parentAlias,
            final String fieldName, final String joinedAlias) {
        if (parentAlias == null || fieldName == null || joinedAlias == null)
            return this;

        LEFT_OUTER_JOIN().pad().append(parentAlias).dot().append(fieldName).pad().append(
                joinedAlias).pad();
        return this;
    }

    /**
     * For each properties: if the property is a dotted property (contains one
     * or more "." character), append an inner join statement for each field
     * following the dot.
     * 
     * @param parentAlias
     *            the alias of the parent part of all join statement
     * @param properties
     *            properties array containing possible dotted properties.
     *            Nothing appended if null or empty
     * @return this query builder
     */
    public HibernateQueryHelper appendMultipleInnerJoinStatement(final String parentAlias,
            final String... properties) {
        if (properties == null || properties.length == 0)
            return this;

        for (final String property : properties) {
            if (property.contains(".")) {
                final String[] propSplit = property.split("\\.");
                final String fieldName = propSplit[0];
                appendInnerJoinStatement(parentAlias, fieldName, fieldName);
                for (int i = 1; i < propSplit.length - 1; i++) {
                    appendInnerJoinStatement(propSplit[i - 1], propSplit[i], propSplit[i]);
                }
            }
        }
        return this;
    }

    /**
     * For each properties: if the property is a dotted property (contains one
     * or more "." character), append an left outer join statement for each
     * field following the dot.
     * 
     * @param parentAlias
     *            the alias of the parent part of all join statement
     * @param properties
     *            properties array containing possible dotted properties.
     *            Nothing appended if null or empty
     * @return this query builder
     */
    public HibernateQueryHelper appendMultipleLeftOuterJoinStatement(final String parentAlias,
            final String... properties) {
        if (properties == null || properties.length == 0)
            return this;

        for (final String property : properties) {
            if (property.contains(".")) {
                final String[] propSplit = property.split("\\.");
                final String fieldName = propSplit[0];
                appendLeftOuterJoinStatement(parentAlias, fieldName, fieldName);
                for (int i = 1; i < propSplit.length - 1; i++) {
                    appendLeftOuterJoinStatement(propSplit[i - 1], propSplit[i], propSplit[i]);
                }
            }
        }
        return this;
    }

    /**
     * Appends a select distinct statement with the simpleClazzName.toLowerCase
     * as alias
     * 
     * @param simpleClazzName
     *            the simple class name used as alias
     * @param fullClazzName
     *            the full class name used for the select statement
     */
    public HibernateQueryHelper appendSelectDistinctStatement(final String simpleClazzName,
            final String fullClazzName) {
        SELECT().pad().DISTINCT().pad().append(simpleClazzName.toLowerCase()).pad().FROM().pad()
                .append(fullClazzName).pad().append(simpleClazzName.toLowerCase()).pad();
        return this;
    }

    /**
     * Appends a select statement with the simpleClazzName.toLowerCase as alias
     * 
     * @param simpleClazzName
     *            the simple class name used as alias
     * @param fullClazzName
     *            the full class name used for the select statement
     */
    public HibernateQueryHelper appendSelectStatement(final String simpleClazzName,
            final String fullClazzName) {
        SELECT().pad().append(simpleClazzName.toLowerCase()).pad().FROM().pad().append(
                fullClazzName).pad().append(simpleClazzName.toLowerCase()).pad();
        return this;
    }

    /**
     * Append an "order by" sql clause to the inner sql string. The property on
     * which the order by is applied is defined in the request options
     * 
     * @param requestOptions
     *            request option defining sort criteria and pagination
     * @param clazzAlias
     *            the alias of the entities that will be ordered
     * @return this query builder
     */
    public HibernateQueryHelper appendSortOption(final RequestOptions requestOptions,
            final String clazzAlias) {
        if (requestOptions != null) {
            if (requestOptions.hasSortOption()) {
                ORDERBY().pad().append(clazzAlias).dot().append(requestOptions.getSortCriteria())
                        .pad();
                if (requestOptions.isSortAscendingly()) {
                    append(SORT_ASC).pad();
                } else {
                    append(SORT_DESC).pad();
                }
            }
        }
        return this;
    }

    /**
     * See {@link #appendInList(String, List)}. Append a "and" operator before
     * appending list condition
     * 
     * @param term
     *            a string term like "foo"
     * @param list
     *            a list of values like "bar1", "bar2". Nothing appended if null
     *            or empty
     * @return this query builder
     */
    public HibernateQueryHelper appendInListWithAnd(String term, List<?> list) {
        if (list == null || list.size() == 0)
            return this;

        AND().pad();

        return appendInList(term, list);
    }

    /**
     * Appends a "in" list condition on a particuler term. For exemple, if the
     * term is "foo" and the list contains the given string values ("bar1",
     * "bar2"), then the appended string is : foo in ("?", "?") ; in the same
     * time two values are added to the builder inner values vector (bar1, bar2)
     * 
     * @param term
     *            a string term like "foo"
     * @param list
     *            a list of values like "bar1", "bar2". Nothing appended if null
     *            or empty
     * @return
     */
    public HibernateQueryHelper appendInList(String term, List<?> list) {
        if (list == null || list.size() == 0)
            return this;

        append(term).pad().IN().pad().openParen().pad();
        int count = 0;
        for (Object item : list) {
            param();
            addValue(item);

            if (count + 1 < list.size())
                comma().pad();

            count++;
        }
        closeParen().pad();
        return this;
    }

    /**
     * Add a values to the inner values vector
     * 
     * @param obj
     *            the value to add
     */
    public void addValue(Object obj) {
        this.values.addElement(obj);
    }

    /**
     * Build a JPA query from the inner sql statement and values
     * 
     * @param em
     *            a JPA entity manager
     * @return a JPA {@link Query}
     */
    public Query buildHibernateQuery(Session session) {
        Query qry = session.createQuery(sql.toString());

        /*
         * System.out.println("######## " + sql.toString());
         * System.out.println("######## " + values);
         */

        for (int i = 0; i < values.size(); i++)
            qry.setParameter(i, values.elementAt(i));

        return qry;
    }

    /**
     * Build a JPA query from the inner sql statement and values. Paginate it
     * following the given request options.
     * 
     * @param em
     *            a JPA entity manager
     * @return a JPA {@link Query}
     */
    public Query buildHibernatePaginatedQuery(Session session, RequestOptions requestOptions) {
        Query query = buildHibernateQuery(session);
        if (requestOptions != null) {
            // --- 2: paginate
            if (requestOptions.hasPagination()) {
                query.setFirstResult(requestOptions.getFirstResult());
                query.setMaxResults(requestOptions.getNbOfResults());
            }
        }
        return query;
    }

    @Override
    public String toString() {
        StringBuffer buffer = new StringBuffer(sql.toString());
        buffer.append("\n\n");

        for (int i = 0; i < values.size(); i++) {
            Object obj = values.elementAt(i);

            buffer.append(i + 1);
            buffer.append("\t");
            buffer.append(obj.getClass().getName());
            buffer.append("\t");
            buffer.append(obj.toString());
            buffer.append("\n");
        }

        return buffer.toString();
    }

    /**
     * A class that handles parameter name, value and predicate
     * 
     * @author ofabre - ebmwebsourcing
     * 
     */
    public static class Parameter {
        /**
         * A parameter name
         */
        private String name;

        /**
         * A parameter value
         */
        private Object value;

        /**
         * A predicate like "equals" "like"
         */
        private String predicate;

        /**
         * Build a new {@link Parameter} with the given name, value and
         * predicate
         */
        public Parameter(String name, Object value, String predicate) {
            this.name = name;
            this.value = value;
            this.predicate = predicate;
        }

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public Object getValue() {
            return value;
        }

        public void setValue(Object value) {
            this.value = value;
        }

        public String getPredicate() {
            return predicate;
        }

        public void setPredicate(String predicate) {
            this.predicate = predicate;
        }

    }

    @SuppressWarnings("unchecked")
    public static List sortAndPaginateByCriteria(HibernateTemplate template,
            final RequestOptions requestOptions, final DetachedCriteria detachedCriteria) {
        List result;
        if (requestOptions != null) {
            // --- 1: sort
            if (requestOptions.hasSortOption()) {
                if (requestOptions.isSortAscendingly()) {
                    detachedCriteria.addOrder(Order.asc(requestOptions.getSortCriteria()));
                } else {
                    detachedCriteria.addOrder(Order.desc(requestOptions.getSortCriteria()));
                }
            }
            // --- 2: paginate
            if (requestOptions.hasPagination()) {
                result = template.findByCriteria(detachedCriteria, requestOptions.getFirstResult(),
                        requestOptions.getNbOfResults());
            } else {
                result = template.findByCriteria(detachedCriteria);
            }
        } else {
            result = template.findByCriteria(detachedCriteria);
        }
        return result;
    }

    public static ISearch createSearchContext(Class<?> clazz, String[] criterii,
            String[] properties, RequestOptions requestOptions, String predicate) {
        Search search = new Search();
        search.setSearchClass(clazz);
        search.setDistinct(true);

        // Filter on criteria and properties
        if ((criterii != null) && (properties != null) && (criterii.length > 0)
                && (properties.length > 0)) {
            for (String criteria : criterii) {
                Filter filter = Filter.or();
                for (String property : properties) {
                    if (HibernateQueryHelper.PREDICATE_EQUALS.equalsIgnoreCase(predicate)) {
                        if (requestOptions != null && !requestOptions.isCaseSensitive()) {
                            filter.add(Filter.iequal(property, criteria));
                        } else {
                            filter.add(Filter.equal(property, criteria));
                        }
                    } else if (HibernateQueryHelper.PREDICATE_LIKE.equalsIgnoreCase(predicate)) {
                        if (requestOptions != null && !requestOptions.isCaseSensitive()) {
                            filter.add(Filter.ilike(property, HibernateQueryHelper.WILDCARD
                                    + criteria + HibernateQueryHelper.WILDCARD));
                        } else {
                            filter.add(Filter.like(property, HibernateQueryHelper.WILDCARD
                                    + criteria + HibernateQueryHelper.WILDCARD));
                        }
                    }
                }
                search.addFilterAnd(filter);
            }
        }

        applyRequestOptions(search, requestOptions);

        return search;
    }

    public static void applyRequestOptions(Search search, RequestOptions requestOptions) {
        if (requestOptions != null) {
            // Set Paging
            if (requestOptions.hasPagination()) {
                search.setFirstResult(requestOptions.getFirstResult());
                search.setMaxResults(requestOptions.getNbOfResults());
            }

            // Set Sort
            if (requestOptions.hasSortOption()) {
                Sort sort = new Sort();
                sort.setProperty(requestOptions.getSortCriteria());
                if (requestOptions.isSortAscendingly()) {
                    sort.setDesc(false);
                } else {
                    sort.setDesc(true);
                }
                search.addSort(sort);
            }

            // Set Fetch
            if (requestOptions.hasFetchOption()) {
                search.addFetch(requestOptions.getFetchCriteria());
            }
        }
    }

    public static void applyRequestOptions(org.hibernate.search.FullTextQuery fullTextQuery,
            RequestOptions requestOptions, Class clazz, Session session) {
        if (requestOptions != null) {
            // Set Paging
            if (requestOptions.hasPagination()) {
                fullTextQuery.setFirstResult(requestOptions.getFirstResult());
                fullTextQuery.setMaxResults(requestOptions.getNbOfResults());
            }

            // Set Sort
            if (requestOptions.hasSortOption()) {
                org.apache.lucene.search.Sort sort = new org.apache.lucene.search.Sort(
                        new SortField(requestOptions.getSortCriteria(), !requestOptions
                                .isSortAscendingly()));
                fullTextQuery.setSort(sort);
            }

            // Set Fetch
            if (requestOptions.hasFetchOption()) {
                Criteria criteria = session.createCriteria(clazz).setFetchMode(
                        requestOptions.getFetchCriteria(), FetchMode.JOIN);
                fullTextQuery.setCriteriaQuery(criteria);
            }
        }
    }

}
