/**
 * Web commons : user service.
 * Copyright (c) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * RoleGroupManager.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.webcommons.user.api.service;

/**
 * A manager of Role Group. Provides methods to create, retrieve, delete
 * or update role Group and to manage Role related to role group. 
 * Method arguments must be non null, unless the
 * contrary is explicitly specified.
 * 
 * @author strino - eBM WebSourcing
 * 
 */

import java.io.IOException;
import java.util.List;

import org.springframework.transaction.annotation.Transactional;

import com.ebmwebsourcing.webcommons.user.api.to.RoleGroupTO;
import com.ebmwebsourcing.webcommons.user.persistence.bo.RoleGroup;

@Transactional
public interface RoleGroupManager {

    /**
     * Create default role groups if they don't exist in database
     * 
     */
    public void init() throws IOException, RoleGroupException;

    /**
     * Add a new Role Group in registry. The Role Group name must be specified.
     * 
     * @param roleGroupTO
     *            the {@link RoleGroupTO} handling data of the {@link RoleGroup}
     *            to create
     * @return the id of the added {@link RoleGroup}
     * @throws RoleGroupException
     *             if you try to create an role group with name that already
     *             exists or an role group without name
     */
    public String createRoleGroup(RoleGroupTO roleGroupTO) throws RoleGroupException;

    /**
     * Return the list of all role group in the registry
     * 
     * @return the {@link List} of all {@link RoleGroupTO} in the registry,
     *         couldn't be null, can be empty
     */
    @Transactional(readOnly = true)
    public List<RoleGroupTO> getAllRoleGroup() throws RoleGroupException;

    /**
     * Return the list of role group associate with user
     * 
     * @param idUser
     *            an {@link User} id
     * @return the {@link List} of role group associate with {@link user} ,
     *         couldn't be null, can be empty
     */
    @Transactional(readOnly = true)
    public List<RoleGroupTO> getRoleGroupNotInUser(String idUser) throws RoleGroupException;

    /**
     * Retrieve the role group matching the given name
     * 
     * @param nameRoleGroup
     *            an {@link RoleGroup} name
     * @return the {@link RoleGroupTO} handling data of the {@link RoleGroup}
     *         that matches the given id
     * @throws RoleGroupException
     *             if no role group found for the given name
     */
    @Transactional(readOnly = true)
    public RoleGroupTO getRoleGroup(String nameRoleGroup) throws RoleGroupException;

    /**
     * Remove the role group matching the given id from the registry.
     * 
     * @param roleGroupId
     *            an {@link RoleGroup} id
     */
    public void removeRoleGroup(String roleGroupId) throws RoleGroupException;

    /**
     * Update the role group in the registry.
     * 
     * @param roleGroupTO
     *            the {@link RoleGroupTO} handling data of the {@link RoleGroup}
     *            to update
     * @return the updated role group ID
     * @throws RoleGroupException
     *             if an error occurs during role group update : role group to
     *             update doesn't exist or have the same name as an other
     *             registered role group
     */
    public String updateRoleGroup(RoleGroupTO roleGroupTO) throws RoleGroupException;

    /**
     * Add the user matching the given idUser to the roleGroup matching the
     * given idRoleGroup
     * 
     * @param idUser
     *            an {@link User} id
     * @param idRoleGroup
     *            a {@link RoleGroup} id
     * @throws RoleGroupException
     *             if at least one of the given id doesn't exist in database
     */
    public void addUser(String idUser, String idRoleGroup) throws RoleGroupException;

    /**
     * Remove the user matching the given idUser from the list of users of the
     * roleGroup matching the given idRoleGroup. The user isn't removed from the
     * registry and can be re-affected to any roles
     * 
     * @param idUser
     *            an {@link User} id
     * @param idRoleGroup
     *            a {@link RoleGroup} id
     * @throws RoleGroupException
     *             if no organization found for the given id
     */
    public void removeUser(String idUser, String idRoleGroup) throws RoleGroupException;
}
