/**
 * easySchema - easyWSDL toolbox Platform.
 * Copyright (c) 2008,  eBM Websourcing
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.ebmwebsourcing.agreement.definition.api;

import java.net.URI;
import java.util.Map;

import javax.xml.transform.dom.DOMSource;

import org.w3c.dom.Document;
import org.xml.sax.InputSource;

/**
 * This interface describes a collection of methods that enable conversion of a
 * SchemaImpl document (in XML, following the SchemaImpl parent described in the
 * SchemaImpl specification) into a SchemaImpl model.
 * 
 * @author Nicolas Salatge - eBM WebSourcing
 */
public interface WSAgreementReader {

    /**
     * Constants for the Message Exchange Patterns.
     * 
     */
    public enum FeatureConstants {
        VERBOSE("org.ow2.easywsdl.schema.verbose"), IMPORT_DOCUMENTS(
                "org.ow2.easywsdl.schema.importDocuments");

        private final String value;

        /**
         * Creates a new instance of {@link FeatureConstants}
         * 
         * @param value
         */
        private FeatureConstants(final String value) {
            this.value = value;
        }

        /**
         * 
         * @return
         */
        public String value() {
            return this.value;
        }

        /**
         * Please use this equals method instead of using :<code>
         * value().equals(value)
         * </code> which is
         * almost 10 times slower...
         * 
         * @param value
         * @return
         */
        public boolean equals(final String val) {
            return this.toString().equals(val);
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.lang.Enum#toString()
         */
        @Override
        public String toString() {
            return this.value;
        }
    }

    /**
     * Sets the specified feature to the specified value.
     * <p>
     * The minimum features that must be supported are:
     * <p>
     * <table border=1>
     * <tr>
     * <th>Name</th>
     * <th>Description</th>
     * <th>Default Value</th>
     * </tr>
     * <tr>
     * <td><center>org.ow2.easywsdl.schema.verbose</center></td>
     * <td>If set to true, status messages will be displayed.</td>
     * <td><center>type: boolean - default value: false</center></td>
     * </tr>
     * <tr>
     * <td><center>org.ow2.easywsdl.schema.importDocuments</center></td>
     * <td>If set to true, imported WSDL documents will be retrieved and
     * processed.</td>
     * <td><center>type: boolean - default value: true</center></td>
     * </tr>
     * <tr>
     * <td>
     * <center>org.ow2.easywsdl.schema.pathDirectoryOfImportLocations
     * </center></td>
     * <td>If the location is set, imported WSDL documents will be retrieved at
     * this location (Set the importDocuments Features at true).</td>
     * <td><center>type: String</center></td>
     * </tr>
     * </table>
     * <p>
     * All feature names must be fully-qualified, Java package style. All names
     * starting with com.ebmwebsourcing. are reserved for features defined by
     * the specification. It is recommended that implementation- specific
     * features be fully-qualified to match the package name of that
     * implementation. For example: com.abc.featureName
     * 
     * @param name
     *            the name of the feature to be set.
     * @param value
     *            the value to set the feature to.
     * @throws IllegalArgumentException
     *             if the feature name is not recognized.
     * @see #getFeature(String)
     */
    public void setFeature(FeatureConstants name, Object value) throws IllegalArgumentException;

    /**
     * Gets the value of the specified feature.
     * 
     * @param name
     *            the name of the feature to get the value of.
     * @return the value of feature
     * @throws IllegalArgumentException
     *             if the feature name is not recognized.
     * @see #setFeature(String, boolean)
     */
    public Object getFeature(FeatureConstants name);

    /**
     * Gets all features.
     * 
     * @return the features
     * @see #setFeature(String, boolean)
     */
    public Map<FeatureConstants, Object> getFeatures();

    /**
     * Read the SchemaImpl document accessible via the specified URI into a
     * SchemaImpl definition.
     * 
     * @param schemaURI
     *            a URI (can be a filename or URL) pointing to a SchemaImpl XML
     *            definition.
     * @return the definition.
     */
    public Agreement readAgreement(URI uri) throws WSAgreementException;

    /**
     * Read the specified SchemaImpl document into a SchemaImpl definition.
     * 
     * @param documentBaseURI
     *            the document base URI of the SchemaImpl definition described
     *            by the document. Will be set as the documentBaseURI of the
     *            returned Definition. Can be null, in which case it will be
     *            ignored.
     * @param wsdlDocument
     *            the SchemaImpl document, an XML document obeying the
     *            SchemaImpl parent.
     * @return the definition described in the document.
     */
    public Agreement readAgreement(Document doc) throws WSAgreementException;

    /**
     * Read a SchemaImpl document into a SchemaImpl definition.
     * 
     * @param schemaURI
     *            the uri of the parent
     * @param inputSource
     *            an InputSource pointing to the SchemaImpl document, an XML
     *            document obeying the SchemaImpl parent.
     * @return the definition described in the document pointed to by the
     *         InputSource.
     */
    public Agreement readAgreement(URI uri, InputSource inputSource) throws WSAgreementException;

}
