/**
 * model-manager - Handles models on client side for stuffs like undo/redo, methods observers, uibinding ... - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.cloneable;

import java.lang.reflect.Modifier;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.Utils;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.cloneable.ICloneable;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.cloneable.ICloneableProxy;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.cloneable.annotation.CloneableField;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.cloneable.annotation.CloneableFields;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.Body;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.ClassTypeHelper;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.ComposerHelper;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.JClassTypeHelper;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.JTypeHelper;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.Method;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.Visibility;
import com.google.gwt.core.client.GWT;
import com.google.gwt.core.ext.Generator;
import com.google.gwt.core.ext.GeneratorContext;
import com.google.gwt.core.ext.TreeLogger;
import com.google.gwt.core.ext.UnableToCompleteException;
import com.google.gwt.core.ext.typeinfo.JClassType;
import com.google.gwt.core.ext.typeinfo.JField;
import com.google.gwt.core.ext.typeinfo.JType;

public class CloneableProxyGenerator extends Generator {

	private static final String getCloneableProxyMethod = "getCloneableProxy";
	private static final String mapName = "alreadyCopied";


	private String generatedClassName;
	private JClassType type;
	private CloneableFields cloneableFields;
	private ComposerHelper composerHelper;
	private JClassTypeHelper helper;
	private int loopCount = 0;
	private int varCount = 0;


	@Override
	public String generate(TreeLogger logger, GeneratorContext context,
			String typeName) throws UnableToCompleteException {
		helper = new JClassTypeHelper(type);

		generatedClassName = type.getSimpleSourceName()+"_CloneableProxy";
		composerHelper	= new ComposerHelper(context, logger, type.getPackage().getName(), generatedClassName);
		composerHelper.setSuperClass(typeName);//typeName should be the name of a reflection proxy
		composerHelper.addImport(GWT.class);
		composerHelper.addImport(HashMap.class);

		composerHelper.addInterface(ICloneableProxy.class.getCanonicalName()+"<"+type.getSimpleSourceName()+">");

		try {
			validateAnnotation();
		} catch (IncompleteAnnotation e) {
			e.printStackTrace();
			throw new UnableToCompleteException();
		}

		cloneableFields = type.getAnnotation(CloneableFields.class);
		if(cloneableFields!=null && cloneableFields.cloneableFields().length>0) {
			generateCloneableFieldsInstantiationMethod();
		}

		generateSimpleCopyMethod();

		generateCopyMethod();

		composerHelper.commit();
		return composerHelper.getCreatedClassName();
	}



	private void generateSimpleCopyMethod() {
		Method copyMethod = new Method(Visibility.PUBLIC,void.class,"copy");
		String copyParam = "other";
		copyMethod.addParameter(copyMethod.new Parameter(type, copyParam));

		copyMethod.getBody().append("copy("+copyParam+", new HashMap())");

		composerHelper.addMethod(copyMethod);
	}




	private void generateCopyMethod() {
		Method copyMethod = new Method(Visibility.PUBLIC,void.class,"copy");
		String copyParam = "other";
		copyMethod.addParameter(copyMethod.new Parameter(type, copyParam));
		copyMethod.addParameter(copyMethod.new Parameter(Map.class, mapName));

		for(JField field : helper.getAllFields()) {
			if(ignoreField(field)) continue;

			Class<?> fieldClass = JTypeHelper.getClass(field.getType());
			String getter = Utils.getGetterMethodByFieldName(field.getName(), Boolean.class.isAssignableFrom(fieldClass));
			String setter = Utils.getSetterMethodByFieldName(field.getName());

			String attachedElt = appendFieldAttachingCode(field.getType(), copyParam+"."+getter+"()", copyMethod.getBody());
			copyMethod.getBody().append(setter+"("+attachedElt+")");
		}

		composerHelper.addMethod(copyMethod);
	}


	private String appendFieldAttachingCode(JType copySourceType, String copySourceName, Body methodBody) {
		Class<?> copySourceClass = JTypeHelper.getClass(copySourceType);
		String copySourceClassName = copySourceClass.getCanonicalName();

		if(copySourceType.isArray()!=null || Collection.class.isAssignableFrom(copySourceClass)) {
			//if the field is a collection : iterate through it and copy all the fields it contains
			//we have to visit all the elements of the collection in case they would need to be
			//rebinded as cloneableproxies
			String collectionName = getVariableName();
			methodBody.append(copySourceClassName+" "+collectionName+" = new "+ClassTypeHelper.getCollectionSubTypeName(copySourceClass)+"()");

			JClassType collectionParamType = copySourceType.isParameterized().getTypeArgs()[0];

			String collectionEltName = getCollectionElementName();
			methodBody.append("for("+collectionParamType.getQualifiedSourceName()+" "+collectionEltName+" : "+copySourceName+") {");
			String attachedElt = appendFieldAttachingCode(collectionParamType, collectionEltName, methodBody);
			methodBody.append(collectionName+".add("+attachedElt+")");
			methodBody.append("}");

			return collectionName;
		}
		else if(Map.class.isAssignableFrom(copySourceClass)) {
			//TODO
			return null;
		}
		else {
			if(ClassTypeHelper.isPrimitive(copySourceClass) 
					|| String.class.equals(copySourceClass)
					|| copySourceClass.isEnum()) {
				return copySourceName;
			}
			else {
				if(ICloneable.class.isAssignableFrom(copySourceClass)
						|| ICloneableProxy.class.isAssignableFrom(copySourceClass)) {
					//if the field is cloneable :  
					String varName = getVariableName();
					methodBody.append(copySourceClassName+" "+varName+" = null");

					//if the copy source is null do nothing
					methodBody.append("if("+copySourceName+"!=null) {");

					//check if a cloneable proxy already has been generated for it 
					methodBody.append("if("+mapName+".containsKey("+copySourceName+")) {");
					methodBody.append(varName+" = ("+copySourceClassName+") "+mapName+".get("+copySourceName+")");
					methodBody.append("}");

					//else GWT.create of its type in order to get a cloneableproxy from it.
					methodBody.append("else {");
					if(copySourceClass.isInterface()) {
						methodBody.append(varName+" = ("+copySourceClassName+") "+getCloneableProxyMethod+"("+copySourceName+")");
					}
					else {
						methodBody.append(varName+" = GWT.create("+copySourceClassName+".class)");
					}

					methodBody.append(mapName+".put("+copySourceName+","+varName+")");

					//and copy the cloneable into the cloneable proxy
					methodBody.append("(("+ICloneableProxy.class.getCanonicalName()+")"+varName+").copy("+copySourceName+","+mapName+")");

					methodBody.append("}");
					methodBody.append("}");

					return varName;
				} else {
					return copySourceName;
				}
			}
		}
	}


	private String getVariableName() {
		return "var"+varCount++;
	}

	private String getCollectionElementName() {
		return "elt"+loopCount++;
	}

	private boolean ignoreField(JField field) {
		return field.isFinal();
	}



	private void generateCloneableFieldsInstantiationMethod() {
		Method method = new Method(Visibility.PUBLIC,ICloneableProxy.class,getCloneableProxyMethod);
		String param = "cloneable";
		method.addParameter(method.new Parameter(ICloneable.class, param));

		method.getBody().append("if("+param+"==null) { return null; }");

		boolean firstField = true;
		for(CloneableField f : cloneableFields.cloneableFields()) {
			if(f.fieldClass().isInterface() || Modifier.isAbstract(f.fieldClass().getModifiers())) {
				if(firstField) {
					firstField = false;
					method.getBody().append("if("+param+" instanceof "+f.fieldClass().getCanonicalName()+") {");
				}
				else {
					method.getBody().append("else if("+param+" instanceof "+f.fieldClass().getCanonicalName()+") {");
				}

				boolean firstPossibleImplem = true;
				for(Class<? extends ICloneable> possibleImplem : f.possibleImplemClasses()) {
					if(firstPossibleImplem) {
						firstPossibleImplem = false;
						method.getBody().append("if("+param+" instanceof "+possibleImplem.getCanonicalName()+") {");
					}
					else {
						method.getBody().append("else if("+param+" instanceof "+possibleImplem.getCanonicalName()+") {");
					}
					method.getBody().append("return ("+ICloneableProxy.class.getCanonicalName()+") GWT.create("+possibleImplem.getCanonicalName()+".class)");
					method.getBody().append("}");
				}

				method.getBody().append("}");
			}
			else {
				//TODO warning annotation ignored because invalid
			}
		}

		method.getBody().append("throw new java.lang.IllegalStateException("+param+".getClass().toString())");

		composerHelper.addMethod(method);
	}



	/**
	 * Checks that all the fields that implement Icloneable and whose type are interfaces
	 * have corresponding data in the annotation 
	 * @throws UnableToCompleteException 
	 */
	private void validateAnnotation() throws IncompleteAnnotation {
		Set<Class<? extends ICloneable>> s = new HashSet<Class<? extends ICloneable>>();
		if(cloneableFields!=null) {
			for(CloneableField f : cloneableFields.cloneableFields()) {
				s.add(f.fieldClass());
			}
		}

		for(JField field : helper.getAllFields()) {
			if(!(field instanceof ICloneable)) continue;
			if(ignoreField(field)) continue;

			Class<?> fieldClass = JTypeHelper.getClass(field.getType());
			if(fieldClass.isInterface() || Modifier.isAbstract(fieldClass.getModifiers())) {
				boolean foundSuperInterfaceInAnnotation = false;
				for(Class<? extends ICloneable> c : s) {
					if(c.isAssignableFrom(fieldClass)) {
						foundSuperInterfaceInAnnotation = true;
						break;
					}
				}
				if(!foundSuperInterfaceInAnnotation) {
					throw new IncompleteAnnotation(field);
				}
			}
		}
	}



	public void setType(JClassType type) {
		this.type = type;
	}


	private class IncompleteAnnotation extends Exception {
		private static final long serialVersionUID = -2765594476523830023L;

		private IncompleteAnnotation(JField field) {
			super("The cloneable field "+field.getName()+" is declared as an interface and has no implementation" +
					" defined in a @CloneableField annotation");
		}
	}

}