/**
 * model-manager - Handles models on client side for stuffs like undo/redo, methods observers, uibinding ... - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Set;

import com.google.gwt.core.ext.typeinfo.JClassType;
import com.google.gwt.core.ext.typeinfo.JType;


public class Method{

	private String name;
	private Visibility visibility;
	private Body body;
	private Class<?> returnType;
	private HashSet<Parameter> parameters;
	private HashSet<Annotation> annotations;
	private LinkedHashSet<Class<?>> returnTypeParams;
	private JType jtype;

	private boolean hasBody = true;

	public Method(Visibility visibility,String name) {
		this.parameters 	= new LinkedHashSet<Parameter>();
		this.annotations 	= new HashSet<Annotation>();
		this.returnTypeParams = new LinkedHashSet<Class<?>>();
		this.visibility 	= visibility;
		this.name 			= name;
		this.body 			= new Body();
	}

	public void setJtype(JType jtype) {
		this.jtype = jtype;
	}

	public Method(Visibility visibility,Class<?> returnType,String name) {
		this(visibility,name);
		this.returnType = returnType;
	}

	public void addReturnTypeParameter(Class<?> param){
		this.returnTypeParams.add(param);
	}

	public void setHasBody(boolean hasBody) {
		this.hasBody = hasBody;
	}

	public void addAnnotation(Annotation annotation){
		this.annotations.add(annotation);
	}

	public void addParameter(Parameter parameter){
		this.parameters.add(parameter);
	}

	public void setReturnType(Class<?> returnType) {
		this.returnType = returnType;
	}

	public Class<?> getReturnType() {
		return returnType;
	}


	public void setName(String name) {
		this.name = name;
	}


	public Body getBody() {
		return body;
	}

	public void setBody(Body body) {
		this.body = body;
	}

	public void setVisibility(Visibility visibility) {
		this.visibility = visibility;
	}

	public String getName() {
		return name;
	}

	public Visibility getVisibility() {
		return visibility;
	}

	public void isOverride(boolean override){

		Annotation overrideAnnotation = new Annotation("Override");

		if (override==true){

			this.addAnnotation(overrideAnnotation);


		}else{

			this.annotations.remove(overrideAnnotation);

		}

	}

	public String call(String ...params){
		StringBuilder sb = new StringBuilder();
		sb.append(this.name);
		sb.append("(");

		int max = params.length;
		int cur = 1;

		for(String param:params){
			sb.append(param);
			if (cur<max) sb.append(",");
			cur++;
		}

		sb.append(")");
		return sb.toString();
	}
	
	
	public String call() {
		String[] params = new String[parameters.size()];
		int i = 0;
		for(Parameter p : parameters) {
			params[i] = p.getName();
		}
		return call(params);
	}

	
	public Class<?>[] getParamTypes(){
		ArrayList<Class<?>> types = new ArrayList<Class<?>>();

		for(Parameter p : parameters){
			types.add(p.getType());
		}

		return types.toArray(new Class<?>[types.size()]);
	}

	
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();

		sb.append(getAnnotations());

		sb.append("\n"+this.visibility+" ");

		if (returnType!=null && jtype==null){
			sb.append(returnType.getCanonicalName()+getReturnTypeParameters()+" ");
		}else if (jtype!=null) {
			sb.append(jtype.getQualifiedSourceName()+getReturnTypeParameters()+" ");
		}

		sb.append(name);

		sb.append("("+getParametersAsString()+")");

		if (hasBody){
			sb.append("{\n");
			sb.append(body+"\n");
			sb.append("}");
		}else{
			sb.append(";\n");
		}

		return sb.toString();
	}

	
	private String getParametersAsString() {
		if (parameters.size()==0) return "";
		return parameters.toString().replace("[", "").replace("]", "");
	}
	
	
	public Set<Parameter> getParameters() {
		return parameters;
	}
	
	
	private String getReturnTypeParameters() {
		if (returnTypeParams.size()==0) return "";

		StringBuilder result = new StringBuilder();
		result.append("<");

		Iterator<Class<?>> typeIt = returnTypeParams.iterator();

		int max = returnTypeParams.size();
		int cur = 1;

		while(typeIt.hasNext()){
			Class<?> c = typeIt.next();
			result.append(c.getCanonicalName());
			if (cur<max){
				result.append(",");
			}
			cur++;
		}

		result.append(">");
		return result.toString();
	}


	private String getAnnotations(){
		if (annotations.size()==0) return "";

		StringBuilder stringBuilder = new StringBuilder();

		int max = annotations.size();
		int cur = 1;
		for(Annotation annotation:annotations){
			stringBuilder.append(annotation.toString());
			if (cur<max) stringBuilder.append("\n");
			cur++;
		}

		return stringBuilder.toString();
	}


	
	public class Annotation {

		private String name;

		public Annotation(String name) {
			this.name = name;
		}

		@Override
		public String toString() {
			return "@"+name;
		}

		@Override
		public boolean equals(Object obj) {

			Annotation toCompare = (Annotation) obj;

			return toCompare.name.equals(this.name);
		}

		@Override
		public int hashCode() {
			return name.hashCode();
		}

	}


	public class Parameter {

		private String typeCanonicalName;
		private Class<?> type;
		private String name;
		private JClassType[] typeParams;
		private JType jtype;
		private boolean isArray;

		public Parameter(Class<?> type,String name) {
			this.type = type;
			this.name = name;
		}

		public Parameter(String typeCanonicalName,String name) {
			this.typeCanonicalName = typeCanonicalName;
			this.name = name;
		}

		public void setArray(boolean isArray) {
			this.isArray = isArray;
		}

		public boolean isArray() {
			return isArray;
		}

		public Parameter(Class<?> type,String name,JClassType...typeParams){
			this(type,name);
			this.typeParams = typeParams;
		}

		public Parameter(JType type,String name){
			this.jtype = type;
			this.name  = name;
		}

		public Parameter(JType type,String name,JClassType...typeParams){
			this(type,name);
			this.typeParams = typeParams;
		}

		public Class<?> getType() {
			return type;
		}

		public String getName() {
			return name;
		}

		@Override
		public String toString() {
			StringBuilder sb = new StringBuilder();
			if (typeParams!=null){
				sb.append("<");
				int max = typeParams.length;
				int cur = 1;
				for(JClassType type:typeParams){
					sb.append(type.getQualifiedSourceName());
					if (cur<max) sb.append(",");
				}
				sb.append(">");
			}

			if (isArray){
				sb.append("...");
			}

			if (jtype==null && type!=null && typeCanonicalName==null) {
				return type.getCanonicalName()+sb.toString()+" "+name;
			}
			else if (jtype!=null && type==null && typeCanonicalName==null) {
				return jtype.getQualifiedSourceName() + sb.toString()+" "+name;
			}
			else if (jtype==null && type==null && typeCanonicalName!=null) {
				return typeCanonicalName + sb.toString()+" "+name;
			}
			return null;

		}

	}

}