/**
 * governance - A Web based designer for BPMN 2.0 standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.governance.client.ui;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.ebmwebsourcing.geasywebeditor.client.file.repository.FileRepositoryDescription;
import com.ebmwebsourcing.geasywebeditor.client.file.repository.service.FileRepositoryService;
import com.ebmwebsourcing.geasywebeditor.client.file.repository.service.FileRepositoryServiceAsync;
import com.ebmwebsourcing.geasywebeditor.client.file.repository.ui.FileRepositoryInquiryWidget;
import com.ebmwebsourcing.geasywebeditor.client.file.repository.ui.SearchForm;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstance;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstanceFormat;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstanceMetaData;
import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Widget;
import com.gwtext.client.core.EventObject;
import com.gwtext.client.data.FieldDef;
import com.gwtext.client.data.Record;
import com.gwtext.client.data.RecordDef;
import com.gwtext.client.data.Store;
import com.gwtext.client.data.StringFieldDef;
import com.gwtext.client.widgets.Button;
import com.gwtext.client.widgets.MessageBox;
import com.gwtext.client.widgets.Panel;
import com.gwtext.client.widgets.event.ButtonListenerAdapter;
import com.gwtext.client.widgets.form.Checkbox;
import com.gwtext.client.widgets.form.FormPanel;
import com.gwtext.client.widgets.form.Label;
import com.gwtext.client.widgets.form.MultiFieldPanel;
import com.gwtext.client.widgets.form.TextField;
import com.gwtext.client.widgets.form.event.CheckboxListenerAdapter;
import com.gwtext.client.widgets.grid.BaseColumnConfig;
import com.gwtext.client.widgets.grid.CheckboxColumnConfig;
import com.gwtext.client.widgets.grid.CheckboxSelectionModel;
import com.gwtext.client.widgets.grid.ColumnConfig;
import com.gwtext.client.widgets.grid.ColumnModel;
import com.gwtext.client.widgets.grid.GridPanel;
import com.gwtext.client.widgets.grid.GridView;
import com.gwtext.client.widgets.layout.AnchorLayout;
import com.gwtext.client.widgets.layout.AnchorLayoutData;
import com.gwtext.client.widgets.layout.ColumnLayoutData;
import com.gwtext.client.widgets.layout.HorizontalLayout;
import com.gwtext.client.widgets.layout.RowLayoutData;

public class FileRepositoryInquiryPanel extends FileRepositoryInquiryWidget {
    
    private IProjectInstance instance;
    private IProjectInstanceFormat format;
    private List<IProjectInstanceMetaData> projectMetadata;
    
    private FileRepositoryServiceAsync fileRepoService = (FileRepositoryServiceAsync) GWT.create(FileRepositoryService.class);
    private FileRepositoryDescription chosenFileRepoDesc;
    private List<FileRepositoryDescription> repoDescs;
    private Panel fileRepoPanel;
    private CheckboxSelectionModel cbSelectionModel;

    public FileRepositoryInquiryPanel(IProjectInstance instance, IProjectInstanceFormat format, List<FileRepositoryDescription> repoDescs) {
        super();
        this.instance = instance;
        this.format = format;
        this.repoDescs = repoDescs;
        
        this.projectMetadata = new ArrayList<IProjectInstanceMetaData>();
        
        this.fileRepoPanel = new Panel();
        fileRepoPanel.setPaddings(10);
        //fileRepoPanel.setLayout(new FitLayout());
        fileRepoPanel.setLayout(new AnchorLayout());
    }

    
    @Override
    protected void displayRepositoryFinder(final FindRepositoryCallback callBack) {
        Panel mainPanel = new Panel();
        mainPanel.setBorder(false);
        mainPanel.setLayout(new AnchorLayout());
        
        final FormPanel formPanel = new FormPanel();
        formPanel.setLabelWidth(150);  
        formPanel.setBorder(false);
        formPanel.setPaddings(5);
        
        MultiFieldPanel mfp = new MultiFieldPanel();
        mfp.setBorder(false);
        formPanel.add(mfp);
        
        final Label exampleLabel = new Label();
        boolean first = true;
        int i = 1;
        final Map<Checkbox,FileRepositoryDescription> checkBoxes = new HashMap<Checkbox, FileRepositoryDescription>();
        for(FileRepositoryDescription desc : repoDescs){
            Checkbox r = new Checkbox(desc.getDisplayName());
            if(first){
                 r.setLabel("Kind of Repository:");
                 r.setChecked(true);
                 first = false;
                 mfp.addToRow(r, 250);
                 chosenFileRepoDesc = desc;
                 
                 exampleLabel.setHtml("eg : "+(chosenFileRepoDesc.getAddressExample()==null ? "" : chosenFileRepoDesc.getAddressExample()) );
            }
            else{
                mfp.addToRow(r,new ColumnLayoutData(i));
                i++;
            }
            checkBoxes.put(r,desc);
            
            r.addListener(new CheckboxListenerAdapter() {
                @Override
                public void onCheck(Checkbox field, boolean checked) {
                    if(checked){
                        chosenFileRepoDesc = checkBoxes.get(field);
                        for(Checkbox cb : checkBoxes.keySet()){
                            if(cb.getValue() && !cb.equals(field)){
                                cb.setChecked(false);
                            }
                        }
                        
                        exampleLabel.setHtml("eg : "+(chosenFileRepoDesc.getAddressExample()==null ? "" : chosenFileRepoDesc.getAddressExample()) );
                        formPanel.doLayout();
                    }
                }
            });
        }
        
        final TextField urlField = new TextField("Address of a file repository", "url", 250);  
        urlField.setBlankText("This field is required.");
        urlField.setAllowBlank(false); 
        formPanel.add(urlField);
        
        formPanel.add(exampleLabel);

        
        Button buttonTry = new Button("Try this URL", new ButtonListenerAdapter(){
            public void onClick(Button button, EventObject e) {
                final String url = urlField.getText();  
                
                showWaitingBar();
                fileRepoService.testURL(url, chosenFileRepoDesc.getFileRepositoryInquiryService(), new AsyncCallback<Boolean>(){
                    @Override
                    public void onSuccess(Boolean arg0){
                        if(arg0){
                            removeWaitingBar();
                            callBack.onRepositoryFound(chosenFileRepoDesc.getSearchForm());
                        }
                        else{
                            removeWaitingBar();
                            callBack.onError(url+" is not a valid URL for a "+chosenFileRepoDesc.getDisplayName()+" server.");
                        }
                    }
                    @Override
                    public void onFailure(Throwable arg0) {
                        removeWaitingBar();
                        callBack.onError(arg0);
                    }
                });
            }
        });
        Panel buttonPanel = new Panel();
        buttonPanel.setBorder(false);
        buttonPanel.add(buttonTry);
        buttonPanel.setPaddings(5);
        
        mainPanel.add(formPanel, new AnchorLayoutData("100% 80%"));
        mainPanel.add(buttonPanel, new AnchorLayoutData("100% 20%"));
        
        fileRepoPanel.clear();
        fileRepoPanel.add(mainPanel, new AnchorLayoutData("100% 100%"));
        fileRepoPanel.doLayout();
    }

    
    @Override
    protected void displaySearchForm(final SearchForm form, final SearchCallback callback) {
        Panel p = new Panel();
        p.setBorder(false);
        p.add(form.asWidget());
        Panel hp = new Panel();
        hp.setBorder(false);
        hp.setLayout(new HorizontalLayout(15));

        hp.add(new Button("Search in repository", new ButtonListenerAdapter(){
            @Override
            public void onClick(Button button, EventObject e) {
                String msg = form.isInformationReady();
                if(msg!=null){
                    MessageBox.alert(msg);
                    return;
                }
                
                showWaitingBar();
                fileRepoService.searchFile(form.getInformation(), new AsyncCallback<Map<String,String>>(){
                    @Override
                    public void onSuccess(Map<String,String> arg0){
                        if(arg0==null){
                            removeWaitingBar();
                            callback.onError("An error occured during the query.");
                        }
                        else if(arg0.isEmpty()){
                            removeWaitingBar();
                            callback.onError("No result was found for the information you provided.");
                        }
                        else{
                            removeWaitingBar();
                            callback.onSearchPerformed(arg0);
                        }
                    }
                    @Override
                    public void onFailure(Throwable arg0) {
                        removeWaitingBar();
                        callback.onError("An error occured during the query.<br>"+arg0.getMessage());
                    }
                });
            }
        }));

        hp.add(new Button("Change server", new ButtonListenerAdapter(){
            @Override
            public void onClick(Button button, EventObject e) {
                callback.changeRepository();
            }
        }));
        p.add(hp);
        
        fileRepoPanel.clear();
        fileRepoPanel.add(p);
        fileRepoPanel.doLayout();
    }

    
    @Override
    protected void displaySearchResult(final Map<String, String> searchResults,
            final SearchFinishedCallback callback) {
        RecordDef recordDef = new RecordDef(  
                new FieldDef[]{  
                        new StringFieldDef("filename"),  
                }
        );
        
        cbSelectionModel = new CheckboxSelectionModel();
        
        BaseColumnConfig[] columns = new BaseColumnConfig[]{
                new CheckboxColumnConfig(cbSelectionModel), 
                new ColumnConfig("File", "filename"),  
        };
        
        Store store = new Store(recordDef);
        for(String id : searchResults.keySet()){
            Record r = recordDef.createRecord(new Object[]{id});
            store.add(r);
        }

        GridPanel grid = new GridPanel(store, new ColumnModel(columns));
        grid.setStripeRows(true); 
        grid.setSelectionModel(cbSelectionModel);  
        GridView view = new GridView();
        view.setForceFit(true);
        grid.setView(view);
        
        Panel gridPanel_ = new Panel();
        grid.setAutoScroll(true);
        gridPanel_.setBorder(false);
        gridPanel_.add(grid);
        
        Panel panel = new Panel();
        panel.setBorder(false);
        panel.setLayout(new HorizontalLayout(15));
        Button button = new Button("Get selected "+chosenFileRepoDesc.getFileFormatDisplayName()+" files", new ButtonListenerAdapter(){
            @Override
            public void onClick(Button button, EventObject e) {
                showWaitingBar();
                Record[] records = cbSelectionModel.getSelections();
                
                Map<String, String> files = new HashMap<String, String>();
                for(int i=0; i<records.length; i++){
                    final String fileName = records[i].getAsString("filename");
                    files.put(fileName, searchResults.get(fileName));
                }
                
                fileRepoService.attachFilesFromRepoDescription(files, instance, format, new AsyncCallback<List<IProjectInstanceMetaData>>() {
                    @Override
                    public void onSuccess(List<IProjectInstanceMetaData> arg0) {
                        removeWaitingBar();
                        projectMetadata.addAll(arg0);
                        MessageBox.alert("Success","The WSDL file has been uploaded to the server and is now available for being imported in your BPMN model.");
                        callback.onSearchFinished();
                    }
                    
                    @Override
                    public void onFailure(Throwable arg0) {
                        removeWaitingBar();
                        callback.onError("An error occured during the recovery of the file.<br>"+arg0.getMessage());
                    }
                });
            }
        });
        panel.add(button);
        
        panel.add(new Button("Refine research",new ButtonListenerAdapter(){
            @Override
            public void onClick(Button button, EventObject e) {
                callback.onSearchFinished();
            }
        }));
        
        fileRepoPanel.clear();
        fileRepoPanel.add(gridPanel_);
        fileRepoPanel.add(panel, new RowLayoutData(30));
        fileRepoPanel.doLayout();
    }

    @Override
    protected void displayError(Throwable t) {
        MessageBox.alert("RPC failure", t.getMessage());
    }
    
    @Override
    protected void displayError(String msg) {
        MessageBox.alert(msg);
    }

    @Override
    public Widget asWidget() {
        return fileRepoPanel;
    }

    public List<IProjectInstanceMetaData> getProjectMetadata() {
        return projectMetadata;
    }
    
    private void showWaitingBar(){
        fileRepoPanel.getEl().mask("Please wait...", true);
    }
    
    private void removeWaitingBar(){
        fileRepoPanel.getEl().unmask();
    }
    
}
