/**
 * service - BPMN Editor service - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.server.service.bpmndi.serverToClient;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;

import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNDiagram;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNLabel;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNShape;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.Bounds;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.DiagramElement;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.Font;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.diagramcommon.layout.IBounds;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.diagramcommon.layout.IPoint;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.diagramcommon.layout.Point;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.extension.client.AttributeExtension;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagramElement;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IStyle;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.foundation.IBaseElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.DefinitionsHelper;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.DefinitionsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNDiagram;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNEdge;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNLabel;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNLabelStyle;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNPlane;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNShape;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.MessageVisibleKind;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.ParticipantBandKind;
import com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNEdge;
import com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNLabelStyle;
import com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNPlane;
import com.ebmwebsourcing.petalsbpm.server.service.extension.ExtensionBindingManager;


public class BPMNDiagramParser {

	private static IBPMNDiagram diagram;
	private static IBPMNPlane plane;
	private static DefinitionsBean defs;
	private static Map<BPMNEdge, QName> edgeSources;
	private static Map<BPMNEdge, QName> edgeTargets;
	private static Map<com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNLabel, QName> labelStyles;
	private static Map<com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNShape, QName> choreoShapes;
	private static ExtensionBindingManager bindingManager;
	
	
	public static IBPMNDiagram parseBPMNDiagram(BPMNDiagram bpmn20diagram, DefinitionsBean definitions){
		edgeSources = new HashMap<BPMNEdge, QName>();
		edgeTargets = new HashMap<BPMNEdge, QName>();
		labelStyles = new HashMap<com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNLabel, QName>();
		choreoShapes = new HashMap<com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNShape, QName>();
		bindingManager = new ExtensionBindingManager();
		
		defs = definitions;
		com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNDiagram result = new com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNDiagram();
		diagram = result;
		
		result.setId(bpmn20diagram.getId());
		result.setDocument(bpmn20diagram.getDocumentation());
		result.setName(bpmn20diagram.getName());
		result.setResolution(new Float(bpmn20diagram.getResolution()));
		
		result.setRootElement(getPlane(bpmn20diagram.getBPMNPlane()));
		plane = result.getRootElement();

		HashSet<IStyle> styles = new HashSet<IStyle>();
		if(bpmn20diagram.getBPMNLabelStyle()!=null){
			for(com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNLabelStyle style : bpmn20diagram.getBPMNLabelStyle()){
				styles.add(getStyle(style));
			}
		}
		result.setStyles(styles);
		
		
		assignEdgeExtremities();
		assignLabelStyles();
		assignChoreoShapes();
		
		return result;
	}
	

	private static IBPMNLabelStyle getStyle(com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNLabelStyle style){
		BPMNLabelStyle result = new BPMNLabelStyle();
		
		Font font = style.getFont();
		
		com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.diagramcommon.layout.Font f = new com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.diagramcommon.layout.Font();
		f.setBold(font.isIsBold());
		f.setItalic(font.isIsItalic());
		f.setStrikeThrough(font.isIsStrikeThrough());
		f.setUnderline(font.isIsUnderline());
		f.setName(font.getName());
		f.setSize(font.getSize());
		
		return result;
	}
	
	
	private static IBPMNPlane getPlane(com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNPlane plane){
		BPMNPlane result = new BPMNPlane();
		result.setModelElement(getModelElement(plane.getBpmnElement()));
		adaptExtensions(plane, result);
		
		if(plane.getDiagramElement()!=null){
			for(DiagramElement de : plane.getDiagramElement()){
				result.addDiagramElement(getDiagramElement(de));
			}
			
			buildHierarchy(result.getOwnedElements());
		}
		
		return result;
	}
	
	
	private static IDiagramElement getDiagramElement(DiagramElement de) {
		if(de instanceof BPMNShape){
			return getShape((BPMNShape)de);
		}
		else if(de instanceof com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNEdge){
			return getEdge((com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNEdge)de);
		}
		return null;
	}



	private static IBPMNShape getShape(BPMNShape shape) {
		com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNShape result = new com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNShape();
		
		result.setExpanded(shape.isIsExpanded());
		result.setHorizontal(shape.isIsHorizontal());
		result.setMarkerVisible(shape.isIsMarkerVisible());
		result.setMessageVisible(shape.isIsMessageVisible());
		result.setModelElement(getModelElement(shape.getBpmnElement()));
		
		adaptExtensions(shape, result);
		
		Bounds bounds = shape.getBounds();
		if(bounds!=null){//FIXME used only because of TIBCO xpdl import (tibco pools and lanes don't have coordinates)
			result.setHeight(bounds.getHeight());
			result.setWidth(bounds.getWidth());
			result.setX(bounds.getX());
			result.setY(bounds.getY());
		}
		
		if(shape.getParticipantBandKind()!=null){
			switch(shape.getParticipantBandKind()){
			case bottom_initiating:
				result.setParticipantBandKind(ParticipantBandKind.bottom_initiating);
				break;
			case bottom_non_initiating:
				result.setParticipantBandKind(ParticipantBandKind.bottom_non_initiating);
				break;
			case middle_initiating:
				result.setParticipantBandKind(ParticipantBandKind.middle_initiating);
				break;
			case middle_non_initiating:
				result.setParticipantBandKind(ParticipantBandKind.middle_non_initiating);
				break;
			case top_initiating:
				result.setParticipantBandKind(ParticipantBandKind.top_initiating);
				break;
			case top_non_initiating:
				result.setParticipantBandKind(ParticipantBandKind.top_non_initiating);
				break;
			}
		}
		
		if(shape.getBPMNLabel()!=null){
			result.getOwnedLabels().add(getLabel(shape.getBPMNLabel()));
		}
		
		if(shape.getChoreographyActivityShape()!=null){
			choreoShapes.put(result, shape.getChoreographyActivityShape());
		}
		
		return result;
	}
	
	
	private static void assignChoreoShapes() {
		for(com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNShape shape : choreoShapes.keySet()){
			String choreoShapeId = getModelElement(choreoShapes.get(shape)).getId();
			
			for(IDiagramElement de  : plane.getOwnedElements()){
				if(de instanceof com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNShape){
					String modelId = ((com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNShape)de).getModelElement().getId();

					if(modelId.equals(choreoShapeId)){
						shape.setChoreographyActivityShape((com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNShape)de);
					}
				}
			}
		}
	}		
	
	
	private static IBPMNEdge getEdge(com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNEdge edge) {
		BPMNEdge result = new BPMNEdge();
		
		result.setModelElement(getModelElement(edge.getBpmnElement()));
		
		adaptExtensions(edge, result);
		
		if(edge.getMessageVisibleKind()!=null){
			switch(edge.getMessageVisibleKind()){
			case initiating:
				result.setMessageVisibleKind(MessageVisibleKind.initiating);
				break;
			case non_initiating:
				result.setMessageVisibleKind(MessageVisibleKind.non_initiating);
				break;
			}
		}
		
		if(edge.getBPMNLabel()!=null){
			result.getOwnedLabels().add(getLabel(edge.getBPMNLabel()));
		}
		
		for(int i=0; i<edge.getWaypoint().length; i++){
			result.getWayPoints().add(getPoint(edge.getWaypoint()[i]));
		}
		
		edgeSources.put(result, edge.getSourceElement());
		edgeTargets.put(result, edge.getTargetElement());
		
		return result;
	}
	
	
	private static void assignEdgeExtremities() {
		for(BPMNEdge edge : edgeSources.keySet()){
			String sourceId = getModelElement(edgeSources.get(edge)).getId();
			String targetId = getModelElement(edgeTargets.get(edge)).getId();
			
			boolean sourceFound = false;
			boolean targetFound = false;
			
			for(IDiagramElement de  : plane.getOwnedElements()){
				if(de instanceof com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNShape){
					String modelId = ((com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNShape)de).getModelElement().getId();

					if(modelId.equals(sourceId)){
						edge.setSource(de);
						sourceFound = true;
					}
					else if(modelId.equals(targetId)){
						edge.setTarget(de);
						targetFound = true;
					}
					
					if(targetFound && sourceFound){
						break;
					}
					
				}
			}
		}
	}
	
	
	private static IPoint getPoint(com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.Point waypoint){
		return new Point(waypoint.getX(), waypoint.getY());
	}
	

	
	private static IBPMNLabel getLabel(BPMNLabel label){
		com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNLabel result = new com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNLabel();
		
		adaptExtensions(label, result);
		
		Bounds bounds = label.getBounds();
		if(bounds!=null){
			result.setHeight(bounds.getHeight());
			result.setWidth(bounds.getWidth());
			result.setX(bounds.getX());
			result.setY(bounds.getY());
		}
		
		labelStyles.put(result, label.getLabelStyle());
		
		return result;
	}
	
	
	private static void assignLabelStyles() {
		for(IBPMNLabel label : labelStyles.keySet()){
			QName labelId = labelStyles.get(label);
			
			for(IStyle style : diagram.getStyles()){
				if(style instanceof BPMNLabelStyle){
					if(style.getId().equals(labelId)){
						((com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNLabel)label).setBPMNLabelStyle((BPMNLabelStyle)style);
					}
				}
			}
		}
	}

	
	
	private static void adaptExtensions(DiagramElement elt, IDiagramElement bean){
		if(elt.hasExtension()){
			for(XmlObject obj : elt.getExtension().getAnyXmlObjects()){
				bean.addObjectExtension(bindingManager.serverToClient(obj));
			}
		}
		for(QName qn : elt.getOtherAttributes().keySet()){
			String value = elt.getOtherAttribute(qn);
			bean.addAttributeExtension(new AttributeExtension(qn.getNamespaceURI(), qn.getLocalPart(), value));
		}
	}

	
	
	private static IBaseElementBean getModelElement(QName qn){
		return DefinitionsHelper.getInstance().getElementById(defs, qn.getLocalPart(), qn.getNamespaceURI());
	}
	
	
	private static void buildHierarchy(LinkedHashSet<IDiagramElement> elements) {
	    if(elements==null) return;
	    
	    List<IBPMNShape> shapes = new ArrayList<IBPMNShape>();
	    for(IDiagramElement elt :elements) {
	        if(elt instanceof IBPMNShape) {
	            shapes.add((IBPMNShape) elt);
	        }
	    }

	    for(int i=shapes.size()-1; i>=0; i--) {
	        IBPMNShape shapei = shapes.get(i);
	        int j=i-1;
	        while(j>=0 && !geometricallyContains(shapes.get(j), shapei)) {
	            j--;
	        }
	        if(j>=0) {
	            shapes.get(j).addDiagramElement(shapei);
	            IBounds containerBounds = shapes.get(j).getBounds();
	            IBounds shapeBounds = shapei.getBounds();
	            shapeBounds.setX(shapeBounds.getX()-containerBounds.getX());
                shapeBounds.setY(shapeBounds.getY()-containerBounds.getY());
	        }
	    }
	}
	
	private static boolean geometricallyContains(IBPMNShape container, IBPMNShape shape) {
	    IBounds b1 = container.getBounds();
	    IBounds b2 = shape.getBounds();
	    
	    double x1 = b1.getX();
	    double y1 = b1.getY();
	    double w1 = b1.getWidth();
	    double h1 = b1.getHeight();
	 
	    double x2 = b2.getX();
        double y2 = b2.getY();
        double w2 = b2.getWidth();
        double h2 = b2.getHeight();
        
        //assert that the top left corner of the shape is in the container
        boolean result = x1<=x2 && x2<=(x1+w1) && y1<=y2 && y2<=(y1+h1);
        //assert that the bottom right corner of he shape also belongs to the container
        result = result && (x2+w2)<=(x1+w1) && (y2+h2)<=(y1+h1);
        
        return result;
	}
	
}
