/**
 * easySchema - SOA Tools Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $id.java
 * -------------------------------------------------------------------------
 */
package com.ebmwebsourcing.commons.schema.util;

import java.io.StringWriter;
import java.util.Properties;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


/**
 * @author Nicolas Salatge - eBM WebSourcing
 */
public class XMLUtil {
	/**
	 * <p>
	 * This transformer has the following features:
	 * <ul>
	 * <li>omit the XML declaration,</li>
	 * <li>default encoding</li>
	 * </ul>
	 * </p>
	 */
	private final static ThreadLocal<Transformer> transformerWithoutXmlDeclarationThreadLocal = new ThreadLocal<Transformer>() {

		@Override
		protected Transformer initialValue() {
			try {

				final Transformer transformer = TransformerFactory.newInstance().newTransformer();
				Properties props = new Properties();
				props.put(OutputKeys.OMIT_XML_DECLARATION, "yes");
				props.put(OutputKeys.METHOD, "xml");
				transformer.setOutputProperties(props);
				return transformer;

			} catch (TransformerConfigurationException e) {
				throw new RuntimeException("Failed to create Transformer",
						e);
			}
		}
	};

    /**
     * <p>
     * This transformer has the following features:
     * <ul>
     * <li>doesn't omit the XML declaration,</li>
     * <li>default encoding</li>
     * </ul>
     * </p>
     */
    private final static ThreadLocal<Transformer> defaultTransformerThreadLocal = new ThreadLocal<Transformer>() {

        @Override
        protected Transformer initialValue() {
            try {

                final Transformer transformer = TransformerFactory.newInstance().newTransformer();
                Properties props = new Properties();
                props.put(OutputKeys.OMIT_XML_DECLARATION, "no");
                props.put(OutputKeys.METHOD, "xml");
                transformer.setOutputProperties(props);
                return transformer;

            } catch (TransformerConfigurationException e) {
                throw new RuntimeException("Failed to create Transformer",
                        e);
            }
        }
    };

	/**
	 * Create a String result from a DOM document
	 *
	 * @param document
	 *            the DOM Document. Must not be null
	 * @return a String representation of the DOM Document
	 * @throws TransformerException
	 */
	public static String createStringFromDOMDocument(final Node document)
	throws TransformerException {
		return createStringFromDOMNode(document, false);
	}

	/**
	 * Create a String result from a DOM Node
	 *
	 * @param node
	 *            the DOM Node. Must not be null
	 * @return a String representation of the DOM Document
	 * @throws TransformerException
	 */
	public static String createStringFromDOMNode(final Node node) throws TransformerException {
		return createStringFromDOMNode(node, true);
	}

	/**
	 * Create a {@link String} from a Node list
	 *
	 * @param list
	 * @return
	 * @throws TransformerException
	 */
	public static String createStringFromDOMNodeList(final NodeList list)
	throws TransformerException {
		StringBuffer sb = new StringBuffer("");
		if (list != null) {
			for (int i = 0; i < list.getLength(); i++) {
				sb.append(createStringFromDOMNode(list.item(i)));
			}
		}
		return sb.toString();
	}

	/**
	 * Create a String result from a DOM Node
	 *
	 * @param node
	 *            the DOM Node. Must not be null
	 * @parma moitDeclaration If <code>true</code> no XML declaration will be
	 *        added.
	 * @return a String representation of the DOM Document
	 * @throws TransformerException
	 */
	public static String createStringFromDOMNode(final Node node, final boolean omitDeclaration)
	throws TransformerException {
		node.normalize();
		final Source source = new DOMSource(node);
		final StringWriter out = new StringWriter();
		final Result resultStream = new StreamResult(out);
		if (omitDeclaration) {
			XMLUtil.transformerWithoutXmlDeclarationThreadLocal.get()
			.transform(source, resultStream);
		}
		else {
			XMLUtil.defaultTransformerThreadLocal.get().transform(source,
					resultStream);
		}
		return out.toString();
	}

}
