/*******************************************************************************
 * Copyright (c) 2011 EBM Websourcing.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v2.1
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * 
 * Contributors:
 *     EBM Websourcing - initial API and implementation
 ******************************************************************************/
package com.ebmwebsourcing.easierbsm.sla.manager.impl.thread;

import java.net.URI;
import java.util.Calendar;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Logger;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.ebmwebsourcing.easierbsm.contant.EasierBSMFramework;
import com.ebmwebsourcing.easierbsm.sla.manager.api.AgreementManagerComponentBehaviour;
import com.ebmwebsourcing.easybox.api.XmlObjectFactory;
import com.ebmwebsourcing.easybox.api.XmlObjectReadException;
import com.ebmwebsourcing.easybox.api.XmlObjectWriteException;
import com.ebmwebsourcing.easycommons.research.util.SOAException;
import com.ebmwebsourcing.easycommons.research.util.dom.DOMUtil;
import com.ebmwebsourcing.easycommons.research.util.easybox.SOAUtil;
import com.ebmwebsourcing.easycommons.research.util.esb.ESBUtil;
import com.ebmwebsourcing.easycommons.research.util.esb.EndpointAddress;
import com.ebmwebsourcing.easycommons.research.util.jaxb.SOAJAXBContext;
import com.ebmwebsourcing.easycommons.soap.handler.SOAPSender;
import com.ebmwebsourcing.easycommons.xml.DocumentBuilders;
import com.ebmwebsourcing.easycommons.xml.XMLPrettyPrinter;
import com.ebmwebsourcing.easyesb.rawreport10.api.element.Report;
import com.ebmwebsourcing.easyesb.rawreport10.api.element.ReportList;
import com.ebmwebsourcing.escad10.api.element.AlertDefinition;
import com.ebmwebsourcing.escapnote10.api.element.AlertNoteDefinition;
import com.ebmwebsourcing.escapnote10.api.element.InitiatorIdentifier;
import com.ebmwebsourcing.escapnote10.api.element.ResponderIdentifier;
import com.ebmwebsourcing.esqml10.api.type.ConstraintType;
import com.ebmwebsourcing.wsagreement10.api.element.GuaranteeTerm;
import com.ebmwebsourcing.wsagreement10.api.element.ServiceLevelObjective;
import com.ebmwebsourcing.wsagreement10.api.type.AgreementType;
import com.ebmwebsourcing.wscap12.api.anonymoustype.Alert;

import easybox.org.ggf.schemas.graap._2007._03.ws_agreement.EJaxbAgreementContextType;
import easybox.org.oasis_open.docs.wsn.b_2.ObjectFactory;
import easybox.org.w3._2005._08.addressing.EJaxbEndpointReferenceType;

public class FrequencyViolationThread extends AbstractAlertManagerThread implements Runnable {

	static {
		try {
			SOAJAXBContext.getInstance().addOtherObjectFactory(ObjectFactory.class);
		} catch (SOAException e) {
			// do nothing
			e.printStackTrace();
		}
	}

	private final Logger log = Logger.getLogger(FrequencyViolationThread.class
			.getName());

	private DocumentBuilder db = DocumentBuilders.takeDocumentBuilder();

	private boolean stop;

	private List<Long> notifTimeStamps;

	private AlertDefinition alertDef;


	private ReportList reports;

	private AgreementType agreement;

	private ServiceLevelObjective slo;

	private final SOAPSender soapSender;

	private final AgreementManagerComponentBehaviour behaviour;

	private double freq = 0;

	private Object initiator;

	private Object responder;
	
	private static final long MINUTE_IN_SECONDS = 60000;

	//private final long TIMEOUT = 2000;

	public FrequencyViolationThread(AgreementManagerComponentBehaviour behaviour, ReportList reports, ServiceLevelObjective slo, AgreementType agreement)  {
		super(new SOAPSender(), behaviour, agreement);
		this.reports = reports;
		this.agreement = agreement;
		this.slo = slo;
		this.soapSender = new SOAPSender();
		this.behaviour = behaviour;
		this.stop = false;

		this.notifTimeStamps = new CopyOnWriteArrayList<Long>();

		initiator = this.agreement.getAgreementContext().getAgreementInitiator();
		responder = this.agreement.getAgreementContext().getAgreementResponder();


		try {
			freq = extractFrequencyConstraint(slo);
			initAlertDefinition();
		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		} catch (XmlObjectReadException e) {
			e.printStackTrace();
		}


	}

	private void initAlertDefinition() throws ParserConfigurationException, XmlObjectReadException {
		this.alertDef = this.globalAggreement.getAgreementContext().findExtendedElement(AlertDefinition.class);
		if(alertDef == null) {
			// FIXME: EASYBOX BUG
			Object obj = ((EJaxbAgreementContextType)this.globalAggreement.getAgreementContext().getModelObject()).getAny().iterator().next();
			if(obj != null && obj instanceof Element) {
				Element elmt = (Element)obj;
				Document doc = DOMUtil.getInstance().getDocumentBuilderFactory().newDocumentBuilder().newDocument();
				doc.appendChild(doc.importNode(elmt, true));
				alertDef = SOAUtil.getInstance().getReader(EasierBSMFramework.getInstance()).get().readDocument(doc, AlertDefinition.class);
			}
		}


	}

	private double extractFrequencyConstraint(ServiceLevelObjective slo) throws ParserConfigurationException, XmlObjectReadException {
		double res = -1;

		Object obj = slo.getKPITarget().getCustomServiceLevel();

		if(obj != null && obj instanceof Element) {
			Element elmt = (Element)obj;
			Document doc;

			doc = DOMUtil.getInstance().getDocumentBuilderFactory().newDocumentBuilder().newDocument();
			doc.appendChild(doc.importNode(elmt, true));

			ConstraintType target = SOAUtil.getInstance().getReader(EasierBSMFramework.getInstance()).get().readFragment(doc, ConstraintType.class);
			res = target.getValue().getValue();
			if(target.getValue().getUnit().getName().endsWith("second")){
				res *= 60;
			}//else assume per minute
		}

		return res;
	}

	public void addReport(Report report){
		this.reports.addReport(report);
	}

	public ReportList getReports(){
		return this.reports;
	}

	@Override
	public void run() {
		try {

			//while(!stop){
			//	Thread.sleep(3000);

			/* there MUST be at least 2 notifs to compute a frequence */
			if(this.notifTimeStamps.size()>1){

				/* last timestamp of the list */
				long last = this.notifTimeStamps.get(this.notifTimeStamps.size()-1);
				int cpt = 0;

				System.out.println("LIST SIZE : "+this.notifTimeStamps.size());

				/* we look for the first timestamp that is >= 1 minute from the last */
				for(int i=this.notifTimeStamps.size()-2 ; i>=0 ; i--){
					cpt++;
					
					long window = last - this.notifTimeStamps.get(i);
					
					
					if(window >= MINUTE_IN_SECONDS){

						/* check frequency violation TODO for now only ">" operator managed */
						if(cpt > this.freq){
							String msg = "!!! Too many notifications events, actual frequeny: "+cpt+" notifications/minute !!!";
							Alert a = createAlert(msg, reports, Level.SEVERE, initiator, responder, "Observed");
							System.out.println("SEND ALERT !!!!!!!!!!!");
							sendAlert(a, alertDef);

						}
						break;
					}else if(cpt> (this.freq*window/MINUTE_IN_SECONDS)){
						long actualFreq = cpt*MINUTE_IN_SECONDS/window;
						String msg = "!!! Too many notifications events, actual frequeny: "+actualFreq+" notifications/minute !!!";
						Alert a = createAlert(msg, reports, Level.WARNING, initiator, responder, "Potential");
						System.out.println("SEND ALERT (BEFORE ONE MINUTE) !!!!!!!!!!!");
						sendAlert(a, alertDef);
					}
				}
			}
			//}

		}catch (Exception e) {
			e.printStackTrace();
		}
	}




	
	private Alert createAlert(String msg, GuaranteeTerm g,
			ReportList reports, Level level, Object initiator, Object responder) {

		Alert alert = SOAUtil.getInstance().getXmlContext(EasierBSMFramework.getInstance()).getXmlObjectFactory().create(Alert.class);

		alert.setIdentifier(reports.getReports()[0].getExchangeId());
		alert.setMsgType("Alert");
		alert.setSent(Calendar.getInstance().getTime());

		/* Set Alert note with internal capnote spec. */

		XmlObjectFactory factory = SOAUtil.getInstance().getXmlContext(EasierBSMFramework.getInstance()).getXmlObjectFactory();
		AlertNoteDefinition note = factory.create(AlertNoteDefinition.class);
		note.setMessage(msg);

		if(initiator instanceof EJaxbEndpointReferenceType){
			try{
				EJaxbEndpointReferenceType initiatorEPR = (EJaxbEndpointReferenceType)initiator; 
				EndpointAddress ea = ESBUtil.analyzeURI(new URI(initiatorEPR.getAddress().getValue()));

				InitiatorIdentifier initiatorEndpoint = SOAUtil.getInstance().getXmlContext(EasierBSMFramework.getInstance()).getXmlObjectFactory().create(InitiatorIdentifier.class);
				initiatorEndpoint.setEndpointName(ea.getEndpointname());
				initiatorEndpoint.setNamespaceURI(ea.getNamespace());
				if(ea.getServicename() != null){
					initiatorEndpoint.setServiceName(ea.getServicename());
				}
				note.setInitiatorIdentifier(initiatorEndpoint);

			}catch(Exception e){
				log.severe("Failed to parse initiator URI ...");
			}
		}
		if(responder instanceof EJaxbEndpointReferenceType){
			try{
				EJaxbEndpointReferenceType responderEPR = (EJaxbEndpointReferenceType)responder; 
				EndpointAddress ea = ESBUtil.analyzeURI(new URI(responderEPR.getAddress().getValue()));

				ResponderIdentifier responderEndpoint = SOAUtil.getInstance().getXmlContext(EasierBSMFramework.getInstance()).getXmlObjectFactory().create(ResponderIdentifier.class);
				responderEndpoint.setEndpointName(ea.getEndpointname());
				responderEndpoint.setNamespaceURI(ea.getNamespace());
				responderEndpoint.setServiceName(ea.getServicename());

				note.setResponderIdentifier(responderEndpoint);

			}catch(Exception e){
				log.severe("Failed to parse responder URI ...");
			}
		}

		String noteAsString = "";//"<![CDATA[";
		Document noteAsDOM = db.newDocument();
		try {
			SOAUtil.getInstance().getWriter(EasierBSMFramework.getInstance()).get().writeDocument(note, noteAsDOM);
		} catch (XmlObjectWriteException e) {
			log.severe("Failed to write note as dom in alert.");
		}

		noteAsString += XMLPrettyPrinter.prettyPrint(noteAsDOM);
		//noteAsString += "]]>";
		alert.setNote(noteAsString);

		return alert;
	}

	public void launchNotify(){
		this.notify();
	}


	@Override
	protected void finalize() throws Throwable {
		super.finalize();
		DocumentBuilders.releaseDocumentBuilder(db);
	}

	public void addNotif(ReportList reports2) {
		Long timeStamp = new Long(System.currentTimeMillis());
		System.out.println("TIMESTAMP : "+timeStamp);
		this.notifTimeStamps.add(timeStamp);
		System.out.println("LIST : \n"+this.notifTimeStamps);
		run();

	}

	public boolean isStop() {
		return stop;
	}

	public void stopThread(){
		this.stop = true;
	}

}
