package com.petalslink.easiergov.resources;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Logger;

import javax.xml.namespace.QName;

import com.petalslink.easiergov.GovException;
import com.petalslink.easiergov.config.Configuration;
import com.petalslink.easiergov.resources.api.Resource;
import com.petalslink.easiergov.resources.api.ResourceType;
import com.petalslink.easiergov.resources.api.StorageService;
import com.petalslink.easiersbs.matching.service.SBSFactoryImpl;
import com.petalslink.easiersbs.matching.service.api.EasierSBSException;
import com.petalslink.easiersbs.matching.service.api.SBSFactory;
import com.petalslink.easiersbs.matching.service.api.matcher.similarity.SimilarityMeasure;
import com.petalslink.easiersbs.matching.service.util.SimilarityUtil;

public class StorageServiceImpl implements StorageService {
	
	private static Logger LOG = Logger.getLogger(StorageServiceImpl.class.getName());

    private Map<ResourceType, Map<QName, Resource>> memory = Collections.synchronizedMap(new HashMap<ResourceType, Map<QName, Resource>>());

	private SBSFactory factory = SBSFactoryImpl.getInstance();
	
	private SimilarityMeasure similarity;
	
	private Configuration configuration;
    
    
    public StorageServiceImpl(Configuration conf) throws GovException {
    	this.configuration = conf;
    	try {
			this.similarity = factory.newSimilarityMeasure(conf.getEasiersbsSyntacticSimilarityType(), conf.getEasiersbsSyntacticLevenshteinLimit());
		} catch (EasierSBSException e) {
			throw new GovException(e);
		}
    }


    @Override
    public QName store(Resource r) throws GovException {
        Map<QName, Resource> registry = memory.get(r.getResourceType());

        if(registry == null) {
            registry = Collections.synchronizedMap(new HashMap<QName, Resource>());
            memory.put(r.getResourceType(), registry);
        }

        if(registry.get(r.getId()) != null) {
            LOG.warning("Impossible to store this ressource \"" + r.getId() + "\" - type: " + r.getResourceType() + "!!!\nA ressource with the same type and id already exist!!!");
        }
        registry.put(r.getId(), r);
        return r.getId();
    }

    @Override
    public Resource get(QName id, ResourceType type) {
        Resource r = null;
        if(type != null) {
            Map<QName, Resource> registry = memory.get(type);
            if(registry != null) {
                r = registry.get(id);
            }
        }
        return r;
    }
    
    
    @Override
    public List<Resource> find(String query, ResourceType type) {
    	List<Resource> rscs = new ArrayList<Resource>();
    	List<String> functionalWords = SimilarityUtil.splitInWords(query);
        if(type != null) {
            Map<QName, Resource> registry = memory.get(type);
            if(registry != null) {
            	for(Resource rsc: registry.values()) {
            		List<String> expectedWords = SimilarityUtil.splitInWords(rsc.getId().getLocalPart());
					double finalRate = similarity.measureSimilarity(functionalWords, expectedWords);
					if(finalRate >= this.configuration.getEasiersbsSyntacticAcceptanceThreshold()){
            			rscs.add(rsc);
            		}
            	}
            }
        } else {
        	for(Map<QName, Resource> registry: memory.values()) {
        		if(registry != null) {
                	for(Resource rsc: registry.values()) {
                		List<String> expectedWords = SimilarityUtil.splitInWords(rsc.getId().getLocalPart());
    					double finalRate = similarity.measureSimilarity(functionalWords, expectedWords);
    					if(finalRate >= this.configuration.getEasiersbsSyntacticAcceptanceThreshold()){
                			rscs.add(rsc);
                		}
                	}
                }
        	}
        }
        return rscs;
    }
    
    
	public boolean detectSyntaxSimilarity(String request, String expected) {
		List<String> functionalWords = SimilarityUtil.splitInWords(request);
		List<String> expectedWords = SimilarityUtil.splitInWords(expected);
		double finalRate = similarity.measureSimilarity(functionalWords, expectedWords);
		if(finalRate >= this.configuration.getEasiersbsSyntacticAcceptanceThreshold()) {
			return true;
		}
		return false;
	}
    

    @Override
    public Resource unstore(QName id, ResourceType type) {
        Resource r = null;
        if(type != null) {
            Map<QName, Resource> registry = memory.get(type);
            if(registry != null) {
                r = registry.remove(id);
            }
        }
        return r;
    }

    @Override
    public void clear() {
        memory.clear();
    }

    @Override
    public List<Resource> getAllByResourceType(ResourceType type) {
        Map<QName, Resource> rsc = memory.get(type);
        if(rsc != null) {
            return new ArrayList<Resource>(rsc.values());
        }
        return new ArrayList<Resource>();
    }

    public int getNumberOfResources() {
        int res = 0;
        for(Map<QName, Resource> map: memory.values()) {
            res = res + map.values().size();
        }
        return res;
    }

    public String toString() {
        String res = "";
        for(Entry<ResourceType,Map<QName, Resource>> entry: this.memory.entrySet())  {
            for(QName name: entry.getValue().keySet()) {
                res = res + "id: " + name + " - type: " + entry.getKey().getTypeName() + "\n";
            }
        }
        return res;
    }



}
