/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.registry.service.impl.util;

import java.net.URI;
import java.net.URL;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;

import org.petalslink.abslayer.Factory;
import org.petalslink.abslayer.service.api.Description;
import org.petalslink.abslayer.service.api.Endpoint;
import org.petalslink.abslayer.service.api.Interface;
import org.petalslink.abslayer.service.api.Part;
import org.petalslink.abslayer.service.api.Service;

import com.ebmwebsourcing.easybox.api.XmlContext;
import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectNode;
import com.ebmwebsourcing.easybox.api.XmlObjectReadException;
import com.ebmwebsourcing.easybox.api.XmlObjectReader;
import com.ebmwebsourcing.easybox.api.with.WithName;
import com.ebmwebsourcing.easysawsdl10.api.SawsdlHelper;
import com.ebmwebsourcing.easyschema10.api.SchemaHelper;
import com.ebmwebsourcing.easyschema10.api.element.Choice;
import com.ebmwebsourcing.easyschema10.api.element.ComplexType;
import com.ebmwebsourcing.easyschema10.api.element.Element;
import com.ebmwebsourcing.easyschema10.api.element.Sequence;
import com.ebmwebsourcing.easyschema10.api.type.Type;
import com.ebmwebsourcing.easyschema10.api.with.WithElements;
import com.ebmwebsourcing.easywsdl11.api.element.Definitions;
import com.petalslink.easiersbs.registry.service.api.RegistryException;
import com.petalslink.easiersbs.registry.service.api.model.Operation;
import com.petalslink.easiersbs.registry.service.api.model.SemanticElement;
import com.petalslink.easiersbs.registry.service.api.model.SemanticPart;
import com.petalslink.easiersbs.registry.service.api.model.SemanticProfile;
import com.petalslink.easiersbs.registry.service.api.model.generic.GenericElement;
import com.petalslink.easiersbs.registry.service.impl.model.OperationImpl;
import com.petalslink.easiersbs.registry.service.impl.model.SemanticElementImpl;
import com.petalslink.easiersbs.registry.service.impl.model.SemanticPartImpl;
import com.petalslink.easiersbs.registry.service.impl.model.SemanticProfileImpl;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class ServiceUtil {
	
	public static Set<org.petalslink.abslayer.service.api.Operation> getOperations(Description wsdlDesc){
		Set<org.petalslink.abslayer.service.api.Operation> operations = new HashSet<org.petalslink.abslayer.service.api.Operation>();
		for(Interface inter : wsdlDesc.getInterfaces()){
			operations.addAll(Arrays.asList(inter.getOperations()));
		}
		return operations;
	}
	
	public static org.petalslink.abslayer.service.api.Operation findOperation(QName operationQName, Description wsdlDesc) {
		for(Interface inter : wsdlDesc.getInterfaces()){
			org.petalslink.abslayer.service.api.Operation op = inter.getOperation(operationQName);
			if(op != null){
				return op;
			}
		}
		return null;
	}

	public static List<Service> findServicesByOperation(QName operationQName, Description wsdlDesc){
		org.petalslink.abslayer.service.api.Operation op = findOperation(operationQName, wsdlDesc);
		List<Service> services = wsdlDesc.findServicesImplementingInterface(op.getParentInterface());
		return services;
	}
	
	public static Set<org.petalslink.abslayer.service.api.Operation> findOperationsByService(QName serviceQName, Description wsdlDesc){
		Set<org.petalslink.abslayer.service.api.Operation> ops = new HashSet<org.petalslink.abslayer.service.api.Operation>();
		Service serv = wsdlDesc.findService(serviceQName);
		for(Endpoint end : serv.getEndpoints()){
			ops.addAll(Arrays.asList(end.getBinding().getInterface().getOperations()));
		}
		return ops;
	}
	
	public static <E extends GenericElement<E>> Set<E> getFlattenElements(E element){
		Set<E> listElements = new HashSet<E>();
		return getFlattenElements(element, listElements);
	}
	
	private static <E extends GenericElement<E>> Set<E> getFlattenElements(E element, Set<E> listElements) {
		Set<E> res = new HashSet<E>();
		if(element != null) {
			res.add(element);
			listElements.add(element);
			for(E child : element.getChildElements()){
				if( ! listElements.contains(child)){
					res.addAll(getFlattenElements(child, listElements));
				}
			}
		}
		return res;
	}
	
	public static <E extends GenericElement<E>> Set<URI> getFlattenConcepts(Set<E> elements){
		Set<URI> concepts = new HashSet<URI>();
		for(E element : elements){
			concepts.addAll(element.getSemanticConcepts());
		}
		return concepts;
	}
	
	public static SemanticProfile extractSemanticProfile(QName operationQName, Description wsdlDesc) {
		SemanticProfile profile = new SemanticProfileImpl();
		
		org.petalslink.abslayer.service.api.Operation op = findOperation(operationQName, wsdlDesc);
		
		// Technical information
		List<Service> services = findServicesByOperation(operationQName, wsdlDesc);
		Operation profileOp = null;
		if( ! services.isEmpty()){
			profileOp = new OperationImpl(operationQName, services.get(0).getQName());
		}
		profile.addOperation(profileOp);
		
		// Interface
		SemanticPart interPart = extractSemanticPart(op.getParentInterface().getModel());
		profile.setSemanticInterface(interPart);
		
		// Operation
		SemanticPart opPart = extractSemanticPart(op.getModel());
		profile.setSemanticOperation(opPart);
		
		// Input
		SemanticElement input = null;
		if(op.getInput() != null
				&& op.getInput().getParts().length != 0 
				&& op.getInput().getParts()[0] != null) {
			input = extractAllSemanticElements(op.getInput().getParts()[0]);
		}
		profile.setInputSemanticElement(input);
		
		// Output
		SemanticElement output = null;
		if(op.getOutput() != null
				&& op.getOutput().getParts().length != 0 
				&& op.getOutput().getParts()[0] != null) {
			output = extractAllSemanticElements(op.getOutput().getParts()[0]);
		}
		profile.setOutputSemanticElement(output);
			
		return profile;
		
	}
	
	public static SemanticPart extractSemanticPart(XmlObject object){
		SemanticPart part = new SemanticPartImpl();
		if(object instanceof WithName
				&& ((WithName) object).hasName()) {
			part.setName(((WithName) object).getName());
		}
		URI[] concepts = SawsdlHelper.getModelReference(object);
		for(URI concept : concepts){
			part.addSemanticConcept(concept);
		}
		return part;
	}
	
	/**
	 * Extract SemanticElement and its children from WSDL Part
	 * @param part Part from EasyWSDL (input or output)
	 * @return complete SemanticElement
	 */
	public static SemanticElement extractAllSemanticElements(Part part){
		// Store list of element to avoid infinite loop
		Map<QName, SemanticElement> elementList = new HashMap<QName, SemanticElement>();
		
		// Search for main element in part (input or output)
		Element element = null;
		if(part.getElement() != null){
			element = part.getElement();
		} else if(part.getType() != null) { 
			// Create an element representing the expected one
			element = part.createElement();
			element.setName(part.getQName().getLocalPart());
			element.setType(part.getType());
			// TODO : find why each XPath request is executed twice
			Type type = SchemaHelper.findTypeByQName(part.getDescription().getModel(), part.getType());
			SchemaHelper.findParentSchema(type).addElement(element);
		}
		
		if(element != null){
			// Extract semantic element 
			return extractSemanticElement(element, elementList);
		}
		return null;
	}
	
	/**
	 * Extract SemanticElements from EasySchema Element
	 * @param element EasySchema target element
	 * @param elementList Set of SemanticElement already studied (avoid infinite loop)
	 * @return SemanticElement filled with its children
	 */
	private static SemanticElement extractSemanticElement(Element element, Map<QName, SemanticElement> elementList){
		
		QName elementQName = element.inferQName();
		if(elementList.containsKey(elementQName)){
			return elementList.get(elementQName);
		} else {
			if(element.hasRef()){
				Element realElement = SchemaHelper.findElementByQName(element.getXmlObjectBaseRoot(), element.getRef());
				return extractSemanticElement(realElement, elementList);
				
			} else {
				
				SemanticElement semanticElement = new SemanticElementImpl();
				
				// Extract syntactic part
				semanticElement.setElementQName(element.inferQName());
				semanticElement.setName(element.getName());
				if(element.getMinOccurs() == 0){
					semanticElement.setRequired(false);
				} else {
					semanticElement.setRequired(true);
				}
				// Extract semantic part
				for(URI concept : extractSemanticAnnotations(element)){
					semanticElement.addSemanticConcept(concept);
				}
				
				elementList.put(elementQName, semanticElement);
					
				// If element already exist in parsed elements, 
				// we don't extract its children
				Type type = element.findType();
				if(type instanceof ComplexType){
					XmlObjectNode[] nodes = type.getXmlObjectChildren();
					for(XmlObjectNode node : nodes){
						semanticElement.setChildElements(extractSemanticElementChildren(node, elementList));
					}
				}
				return semanticElement;
			}
		}
	}
	
	/**
	 * Extract semantic element children from Sequence, Choice or All tags
	 * @param node
	 * @param elementList
	 * @return Set<SemanticElement> element children
	 */
	private static Set<SemanticElement> extractSemanticElementChildren(XmlObjectNode node, Map<QName, SemanticElement> elementList){
		Set<SemanticElement> children = new HashSet<SemanticElement>();
		if(node instanceof WithElements){
			for(Element element : ((WithElements)node).getElements()){
				children.add(extractSemanticElement(element, elementList));
			}
		}
		if(node instanceof Sequence || node instanceof Choice){
			for(XmlObjectNode child : node.getXmlObjectChildren()){
				if( ! (child instanceof Element)){
					children.addAll(extractSemanticElementChildren(child, elementList));
				}
			}
		}
		return children;
	}
	
	/**
	 * Extract semantic concepts from EasySchema element and its type
	 * 
	 * @param element EasySchema
	 * @return List of concepts as Set of URIs
	 */
	private static Set<URI> extractSemanticAnnotations(Element element){
		Set<URI> annotations = new HashSet<URI>();

		URI[] concepts = SawsdlHelper.getModelReference(element);
		
		for(URI concept : concepts){
			annotations.add(concept);
		}
		
		URI[] typeConcepts = SawsdlHelper.getModelReference(element.findType());
		for(URI concept : typeConcepts){
			annotations.add(concept);
		}
		
		return annotations;
	}
	
	
	public static Description getDescription(URL wsdlUrl, XmlContext xmlContext) throws RegistryException {
		Definitions def = null;
		try {
			XmlObjectReader reader = xmlContext.createReader();
			def = reader.readDocument(wsdlUrl, Definitions.class);
		} catch (XmlObjectReadException e) {
			throw new RegistryException("Impossible to parse WSDL file at " + wsdlUrl.toString(), e);
		}
		return (Description) Factory.getInstance().wrap(def);
	}
	
}
