/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.matching.service.matcher;

import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.petalslink.easiersbs.matching.service.api.matcher.MatcherProperties;
import com.petalslink.easiersbs.matching.service.api.matcher.MatchingResult;
import com.petalslink.easiersbs.matching.service.api.matcher.SemanticMatcher;
import com.petalslink.easiersbs.matching.service.api.profile.SearchProfile;
import com.petalslink.easiersbs.matching.service.api.profile.inferred.InferredProfile;
import com.petalslink.easiersbs.matching.service.api.profile.rated.RatedSemanticProfile;
import com.petalslink.easiersbs.matching.service.matcher.computation.SemanticComputationImpl;
import com.petalslink.easiersbs.reasoner.api.ReasonerException;
import com.petalslink.easiersbs.reasoner.api.ReasonerFactory;
import com.petalslink.easiersbs.reasoner.api.engine.Reasoner;
import com.petalslink.easiersbs.reasoner.api.ontology.OntologyManager;
import com.petalslink.easiersbs.registry.service.api.SemanticRegistryManager;
import com.petalslink.easiersbs.registry.service.api.model.SemanticProfile;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class SemanticMatcherImpl extends AbstractMatcherImpl implements
		SemanticMatcher {

	private Reasoner reasoner = null;
	private OntologyManager manager = null;
	
	// MultiThread computation
	private ExecutorService exec;
	private ExecutorCompletionService<RatedSemanticProfile> compExec;
	
	public SemanticMatcherImpl(ReasonerFactory factory, SemanticRegistryManager registry, MatcherProperties props) throws ReasonerException{
		this.registry = registry;
		this.props = props;
		this.manager = factory.getOntologyManager();
		this.reasoner = factory.newReasoner(manager);
		exec = Executors.newFixedThreadPool(props.getMaxThreads());
		compExec = new ExecutorCompletionService<RatedSemanticProfile>(exec);
	}
	
	public SemanticMatcherImpl(ReasonerFactory factory, SemanticRegistryManager registry) throws ReasonerException{
		this(factory, registry, new MatcherPropertiesImpl());
	}


	public Reasoner getReasoner() {
		return reasoner;
	}

	public OntologyManager getOntologyManager() {
		return manager;
	}

	public MatchingResult findServices(SearchProfile profile,
			Set<SemanticProfile> serviceList) {
		MatchingResult res = new MatchingResultImpl(profile);
		
		// Infer Search profile
		InferredProfile infProfile = profile.infer(reasoner, props);
		
		Set<Future<RatedSemanticProfile>> results = new HashSet<Future<RatedSemanticProfile>>();

		// Computation of similarity measure for each service
		for(SemanticProfile service : serviceList){
			SemanticComputationImpl compute = new SemanticComputationImpl(props, infProfile, service);
			results.add(compExec.submit(compute));
		}
		
		boolean done = false;
		while(!done){
			done = true;
			for(Future<RatedSemanticProfile> future : results){
				if(!future.isDone()){
					done = false;
					break;
				}
			}
			if(!done){
				try {
					Thread.sleep(50);
				} catch (InterruptedException e) {
					throw new RuntimeException(e);
				}
			}
		}
			
		// Completion of expected result
		for(Future<RatedSemanticProfile> result : results){
			// Add service to result map if it feet with acceptance threshold
			try {
				if(result.get().getProfileRate() >= props.getSemanticThreshold()){
					res.addResultProfile(result.get());
				}
			} catch (InterruptedException e) {
				throw new RuntimeException(e);
			} catch (ExecutionException e) {
				throw new RuntimeException(e);
			}
		}
		return res;
	}
	
}
