/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.matching.service.matcher;

import java.util.Properties;

import com.petalslink.easiersbs.matching.service.api.matcher.MatcherProperties;
import com.petalslink.easiersbs.matching.service.api.matcher.similarity.SimilarityType;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class MatcherPropertiesImpl implements MatcherProperties {

	private int maxThreads = 4;
	
	private double operationWeight = 0.50;
	private double validationRate = 0.75;
	private double semanticWeight = 0.75;
	
	private double hybridAcceptanceThreshold = 0.20;
	private double semanticAcceptanceThreshold = 0.00;
	private double syntacticAcceptanceThreshold = 0.00;
	
	private double exactMark = 1.00;
	private double pluginMark = 0.90;
	private double subsumeMark = 0.50;
	
	private double equivalentMark = 0.90;
	private double specializationMark = 0.60;
	private double generalizationMark = 0.30;

	private int levenshteinLimit = 0;

	private SimilarityType similarityType = SimilarityType.DICE;
	
	private int serviceLimit = 1;
	private int activityLimit = 1;
	
	public MatcherPropertiesImpl(){
		
	}
	
	public MatcherPropertiesImpl(Properties props){
		for(Object k : props.keySet()){
			String key = k.toString().trim();
			if(MAX_THREADS.equals(key)){
				this.setMaxThreads(Integer.parseInt(props.get(key).toString().trim()));
			} else if(OPERATION_WEIGHT.equals(key)){
				this.setOperationWeight(Double.parseDouble(props.get(key).toString().trim()));
			} else if(GENERAL_VALIDATION_RATE.equals(key)){
				this.setGeneralValidationRate(Double.parseDouble(props.getProperty(key).toString().trim()));
			} else if(SEMANTIC_WEIGHT.equals(key)){
				this.setSemanticWeight(Double.parseDouble(props.get(key).toString().trim()));
			} else if(HYBRID_ACCEPTANCE_THRESHOLD.equals(key)){
				this.setHybridThreshold(Double.parseDouble(props.get(key).toString().trim()));
			} else if(SEMANTIC_ACCEPTANCE_THRESHOLD.equals(key)){
				this.setSemanticThreshold(Double.parseDouble(props.get(key).toString().trim()));
			} else if(SYNTACTIC_ACCEPTANCE_THRESHOLD.equals(key)){
				this.setSyntacticThreshold(Double.parseDouble(props.get(key).toString().trim()));
			} else if(EXACT_MARK.equals(key)){
				this.setExactMark(Double.parseDouble(props.get(key).toString().trim()));
			} else if(PLUGIN_MARK.equals(key)){
				this.setPluginMark(Double.parseDouble(props.get(key).toString().trim()));
			} else if(SUBSUME_MARK.equals(key)){
				this.setSubsumeMark(Double.parseDouble(props.get(key).toString().trim()));
			} else if(EQUIVALENT_CONCEPT_MARK.equals(key)){
				this.setSubsumeMark(Double.parseDouble(props.get(key).toString().trim()));
			} else if(SPECIALIZATION_CONCEPT_MARK.equals(key)){
				this.setSpecializationMark(Double.parseDouble(props.get(key).toString().trim()));
			} else if(GENERALIZATION_CONCEPT_MARK.equals(key)){
				this.setGeneralizationMark(Double.parseDouble(props.get(key).toString().trim()));
			} else if(LEVENSHTEIN_LIMIT.equals(key)){
				this.setLevenshteinLimit(Integer.parseInt(props.get(key).toString().trim()));
			} else if(SIMILARITY_TYPE.equals(key)){
				this.setSimilarityType(SimilarityType.valueOf(props.get(key).toString().trim()));
			} else if(COMPOSITION_SERVICE_LIMIT.equals(key)){
				this.setCompositionServiceLimit(Integer.parseInt(props.get(key).toString().trim()));
			} else if(COMPOSITION_ACTIVITY_LIMIT.equals(key)){
				this.setCompositionActivityLimit(Integer.parseInt(props.get(key).toString().trim()));
			}
		}
	}
	

	/**
	 * @return max threads
	 */
	public int getMaxThreads() {
		return maxThreads;
	}
	
	/**
	 * @return limit of studied activities in n-m compositions
	 */
	public int getCompositionActivityLimit() {
		return activityLimit;
	}

	/**
	 * @return limit of studied services in 1-n and n-m compositions
	 */
	public int getCompositionServiceLimit() {
		return serviceLimit;
	}
	
	/**
	 * @return the equivalentMark
	 */
	public double getEquivalentMark() {
		return equivalentMark;
	}

	public double getExactMark() {
		return exactMark;
	}

	/**
	 * @return the generalizationMark
	 */
	public double getGeneralizationMark() {
		return generalizationMark;
	}

	public double getInputOutputWeigth() {
		return (1.0 - operationWeight);
	}

	/**
	 * @return the general validation rate
	 */
	public double getGeneralValidationRate() {
		return validationRate;
	}
	
	/**
	 * @return the levenshteinLimit
	 */
	public int getLevenshteinLimit() {
		return levenshteinLimit;
	}

	public double getOperationWeight() {
		return operationWeight;
	}

	public double getPluginMark() {
		return pluginMark;
	}

	public double getHybridThreshold() {
		return hybridAcceptanceThreshold;
	}

	/**
	 * @return the semanticAcceptanceThreshold
	 */
	public double getSemanticAcceptanceThreshold() {
		return semanticAcceptanceThreshold;
	}

	public double getSemanticThreshold() {
		return semanticAcceptanceThreshold;
	}

	public double getSemanticWeight() {
		return semanticWeight;
	}

	public SimilarityType getSimilarityType() {
		return similarityType;
	}

	/**
	 * @return the specializationMark
	 */
	public double getSpecializationMark() {
		return specializationMark;
	}

	public double getSubsumeMark() {
		return subsumeMark;
	}

	/**
	 * @return the syntacticAcceptanceThreshold
	 */
	public double getSyntacticAcceptanceThreshold() {
		return syntacticAcceptanceThreshold;
	}

	public double getSyntacticThreshold() {
		return syntacticAcceptanceThreshold;
	}

	public double getSyntacticWeigth() {
		return (1 - semanticWeight);
	}


	public void setMaxThreads(int maxThreads) {
		if(maxThreads > 0){
			this.maxThreads = maxThreads;
		} else {
			this.maxThreads = 1;
		}
	}
	
	/**
	 * @param limit of studied activities in n-m compositions
	 */
	public void setCompositionActivityLimit(int limit) {
		if(limit > 0){
			this.activityLimit = limit;
		} else {
			this.activityLimit = 1;
		}
	}
	
	/**
	 * @param limit of studied services in 1-n and n-m compositions
	 */
	public void setCompositionServiceLimit(int limit) {
		if(limit > 0){
			this.serviceLimit = limit;
		} else {
			this.serviceLimit = 1;
		}
	}
	
	/**
	 * @param equivalentMark the equivalentMark to set
	 */
	public void setEquivalentMark(double equivalentMark) {
		this.equivalentMark = norm(equivalentMark);
	}

	public void setExactMark(double mark) {
		exactMark = norm(mark);
	}

	/**
	 * @param generalizationMark the generalizationMark to set
	 */
	public void setGeneralizationMark(double generalizationMark) {
		this.generalizationMark = generalizationMark;
	}
	
	/**
	 * @param levenshteinLimit the levenshteinLimit to set
	 */
	public void setLevenshteinLimit(int levenshteinLimit) {
		this.levenshteinLimit = levenshteinLimit;
	}

	public void setOperationWeight(double weight) {
		operationWeight = norm(weight);
	}

	public void setPluginMark(double mark) {
		pluginMark = norm(mark);
	}

	public void setHybridThreshold(double threshold) {
		this.hybridAcceptanceThreshold = norm(threshold);
	}
	
	/**
	 * @param semanticAcceptanceThreshold the semanticAcceptanceThreshold to set
	 */
	public void setSemanticAcceptanceThreshold(double semanticAcceptanceThreshold) {
		this.semanticAcceptanceThreshold = norm(semanticAcceptanceThreshold);
	}

	public void setSemanticThreshold(double threshold) {
		semanticAcceptanceThreshold = norm(threshold);
	}

	public void setSemanticWeight(double weight) {
		semanticWeight = norm(weight);
	}

	public void setSimilarityType(SimilarityType type) {
		if(type != null) {
			similarityType = type;
		}
	}

	/**
	 * @param specializationMark the specializationMark to set
	 */
	public void setSpecializationMark(double specializationMark) {
		this.specializationMark = specializationMark;
	}

	public void setSubsumeMark(double mark) {
		subsumeMark = norm(mark);
	}

	/**
	 * @param syntacticAcceptanceThreshold the syntacticAcceptanceThreshold to set
	 */
	public void setSyntacticAcceptanceThreshold(double syntacticAcceptanceThreshold) {
		this.syntacticAcceptanceThreshold = norm(syntacticAcceptanceThreshold);
	}
	
	public void setSyntacticThreshold(double threshold) {
		syntacticAcceptanceThreshold = norm(threshold);
	}


	/** 
	 * @param double
	 * @return double between 0 and 1
	 */
	private double norm(double d){
		if(d > 1) {
			return 1;
		} else if (d < 0) {
			return 0;
		} else {
			return d;
		}
	}



	@Override
	public void setGeneralValidationRate(double rate) {
		// TODO Auto-generated method stub
		
	}

	





	
}
