/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.matching.service.util;

import java.net.URI;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.SortedSet;
import java.util.UUID;

import javax.xml.namespace.QName;

import com.petalslink.easiersbs.matching.service.api.CompositionException;
import com.petalslink.easiersbs.matching.service.api.matcher.MatcherProperties;
import com.petalslink.easiersbs.matching.service.api.profile.SearchElement;
import com.petalslink.easiersbs.matching.service.api.profile.SearchPart;
import com.petalslink.easiersbs.matching.service.api.profile.SearchProfile;
import com.petalslink.easiersbs.matching.service.api.profile.inferred.InferredConcept;
import com.petalslink.easiersbs.matching.service.api.profile.inferred.InferredElement;
import com.petalslink.easiersbs.matching.service.api.profile.inferred.RatedURI;
import com.petalslink.easiersbs.matching.service.api.profile.rated.RatedSemanticProfile;
import com.petalslink.easiersbs.matching.service.profile.SearchElementImpl;
import com.petalslink.easiersbs.matching.service.profile.SearchPartImpl;
import com.petalslink.easiersbs.matching.service.profile.SearchProfileImpl;
import com.petalslink.easiersbs.matching.service.profile.inferred.InferredConceptImpl;
import com.petalslink.easiersbs.matching.service.profile.inferred.RatedURIImpl;
import com.petalslink.easiersbs.reasoner.api.engine.Reasoner;
import com.petalslink.easiersbs.registry.service.api.SemanticRegistryManager;
import com.petalslink.easiersbs.registry.service.api.model.Operation;
import com.petalslink.easiersbs.registry.service.api.model.SemanticElement;
import com.petalslink.easiersbs.registry.service.api.model.SemanticPart;
import com.petalslink.easiersbs.registry.service.api.model.SemanticProfile;
import com.petalslink.easiersbs.registry.service.api.model.generic.GenericElement;
import com.petalslink.easiersbs.registry.service.impl.model.SemanticElementImpl;
import com.petalslink.easiersbs.registry.service.impl.model.SemanticPartImpl;
import com.petalslink.easiersbs.registry.service.impl.model.SemanticProfileImpl;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class ProfileUtil {
	
	public static Set<RatedURI> getFlattenRatedUris(InferredElement element){
		Set<InferredElement> listElements = new HashSet<InferredElement>();
		return getFlattenRatedUris(element, listElements);
	}
	
	private static Set<RatedURI> getFlattenRatedUris(InferredElement element, Set<InferredElement> listElements){
		Set<RatedURI> res = new HashSet<RatedURI>();
		if( ! listElements.contains(element)){
			listElements.add(element);
			for(InferredConcept concept : element.getInferedSemanticConcepts()){
				res.addAll(concept.getRatedSemanticConcepts());
			}
			for(InferredElement child : element.getChildElements()){
				res.addAll(getFlattenRatedUris(child, listElements));
			}
		}
		return res;
	}
	
	/**
	 * Convert a SemanticElement hierarchy into SearchElement one
	 * 
	 * @param element SemanticElement source
	 * @return target SearchElement
	 */
	public static SearchElement convertSemanticElement(SemanticElement element){
		return convertSemanticElementHierarchy(element, new HashMap<QName, SearchElement>());
	}
	
	private static SearchElement convertSemanticElementHierarchy(SemanticElement elem, Map<QName, SearchElement> elementList){
		SearchElement res = new SearchElementImpl();
		res.setSemanticConcepts(elem.getSemanticConcepts());
		res.setElementQName(elem.getElementQName());
		res.setName(elem.getName());
		res.setRequired(elem.isRequired());
		elementList.put(elem.getElementQName(), res);
		for(SemanticElement child : elem.getChildElements()){
			if(elementList.containsKey(child.getElementQName())){
				res.addChildElement(elementList.get(child.getElementQName()));
			} else {
				res.addChildElement(convertSemanticElementHierarchy(child, elementList));
			}
		}
		return res;
	}
	
	/**
	 * Search for related concepts in available ontologies
	 * @param concept Initial concept URI
	 * @param reasoner EasierSBS Reasoner
	 * @param props Property File
	 * @return InferedSemanticConcept containing all related concepts
	 */
	public static InferredConcept inferConcept(URI concept, Reasoner reasoner, MatcherProperties props){
		InferredConcept inferedConcept = new InferredConceptImpl();
		// Direct concept
		inferedConcept.addRatedSemanticConcept(new RatedURIImpl(concept, 1.0));
		
		if(reasoner.isConcept(concept)){
			// Equivalent concepts
			for(URI eqConcept : reasoner.getEquivalentClasses(concept)){
				if(!reasoner.isThingOrNothing(eqConcept)){
					inferedConcept.addRatedSemanticConcept(new RatedURIImpl(eqConcept, props.getEquivalentMark()));
				}
			}
			
			// Specialization concepts
			Set<URI> specConcepts = reasoner.getSubClasses(concept);
			specConcepts.addAll(reasoner.getIndividuals(concept));
			for(URI specConcept : specConcepts){
				if(!reasoner.isThingOrNothing(specConcept)){
					inferedConcept.addRatedSemanticConcept(new RatedURIImpl(specConcept, props.getSpecializationMark()));
				}
			}
			
			// Generalization concepts
			for(URI genConcept : reasoner.getSuperClasses(concept)){
				if(!reasoner.isThingOrNothing(genConcept)){
					inferedConcept.addRatedSemanticConcept(new RatedURIImpl(genConcept, props.getGeneralizationMark()));
				}
			}
			
		} else if (reasoner.isInstance(concept)) { // URI refers to an instance
			// Equivalent instances
			for(URI eqInstance : reasoner.getSimilarIndividuals(concept)){
				inferedConcept.addRatedSemanticConcept(new RatedURIImpl(eqInstance, props.getEquivalentMark()));
			}
			
			// Parent concepts
			for(URI genConcept : reasoner.getTypes(concept)){
				if(!reasoner.isThingOrNothing(genConcept)){
					inferedConcept.addRatedSemanticConcept(new RatedURIImpl(genConcept, props.getGeneralizationMark()));
				}
			}
		}
		return inferedConcept;
	}
	
	public static Set<SemanticProfile> getSemanticProfiles(SortedSet<RatedSemanticProfile> ratedProfiles) {
		Set<SemanticProfile> res = new HashSet<SemanticProfile>();
		for(RatedSemanticProfile ratedProfile : ratedProfiles) {
			res.add(ratedProfile.getProfile());
		}
		return res;
	}
	
	/**
	 * Create an equivalent semantic profile from a list of profiles (taking account internal dependencies)
	 * 
	 * @param operations list of involved operations
	 * @param registry SemanticRegistryManager containing involved semantic profiles
	 * @param reasoner semantic reasoner to infer on semantic concepts
	 * @param props MatcherProperties
	 * @return an equivalent semantic profile
	 * @throws CompositionException if a dependency loop exist between involved operations
	 */
	public static SemanticProfile computeEquivalentProfile(Set<Operation> operations, 
			SemanticRegistryManager registry, Reasoner reasoner, MatcherProperties props) throws CompositionException{
		
		Map<Operation, SemanticElement> inputs = new HashMap<Operation, SemanticElement>();
		Map<Operation, SemanticElement> outputs = new HashMap<Operation, SemanticElement>();
		Map<Operation, Set<Operation>> dependencies = new HashMap<Operation, Set<Operation>>();
		Map<QName, QName> elementMatchings = new HashMap<QName, QName>();
		
		// Init maps and result
		SemanticProfile res = new SemanticProfileImpl();
		res.setSemanticInterface(new SemanticPartImpl());
		res.setSemanticOperation(new SemanticPartImpl());

		for(Operation op : operations){
			dependencies.put(op, new HashSet<Operation>());
			SemanticProfile profile = registry.getSemanticProfile(op);
			if(profile.getInputSemanticElement() != null){
				inputs.put(op, profile.getInputSemanticElement());
			}
			if(profile.getOutputSemanticElement() != null){
				outputs.put(op, profile.getOutputSemanticElement());
			}
			
			res.addOperation(op);
			if(res.getPartner() == null){ // All partners are identical
				res.setPartner(profile.getPartner());
			}
			combineSemanticParts(res.getSemanticInterface(), profile.getSemanticInterface());
			combineSemanticParts(res.getSemanticOperation(), profile.getSemanticOperation());
		}
		
		// Search for element matchings
		for(Operation op1 : operations){
			if(inputs.get(op1) != null){
				SearchElement input = convertSemanticElement(inputs.get(op1));
				InferredElement infInput = input.infer(reasoner, props);
				for(Operation op2 : operations){
					if( ! op1.equals(op2) 
							&& outputs.get(op2) != null){
						Map<QName, QName> matchElemOp1Op2 = SimilarityUtil.findElementMatchings(infInput, outputs.get(op2));
						if( ! matchElemOp1Op2.isEmpty()){
							dependencies.get(op1).add(op2);
							elementMatchings.putAll(matchElemOp1Op2);
						}
					}
				}
			}
		}

		// Check for loops
		if(hasCycle(dependencies)){
			throw new CompositionException("There is a dependency loop between operations " 
											+ operations.toString());
		}
		
		// Complete profile with input/output
		res.setInputSemanticElement(createUncoveredElement(new HashSet<SemanticElement>(inputs.values()), 
															elementMatchings.keySet(), SemanticElementImpl.class));
		res.setOutputSemanticElement(createUncoveredElement(new HashSet<SemanticElement>(outputs.values()), 
															new HashSet<QName>(elementMatchings.values()), SemanticElementImpl.class));
		
		return res;
	}
	
	
	/**
	 * Detect cycles in operation dependency graph
	 * 
	 * @param dependencies Map containing list of dependencies for each operation
	 * @return true if a cycle is detected
	 */
	public static boolean hasCycle(Map<Operation, Set<Operation>> dependencies){
		Set<Operation> independentNodes = new HashSet<Operation>();
		// We search for nodes without dependency
		for(Entry<Operation, Set<Operation>> dependency : dependencies.entrySet()){
			if(dependency.getValue().isEmpty()){
				independentNodes.add(dependency.getKey());
			}
		}
		
		// We delete dependencies while it's possible
		while( ! independentNodes.isEmpty()){
			Operation op = independentNodes.iterator().next();
			independentNodes.remove(op);
			for(Entry<Operation, Set<Operation>> dependency : dependencies.entrySet()){
				if(dependency.getValue().remove(op)){ 
					// op was in this dependency list
					if(dependency.getValue().isEmpty()){ 
						// dependency became an independent node
						independentNodes.add(dependency.getKey());
					}
				}
			}
		}
		
		// We search for remaining edges (dependencies)
		for(Set<Operation> dep : dependencies.values()){
			if(! dep.isEmpty()){
				return true;
			}
		}
		return false;
	}
	
	

	
	/**
	 * Create a complementary research profile for service composition (1-n and n-m) 
	 * only containing concepts uncovered by substractProfile ones
	 * 
	 * @param initialProfile
	 * @param substractProfile profile to substract to the initial profile
	 * @param reasoner semantic reasoner to infer on semantic concepts
	 * @param props MatcherProperties
	 * @return new SearchProfile only containing uncovered concepts
	 */
	public static SearchProfile computeComplementaryProfile(SearchProfile initialProfile, SemanticProfile substractProfile, 
			Reasoner reasoner, MatcherProperties props){
		
		SearchProfile res = new SearchProfileImpl();
		
		for(Operation op : substractProfile.getOperations()){
			res.addOperation(op);
		}
		
		res.setSemanticInterface(substractPart(initialProfile.getSemanticInterface(), substractProfile.getSemanticInterface()));
		res.setSemanticOperation(substractPart(initialProfile.getSemanticOperation(), substractProfile.getSemanticOperation()));
		
		res.setInputSemanticElement(substractElement(initialProfile.getInputSemanticElement(), 
													 substractProfile.getInputSemanticElement(), 
													 reasoner, props));
		
		res.setOutputSemanticElement(substractElement(initialProfile.getOutputSemanticElement(), 
				 substractProfile.getOutputSemanticElement(), 
				 reasoner, props));
		
		return res;
	}
	
	
	/**
	 * Create a complementary research part containing concepts uncovered by part ones
	 * 
	 * @param initialPart
	 * @param part
	 * @return complementary research part
	 */
	private static SearchPart substractPart(SearchPart initialPart, SemanticPart part){
		if(initialPart != null){
			SearchPart res = new SearchPartImpl();
			res.setName(initialPart.getName());
			for(URI concept : initialPart.getSemanticConcepts()){
				if(part != null){
					if(! part.getSemanticConcepts().contains(concept)){
						res.addSemanticConcept(concept);
					}
				} else {
					res.addSemanticConcept(concept);
				}
			}
			return res;
		}
		return null;
	}
	
	
	private static SearchElement substractElement(SearchElement initElem, SemanticElement elem,
			Reasoner reasoner, MatcherProperties props){
		if(initElem != null){
			if(elem == null){
				return initElem;
			} else {
				InferredElement infInit = initElem.infer(reasoner, props);
				Map<QName, QName> matchInput = SimilarityUtil.findElementMatchings(infInit, elem);
				return createUncoveredElement(initElem, matchInput.keySet(), new HashSet<SearchElement>(), SearchElementImpl.class);
			}
			
		}
		return null;
	}
	
	
	/**
	 * Add name and concept of part2 into part1
	 * 
	 * @param part1
	 * @param part2
	 */
	private static void combineSemanticParts(SemanticPart part1, SemanticPart part2){
		if(part2 != null){
			part1.setName(part1.getName() 
					+ "-" + part2.getName());
			for(URI concept : part2.getSemanticConcepts()){
				part1.addSemanticConcept(concept);
			}
		}
	}
	
	
	/**
	 * Create from an element list and a list of used element, the complementary element expected in equivalent profiles
	 * TODO: improve this code
	 * 
	 * @param elements
	 * @param usedElements
	 * @return complementary element
	 */
	@SuppressWarnings("unchecked")
	private static <E extends GenericElement<E>, EImpl extends E> E 
			createUncoveredElement(Set<E> elements, Set<QName> elementToDelete, Class<EImpl> type){

		Set<E> results = new HashSet<E>();
		
		for(E element : elements){
			Set<E> usedElements = new HashSet<E>();
			E compElem = createUncoveredElement(element, elementToDelete, usedElements, type);
			if(compElem != null){
				results.add(compElem);
			}
		}
		
		// Return minimum result
		if(results.isEmpty()){
			return null;
		} else if (results.size() == 1){
			return results.iterator().next();
		} else { // results.size() > 1
			E res = null;
			try {
				res = (E) type.getConstructors()[0].newInstance();
			} catch (Exception e) {
				e.printStackTrace();
			}
//			if(results.iterator().next() instanceof SemanticElement){
//				res = (E) new SemanticElementImpl();
//			} else if(results.iterator().next() instanceof SearchElement) {
//				res = (E) new SearchElementImpl();
//			}
			res.setElementQName(new QName("generated", UUID.randomUUID().toString()));
			res.setName("generatedElement");
			for(E child : results){
				res.addChildElement(child);
			}
			return res;
		}
	}
	
	/**
	 *  Semantic Element is returned only if it doesn't belong to the element to delete list 
	 *  and has uncovered children or no child
	 */
	private static <E extends GenericElement<E>, EImpl extends E> E 
			createUncoveredElement(E element, Set<QName> elementToDelete, Set<E> usedElements, Class<EImpl> type){
		
		if(( ! element.isRequired())
				|| elementToDelete.contains(element.getElementQName())){
			return null;
		} 		
		
		E res = cloneElementWithoutHierarchy(element, type);
		
		// Element is required and not covered
		if(element.hasChildElement()
				&& ( ! usedElements.contains(element))){
			
			usedElements.add(element);
			
			Set<E> uncoveredChildren = new HashSet<E>();
			for(E child : element.getChildElements()){
				E compElem = createUncoveredElement(child, elementToDelete, usedElements, type);
				if(compElem != null){
					uncoveredChildren.add(compElem);
				}
			}
			
			if(uncoveredChildren.isEmpty()){ // All children are covered
				return null;
			} else {
				for(E child : uncoveredChildren){
					res.addChildElement(child);
				}
			}
		}
		
		return res;
	}
	
	@SuppressWarnings(value = { "unchecked" })
	private static <E extends GenericElement<E>, EImpl extends E> E cloneElementWithoutHierarchy(E elem, Class<EImpl> type){
		E res = null;
		try {
			res = (E) type.getConstructors()[0].newInstance();
		} catch (Exception e) {
			e.printStackTrace();
		}
//		if(elem instanceof SemanticElement){
//			res = (E) new SemanticElementImpl();
//		} else if(elem instanceof SearchElement) {
//			res = (E) new SearchElementImpl();
//		}
		res.setName(elem.getName());
		res.setElementQName(elem.getElementQName());
		for(URI concept : elem.getSemanticConcepts()){
			res.addSemanticConcept(concept);
		}
		return res;
	}
//	/**
//	 * Create from an element list and a list of used element, the complementary element expected in equivalent profiles
//	 * TODO: improve this code
//	 * 
//	 * @param elements
//	 * @param usedElements
//	 * @return complementary element
//	 */
//	private static SemanticElement createUncoveredElement(Set<SemanticElement> elements, Set<QName> elementToDelete){
//
//		Set<SemanticElement> results = new HashSet<SemanticElement>();
//		
//		for(SemanticElement element : elements){
//			Set<SemanticElement> usedElements = new HashSet<SemanticElement>();
//			SemanticElement compElem = createUncoveredElement(element, elementToDelete, usedElements);
//			if(compElem != null){
//				results.add(compElem);
//			}
//		}
//		
//		// Return minimum result
//		if(results.isEmpty()){
//			return null;
//		} else if (results.size() == 1){
//			return results.iterator().next();
//		} else { // results.size() > 1
//			SemanticElement res = new SemanticElementImpl();
//			res.setElementQName(new QName("generated", UUID.randomUUID().toString()));
//			res.setName("generatedElement");
//			for(SemanticElement child : results){
//				res.addChildElement(child);
//			}
//			return res;
//		}
//	}
//	
//	/**
//	 *  Semantic Element is returned only if it doesn't belong to the element to delete list 
//	 *  and has uncovered children or no child
//	 */
//	private static SemanticElement createUncoveredElement(SemanticElement element, Set<QName> elementToDelete, Set<SemanticElement> usedElements){
//		
//		if(( ! element.isRequired())
//				|| elementToDelete.contains(element.getElementQName())){
//			return null;
//		} 		
//		
//		SemanticElement res = cloneSemanticElementWithoutHierarchy(element);
//		
//		// Element is required and not covered
//		if(element.hasChildElement()
//				&& ( ! usedElements.contains(element))){
//			
//			usedElements.add(element);
//			
//			Set<SemanticElement> uncoveredChildren = new HashSet<SemanticElement>();
//			for(SemanticElement child : element.getChildElements()){
//				SemanticElement compElem = createUncoveredElement(child, elementToDelete, usedElements);
//				if(compElem != null){
//					uncoveredChildren.add(compElem);
//				}
//			}
//			
//			if(uncoveredChildren.isEmpty()){ // All children are covered
//				return null;
//			} else {
//				for(SemanticElement child : uncoveredChildren){
//					res.addChildElement(child);
//				}
//			}
//		}
//		
//		return res;
//	}
//	
//	private static SemanticElement cloneSemanticElementWithoutHierarchy(SemanticElement elem){
//		SemanticElement res = new SemanticElementImpl();
//		res.setName(elem.getName());
//		res.setElementQName(elem.getElementQName());
//		for(URI concept : elem.getSemanticConcepts()){
//			res.addSemanticConcept(concept);
//		}
//		return res;
//	}
}
