package org.petalslink.easiestdemo.wsoui.plugin.wsdl;

/*
 * Copyright 2001-2005 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.logging.Logger;

import javax.xml.namespace.QName;

import org.apache.cxf.Bus;
import org.apache.cxf.BusFactory;
import org.apache.cxf.tools.common.ToolContext;
import org.apache.cxf.tools.wsdlto.WSDLToJava;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.petalslink.easiestdemo.wsoui.core.WSOUIException;
import org.petalslink.easiestdemo.wsoui.core.reflexive.ImplMetaInf;
import org.petalslink.easiestdemo.wsoui.core.wsdl.EventProducerNotifierGenerator;
import org.petalslink.easiestdemo.wsoui.core.wsdl.ExchangeProcessorGenerator;
import org.petalslink.easiestdemo.wsoui.core.wsdl.ImplementationManagerGenerator;
import org.petalslink.easiestdemo.wsoui.core.wsdl.ServerGenerator;
import org.petalslink.easiestdemo.wsoui.core.wsdl.ServiceGenerator;
import org.petalslink.easiestdemo.wsoui.core.wsdl.TestImplementationGenerator;
import org.petalslink.easiestdemo.wsoui.core.wsdl.WSDLJUnitTestGenerator;
import org.petalslink.easiestdemo.wsoui.core.wsdl.WSOUIJaxbContextGenerator;
import org.petalslink.easiestdemo.wsoui.provided.ProviderConfiguration.PositionType;

import com.ebmwebsourcing.easiestdemo.contant.EasiestDEMOFramework;
import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectReadException;
import com.ebmwebsourcing.easycommons.research.util.easybox.SOAUtil;
import com.ebmwebsourcing.easycommons.research.util.io.FileUtil;
import com.ebmwebsourcing.easycommons.uri.URIHelper;
import com.ebmwebsourcing.easysawsdl10.api.SawsdlHelper;
import com.ebmwebsourcing.easyschema10.api.element.Schema;
import com.ebmwebsourcing.easywsdl11.api.element.Definitions;
import com.ebmwebsourcing.easywsdl11.api.element.Import;
import com.ebmwebsourcing.easywsdl11.api.element.PortType;
import com.ebmwebsourcing.easywsdl11.api.element.Service;
import com.ebmwebsourcing.soapbinding11.api.SoapBindingHelper;
import com.ebmwebsourcing.soapbinding11.api.element.Address;


/**
 * @goal wsdl2java
 * @phase generate-sources
 * @description WSOUI WSDL To Java Tool
 * @requiresDependencyResolution test
 */
public class WSDL2JavaGenerateSourcesMojo extends AbstractMojo {

	static Logger LOG = Logger.getLogger(WSDL2JavaGenerateSourcesMojo.class.getName());

	public static String IMAGE_METADATA_PATH = "http://www.petalslink.org/wsoui/image";
	public static String POSITION_METADATA_PATH = "http://www.petalslink.org/wsoui/position";
	public static String TOPICSET_METADATA_PATH = "http://www.petalslink.org/wsoui/topicset";
	public static String TOPICNAMESPACE_METADATA_PATH = "http://www.petalslink.org/wsoui/topicnamespace";


	List<String> resourcesRootList = Arrays.asList("/src/main/resources", "src/test/resources");

	/**
	 * Path where the sources should be placed
	 */
	String sourceRoot = "/src/main/java";

	/**
	 * Path where the test sources should be placed
	 */
	String testsourceRoot = "/src/test/java";

	/**
	 * 
	 */
	String generatedTestSourceRoot = "/target/generated-test-sources/wsoui";


	/**
	 * Path where the generated sources should be placed
	 * 
	 */
	String generatedSourceRoot = "/target/generated-sources/wsoui";

	URI basedir = new File(".").toURI();

	/**
	 * @parameter expression="${project}"
	 * @required
	 */
	MavenProject project;


	/**
	 * @parameter
	 */
	WsdlOption wsdlOptions[];

	/**
	 * @parameter
	 */
	boolean activeGUI = false;


	private List<String> allServerNames = new ArrayList<String>();
	private List<ImplMetaInf> allMetainfs = new ArrayList<ImplMetaInf>();


	File sourceRootFile = null;
	File testsourceRootFile = null;
	File generatedTestSourceRootFile = null;
	File generatedSourceRootFile = null;


	private boolean compileClass = true; 


	public void execute() throws MojoExecutionException {


		List<WsdlOption> effectiveWsdlOptions = Arrays.asList(this.wsdlOptions);

		if (effectiveWsdlOptions.size() == 0) {
			getLog().info("Nothing to generate");
			return;
		}


		if(project != null) {
			basedir = project.getBasedir().toURI();
			sourceRootFile = new File(FileUtil.getURI(basedir, sourceRoot));
			testsourceRootFile = new File(FileUtil.getURI(basedir, testsourceRoot));
			generatedTestSourceRootFile = new File(FileUtil.getURI(basedir, generatedTestSourceRoot));
			generatedSourceRootFile = new File(FileUtil.getURI(basedir, generatedSourceRoot));
		} else {
			sourceRootFile = new File(URI.create(basedir.toString() + sourceRoot));
			testsourceRootFile = new File(URI.create(basedir.toString() + testsourceRoot));
			generatedTestSourceRootFile = new File(URI.create(basedir.toString() + generatedTestSourceRoot));
			generatedSourceRootFile = new File(URI.create(basedir.toString() + generatedSourceRoot));
		}

		try {
			List<Provider> providers = new ArrayList<Provider>();
			for (WsdlOption o : effectiveWsdlOptions) {

				Definitions definitions = SOAUtil.getInstance().getReader(EasiestDEMOFramework.getInstance()).get().readDocument(FileUtil.getURI(basedir, o.getWsdl()).toURL(), Definitions.class);
				if(definitions.getServices().length > 0) {
					// generate endpoint if required
					for(Service s: definitions.getServices()) {
						providers.add(createProviderFromService(s));
					}


					QName eventProducerItfQName = new QName("http://www.petalslink.com/wsn/service/WsnProducer", "NotificationProducer");
					URI itf = this.findInterface(definitions, eventProducerItfQName);
					o.setRelatifEventImport(itf);
					o.setSchemaLocationOfWsAddressing(this.findSchemaLocationImportFromNamespace(definitions, "http://www.w3.org/2005/08/addressing"));

					callWsdl2Java(o);     

					try {
						// find implementation sources
						List<File> implementations = new ArrayList<File>();
						implementations.addAll(findImplementationFile(generatedSourceRootFile));
						implementations.addAll(findImplementationFile(generatedTestSourceRootFile));


						// analyze all implementations
						List<ImplMetaInf> metainfs = new ArrayList<ImplMetaInf>();
						for(File f: implementations) {
							metainfs.add(new ImplMetaInf(f, new File(FileUtil.getURI(basedir, o.getWsdl()))));
						}

						// create implementation manager file
						for(ImplMetaInf imf: metainfs) {
							if(imf.getTargetNamespace().equals(definitions.getTargetNamespace())) {
								ImplementationManagerGenerator img = new ImplementationManagerGenerator(definitions.getTargetNamespace(), imf);
								img.generate(generatedSourceRootFile.toString());
							}
						}

						// create test implementation file
						for(ImplMetaInf imf: metainfs) {
							if(imf.getTargetNamespace().equals(definitions.getTargetNamespace())) {
								TestImplementationGenerator timg = new TestImplementationGenerator(definitions.getTargetNamespace(), imf);
								timg.generate(generatedSourceRootFile.toString());
							}
						}

						// create wsoui service file
						for(ImplMetaInf imf: metainfs) {
							if(imf.getTargetNamespace().equals(definitions.getTargetNamespace())) {
								ServiceGenerator sg = new ServiceGenerator(definitions.getTargetNamespace(), imf);
								sg.generate(generatedSourceRootFile.toString());
							}
						}


						// create wsdl junit test file
						for(ImplMetaInf imf: metainfs) {
							if(imf.getTargetNamespace().equals(definitions.getTargetNamespace())) {
								WSDLJUnitTestGenerator wjutg = new WSDLJUnitTestGenerator(definitions.getTargetNamespace(), imf);
								wjutg.generate(testsourceRootFile.toString());
							}
						}

						if(o.isGenerateBusinessClass() == true) {
							// create exchange processor implementation file
							for(ImplMetaInf imf: metainfs) {
								if(imf.getTargetNamespace().equals(definitions.getTargetNamespace())) {
									ExchangeProcessorGenerator epg = new ExchangeProcessorGenerator(definitions.getTargetNamespace(), imf, o.isOverrideOlderBusinessClass());
									epg.generate(sourceRootFile.toString());
								}
							}


							// create server implementation file
							System.out.println("*******************------------------- o.getProvider() = " + o.getProvider());
							if(o.getProvider()) {
								for(ImplMetaInf imf: metainfs) {
									Provider correspondingProvider = findProviderCorrespondingToImpl(providers, imf); 
									if(correspondingProvider != null) {
										if(imf.getTargetNamespace().equals(definitions.getTargetNamespace())) {

											// generate event producer
											EventProducerNotifierGenerator evg = new EventProducerNotifierGenerator(definitions.getTargetNamespace(), imf, correspondingProvider.getTopicSet(), correspondingProvider.getTopicNameSpace(), o.isOverrideOlderBusinessClass());
											evg.generate(sourceRootFile.toString());

											// generate server
											ServerGenerator sg = new ServerGenerator(definitions.getTargetNamespace(), imf, correspondingProvider, activeGUI, o.isOverrideOlderBusinessClass());
											sg.generate(sourceRootFile.toString());

											// add server class name at AserverNAmes (used for bpel mojo)
											if(!this.allServerNames.contains(sg.getClassName())) {
												this.allServerNames.add(sg.getClassName());
											}
										}
									}
								}
							} 
						} else {
							for(ImplMetaInf imf: metainfs) {
								Provider correspondingProvider = findProviderCorrespondingToImpl(providers, imf); 
								if(correspondingProvider != null) {
									if(imf.getTargetNamespace().equals(definitions.getTargetNamespace())) {

										// generate server
										ServerGenerator sg = new ServerGenerator(definitions.getTargetNamespace(), imf, correspondingProvider, activeGUI, o.isOverrideOlderBusinessClass());
										
										// add server class name at AserverNAmes (used for bpel mojo)
										if(!this.allServerNames.contains(sg.deduceClassName(sourceRootFile.toString()))) {
											this.allServerNames.add(sg.deduceClassName(sourceRootFile.toString()));
										}
									}
								}
							}
						}

						// get all files
						List<File> oldFiles = new ArrayList<File>();
						oldFiles.addAll(implementations);
						oldFiles.addAll(findServiceFile(generatedSourceRootFile));
						oldFiles.addAll(findServiceFile(generatedTestSourceRootFile));
						oldFiles.addAll(findServerFile(generatedSourceRootFile));
						oldFiles.addAll(findServerFile(generatedTestSourceRootFile));

						// delete all files
						for(File impl: oldFiles) {
							impl.delete();
						}

						// find event directory (contains jaxb element on ws-notif)
						FileUtil.deleteDirectory(new File(URIHelper.resolve(generatedSourceRootFile.toURI(), "com/ebmwebsourcing/wsstar/jaxb")));
						FileUtil.deleteDirectory(new File(URIHelper.resolve(generatedSourceRootFile.toURI(), "easybox/org/w3/_2005/_08/addressing")));

						this.allMetainfs.addAll(metainfs);
					} catch(WSOUIException e) {
						throw new MojoExecutionException(e.getMessage(), e);
					} catch (IOException e) {
						throw new MojoExecutionException(e.getMessage(), e);
					} catch (URISyntaxException e) {
						throw new MojoExecutionException(e.getMessage(), e);
					}
				} else {
					LOG.warning("this definitions contains no services: " + o.getWsdl());
				}


			}

			// find objectFactory sources
			List<File> objectFactories = new ArrayList<File>();
			objectFactories.addAll(findObjectFactoryFile(generatedSourceRootFile));
			WSOUIJaxbContextGenerator wjcg = new WSOUIJaxbContextGenerator(objectFactories);
			try {
				wjcg.generate(generatedSourceRootFile.toString());
			} catch (IOException e) {
				throw new MojoExecutionException(e.getMessage(), e);
			}

			// Verify if providers have not same address
			for(Provider p1: providers) {
				String port = this.getPort(p1.getEndpointAddress());
				for(Provider p2: providers) {
					if(p1 != p2 && port.equals(this.getPort(p2.getEndpointAddress()))) {
						throw new MojoExecutionException("CXF Implementation restriction: 2 endpoints must not have the same port: " + p1.getEndpointAddress() + " and " + p2.getEndpointAddress());
					}
				}
			}
		} catch (MalformedURLException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		} catch (XmlObjectReadException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		} catch (WSOUIException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		} catch (URISyntaxException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		} finally {
			// cleanup as much as we can.
			Bus bus = BusFactory.getDefaultBus(false);
			if (bus != null) {
				bus.shutdown(true);
			}
			org.apache.cxf.tools.wsdlto.core.PluginLoader.unload();
		}


		if(this.isCompileClass()) {
			// add all sources to the compilation process
			if (project != null && generatedSourceRootFile != null && generatedSourceRootFile.exists()) {
				project.addCompileSourceRoot(generatedSourceRootFile.getAbsolutePath());
			}
			if (project != null && generatedTestSourceRootFile != null && generatedTestSourceRootFile.exists()) {
				project.addTestCompileSourceRoot(generatedTestSourceRootFile.getAbsolutePath());
			}
			System.gc();
		}
	}

	private URI findInterface(Definitions definitions, QName eventProducerItfQName) {
		PortType res = definitions.getPortTypeByName(eventProducerItfQName.getLocalPart());
		if(definitions.getTargetNamespace().trim().toString().equals(eventProducerItfQName.getNamespaceURI().trim().toString()) && res != null) {
			return res.getXmlObjectBaseURI();
		} else {
			for(Import impt: definitions.getImports()) {
				return findInterface((Definitions) impt.getXmlObjectAdoptedChildren()[0], eventProducerItfQName);
			}
		}
		return null;
	}

	private URI findSchemaLocationImportFromNamespace(Definitions definitions, String ns) throws MalformedURLException, URISyntaxException {
		URI res = null;

		if(definitions.getTypes() != null) {
			for(XmlObject  obj: definitions.getTypes().getAnyXmlObjects()) {
				if(obj instanceof Schema) {
					Schema s = (Schema)obj;
					res = findSchemaLocationImportFromNamespace(s, ns);
				} else if(obj instanceof com.ebmwebsourcing.easyschema10.api.element.Import) {
					com.ebmwebsourcing.easyschema10.api.element.Import impt = (com.ebmwebsourcing.easyschema10.api.element.Import)obj;
					if(impt.getNamespace().equals(ns)) {
						return URIHelper.resolve(impt.getXmlObjectBaseURI(), impt.getSchemaLocation());
					} else {
						res = findSchemaLocationImportFromNamespace((Schema) impt.getXmlObjectAdoptedChildren()[0], ns);
						if(res != null) {
							break;
						}
					}
				}
			}
		}
		if(res == null) {
			for(Import impt: definitions.getImports()) {
				res = findSchemaLocationImportFromNamespace((Definitions) impt.getXmlObjectAdoptedChildren()[0], ns);
				if(res != null) {
					break;
				}
			}
		}
		return res;
	}

	private URI findSchemaLocationImportFromNamespace(Schema schema, String ns) throws MalformedURLException, URISyntaxException {
		URI res = null;
		for(com.ebmwebsourcing.easyschema10.api.element.Import impt: schema.getImports()) {
			if(impt.getNamespace().equals(ns)) {
				return URIHelper.resolve(impt.getXmlObjectBaseURI(), impt.getSchemaLocation());
			} else {
				res = findSchemaLocationImportFromNamespace((Schema) impt.getXmlObjectAdoptedChildren()[0], ns);
				if(res != null) {
					break;
				}
			}
		}
		return res;
	}

	private Provider findProviderCorrespondingToImpl(List<Provider> providers,
			ImplMetaInf imf) {
		Provider res = null;
		for(Provider p: providers) {
			if(imf.getServiceName().equals(p.getServiceQName().getLocalPart())) {
				res = p;
				break;
			}
		}
		return res;
	}

	private String getPort(String endpointAddress) {
		String port = "80";

		if(endpointAddress != null && endpointAddress.contains(":")) {
			port = endpointAddress.replace("http://", "");
			port = port.substring(port.indexOf(":")+1, port.length());
			port = port.substring(0, port.indexOf("/"));
		}
		return port;
	}

	private Provider createProviderFromService(Service service) throws WSOUIException {
		Provider res = new Provider();
		List<URI> uris = Arrays.asList(SawsdlHelper.getModelReference(service));
		String image = findImage(uris);
		String topicset = findTopicSet(uris);
		String topicnamespace = findTopicNamespace(uris);
		PositionType pos = findPosition(uris);
		if(image != null) {
			res.setImage(image.toString());
		}
		if(pos != null) {
			res.setPosition(pos);
		}
		if(topicset != null) {
			res.setTopicSet(topicset);
		}
		if(topicnamespace != null) {
			res.setTopicNameSpace(topicnamespace);
		}

		if(service.getPorts().length != 1) {
			throw new WSOUIException("EasiestDemo Restriction: Each service must contain only one endpoint");
		}

		res.setServiceQName(service.inferQName());

		Address address = SoapBindingHelper.getAddress(service.getPorts()[0]);
		if(address != null) {
			res.setEndpointAddress(address.getLocation());
		} else {
			throw new WSOUIException("Impossible to find endpoint address: " + service.getPorts()[0].getName());
		}
		return res;
	}



	private PositionType findPosition(List<URI> uris) {
		PositionType res = null;
		for(URI uri: uris) {
			if(uri.toString().startsWith(WSDL2JavaGenerateSourcesMojo.POSITION_METADATA_PATH)) {
				String position = uri.toString().replace(WSDL2JavaGenerateSourcesMojo.POSITION_METADATA_PATH + "#", "");
				res = PositionType.getPositionTypeFromString(position);
				break;
			}
		}
		return res;
	}



	private String findImage(List<URI> uris) throws WSOUIException {
		String res = null;
		try {
			for(URI uri: uris) {
				if(uri.toString().startsWith(WSDL2JavaGenerateSourcesMojo.IMAGE_METADATA_PATH)) {
					String imagePath = uri.toString().replace(WSDL2JavaGenerateSourcesMojo.IMAGE_METADATA_PATH + "#", "");
					if(imagePath.startsWith("classpath://")) {
						res = imagePath.replace("classpath://","");

						URL url = Thread.currentThread().getContextClassLoader().getResource(res);
						if(url == null) {
							// find in resource directory of projects himself
							for(String resourcesDir: this.resourcesRootList) {
								File resourceRootFile = new File(FileUtil.getURI(basedir, resourcesDir));
								String resource = resourceRootFile + "/" + res;
								url = new File(resource).toURI().toURL();
							}
						}
						if(url == null) {
							throw new WSOUIException("Impossible to find resource image: " + res);
						}
						break;
					} else {
						res = imagePath;
						break;
					}
				}
			}
		} catch (MalformedURLException e) {
			throw new WSOUIException(e);
		}
		return res;
	}

	private String findTopicSet(List<URI> uris) throws WSOUIException {
		String res = null;
		try {
			for(URI uri: uris) {
				if(uri.toString().startsWith(WSDL2JavaGenerateSourcesMojo.TOPICSET_METADATA_PATH)) {
					String topicSetPath = uri.toString().replace(WSDL2JavaGenerateSourcesMojo.TOPICSET_METADATA_PATH + "#", "");
					if(topicSetPath.startsWith("classpath://")) {
						res = topicSetPath.replace("classpath://","");
						URL url = Thread.currentThread().getContextClassLoader().getResource(res);
						if(url == null) {
							// find in resource directory of projects himself
							for(String resourcesDir: this.resourcesRootList) {
								File resourceRootFile = new File(FileUtil.getURI(basedir, resourcesDir));
								String resource = resourceRootFile + "/" + res;
								url = new File(resource).toURI().toURL();
							}
						}
						if(url == null) {
							throw new WSOUIException("Impossible to find resource image: " + res);
						}
						break;
					} else {
						res = topicSetPath;
						break;
					}
				}
			}
		} catch (MalformedURLException e) {
			throw new WSOUIException(e);
		}
		return res;
	}

	private String findTopicNamespace(List<URI> uris) throws WSOUIException {
		String res = null;
		try {
			for(URI uri: uris) {
				if(uri.toString().startsWith(WSDL2JavaGenerateSourcesMojo.TOPICNAMESPACE_METADATA_PATH)) {
					String topicNsPath = uri.toString().replace(WSDL2JavaGenerateSourcesMojo.TOPICNAMESPACE_METADATA_PATH + "#", "");
					if(topicNsPath.startsWith("classpath://")) {
						res = topicNsPath.replace("classpath://","");
						URL url = Thread.currentThread().getContextClassLoader().getResource(res);
						if(url == null) {
							// find in resource directory of projects himself
							for(String resourcesDir: this.resourcesRootList) {
								File resourceRootFile = new File(FileUtil.getURI(basedir, resourcesDir));
								String resource = resourceRootFile + "/" + res;
								url = new File(resource).toURI().toURL();
							}
						}
						if(url == null) {
							throw new WSOUIException("Impossible to find resource image: " + res);
						}
						break;
					} else {
						res = topicNsPath;
						break;
					}
				}
			}
		} catch (MalformedURLException e) {
			throw new WSOUIException(e);
		}
		return res;
	}


	private Collection<? extends File> findImplementationFile(File dir) {
		List<File> res = new ArrayList<File>();
		if(dir.isDirectory()) {
			for(File f: dir.listFiles()) {
				res.addAll(findImplementationFile(f));
			}
		} else if(dir.toString().endsWith("Impl.java")) {
			res.add(dir);
		}
		return res;
	}

	private Collection<? extends File> findServiceFile(File dir) {
		List<File> res = new ArrayList<File>();
		if(dir.isDirectory()) {
			for(File f: dir.listFiles()) {
				res.addAll(findServiceFile(f));
			}
		} else if(dir.toString().endsWith("_Service.java")) {
			res.add(dir);
		}
		return res;
	}

	private Collection<? extends File> findObjectFactoryFile(File dir) {
		List<File> res = new ArrayList<File>();
		if(dir.isDirectory()) {
			for(File f: dir.listFiles()) {
				res.addAll(findObjectFactoryFile(f));
			}
		} else if(dir.toString().endsWith("ObjectFactory.java")) {
			res.add(dir);
		}
		return res;
	}


	private Collection<? extends File> findServerFile(File dir) {
		List<File> res = new ArrayList<File>();
		if(dir.isDirectory()) {
			for(File f: dir.listFiles()) {
				res.addAll(findServerFile(f));
			}
		} else if(dir.toString().endsWith("_Server.java")) {
			res.add(dir);
		}
		return res;
	}

	private void callWsdl2Java(WsdlOption wsdlOption) throws MojoExecutionException {
		File outputDirFile = this.generatedSourceRootFile;
		outputDirFile.mkdirs();
		try {
			WSOUIBindingGenerator gen = new WSOUIBindingGenerator(wsdlOption);
			List<String> list = wsdlOption.generateCommandLine(basedir, outputDirFile.toString());
			String[] args = (String[])list.toArray(new String[list.size()]);
			getLog().info("Calling wsdl2java with args: " + Arrays.toString(args));
			System.out.println("Calling wsdl2java with args: " + Arrays.toString(args));

			new WSDLToJava(args).run(new ToolContext());
		} catch (Throwable e) {
			getLog().debug(e);
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}



	public WsdlOption[] getWsdlOptions() {
		return wsdlOptions;
	}

	public void setWsdlOptions(WsdlOption[] wsdlOptions) {
		this.wsdlOptions = wsdlOptions;
	}

	public String getSourceRoot() {
		return sourceRoot;
	}

	public void setSourceRoot(String sourceRoot) {
		this.sourceRoot = sourceRoot;
	}

	public String getTestsourceRoot() {
		return testsourceRoot;
	}

	public void setTestsourceRoot(String testsourceRoot) {
		this.testsourceRoot = testsourceRoot;
	}

	public String getGeneratedTestSourceRoot() {
		return generatedTestSourceRoot;
	}

	public void setGeneratedTestSourceRoot(String generatedTestSourceRoot) {
		this.generatedTestSourceRoot = generatedTestSourceRoot;
	}

	public String getGeneratedSourceRoot() {
		return generatedSourceRoot;
	}

	public void setGeneratedSourceRoot(String generatedSourceRoot) {
		this.generatedSourceRoot = generatedSourceRoot;
	}

	public boolean isActiveGUI() {
		return activeGUI;
	}

	public void setActiveGUI(boolean activeGUI) {
		this.activeGUI = activeGUI;
	}

	public List<String> getAllServerNames() {
		return this.allServerNames;
	}

	public MavenProject getProject() {
		return project;
	}

	public void setProject(MavenProject project) {
		this.project = project;
	}

	public boolean isCompileClass() {
		return compileClass;
	}

	public void setCompileClass(boolean compileClass) {
		this.compileClass = compileClass;
	}

	public List<ImplMetaInf> getAllMetainfs() {
		return allMetainfs;
	}

}
