package com.ebmwebsourcing.easybox.api.analysis;

import static com.ebmwebsourcing.easybox.api.ClassMetadataConstants.*;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;

import com.ebmwebsourcing.easybox.api.ModelObject;
import com.ebmwebsourcing.easybox.api.XmlContext;
import com.ebmwebsourcing.easybox.api.XmlObjectBinding;
import com.ebmwebsourcing.easybox.api.XmlObjectNode;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;

public final class Analyzer {

    private static final String SUFFIXE_TYPE_PACKAGE = ".type";
	private final XmlObjectBinding xmlObjectBinding;
    private final ClassMetadata classMetadata;

    public Analyzer(XmlObjectBinding xmlObjectBinding,
            ClassMetadata classMetadata) {
        this.xmlObjectBinding = xmlObjectBinding;
        this.classMetadata = classMetadata;
    }

    public void analyze() throws AnalyzerException {
        for (Class<? extends XmlObjectNode> xmlObjectNodeImplClass : xmlObjectBinding
                .getFactorableClasses()) {
            classMetadata.put(xmlObjectNodeImplClass,
                    IMPLEMENTATION_CLASS_BINDING, xmlObjectBinding);
            analyzeClass(xmlObjectNodeImplClass);
            
            Class<? extends XmlObjectNode> xmlObjectNodeInterfaceClass =
                classMetadata.get(xmlObjectNodeImplClass, IMPLEMENTATION_CLASS_INTERFACE_CLASS);
            classMetadata.put(xmlObjectNodeInterfaceClass,
                    INTERFACE_CLASS_BINDING, xmlObjectBinding);
            
        }
    }

    void analyzeClass(Class<? extends XmlObjectNode> xmlObjectNodeImplClass)
            throws AnalyzerException {
        checkUnexpectedImplementationKind(xmlObjectNodeImplClass);
        checkUnexpectedPublicImplementation(xmlObjectNodeImplClass);
        checkMultipleInterfacesImplemented(xmlObjectNodeImplClass);
        checkInterfaceClassIsPublic(xmlObjectNodeImplClass);
        checkImplementationClassConstructor(xmlObjectNodeImplClass);

        // TODO this rule should move on JAXB side ; rule on .element, .type,
        // .anonymoustype packages as well!
        checkConstantQName(xmlObjectNodeImplClass);

    }

    private void checkUnexpectedImplementationKind(
            Class<? extends XmlObjectNode> xmlObjectNodeImplClass)
            throws UnexpectedImplementationKindException {
        if (xmlObjectNodeImplClass.isInterface()) {
            throw new UnexpectedImplementationKindException(
                    xmlObjectNodeImplClass);
        }
    }

	@SuppressWarnings("unchecked")
    private void checkUnexpectedPublicImplementation(
            Class<? extends XmlObjectNode> xmlObjectNodeImplClass)
            throws UnexpectedPublicImplementationException {
    	if(xmlObjectNodeImplClass.getInterfaces().length == 1) {
			Class<? extends XmlObjectNode> interfacesExtendingXmlObject = (Class<? extends XmlObjectNode>) xmlObjectNodeImplClass.getInterfaces()[0];
    		if((!interfacesExtendingXmlObject.getPackage().getName().endsWith(SUFFIXE_TYPE_PACKAGE))&&(Modifier.isPublic(xmlObjectNodeImplClass.getModifiers()))) {
                throw new UnexpectedPublicImplementationException(
                        xmlObjectNodeImplClass);
            } 
    	} else if (Modifier.isPublic(xmlObjectNodeImplClass.getModifiers())) {
            throw new UnexpectedPublicImplementationException(
                    xmlObjectNodeImplClass);
        }
    }

    @SuppressWarnings("unchecked")
    private void checkMultipleInterfacesImplemented(
            Class<? extends XmlObjectNode> xmlObjectNodeImplClass)
            throws MultipleInterfacesImplementedException {

        List<Class<? extends XmlObjectNode>> interfacesExtendingXmlObject = new ArrayList<Class<? extends XmlObjectNode>>();
        for (Class<?> interfaceClass : xmlObjectNodeImplClass.getInterfaces()) {
            if (XmlObjectNode.class.isAssignableFrom(interfaceClass)) {
                interfacesExtendingXmlObject
                        .add((Class<? extends XmlObjectNode>) interfaceClass);
            }
        }
        if (interfacesExtendingXmlObject.size() != 1) {
            throw new MultipleInterfacesImplementedException(
                    xmlObjectNodeImplClass);
        }
        classMetadata.put(xmlObjectNodeImplClass,
                IMPLEMENTATION_CLASS_INTERFACE_CLASS,
                interfacesExtendingXmlObject.get(0));
        classMetadata.put(interfacesExtendingXmlObject.get(0),
                INTERFACE_CLASS_IMPLEMENTATION_CLASS, xmlObjectNodeImplClass);
    }

    private void checkInterfaceClassIsPublic(
            Class<? extends XmlObjectNode> xmlObjectNodeImplClass)
            throws InterfaceClassMustBePublicException {
        Class<? extends XmlObjectNode> interfaceClass = classMetadata.get(
                xmlObjectNodeImplClass, IMPLEMENTATION_CLASS_INTERFACE_CLASS);
        if (!Modifier.isPublic(interfaceClass.getModifiers())) {
            throw new InterfaceClassMustBePublicException(interfaceClass);
        }
    }

    @SuppressWarnings("unchecked")
    private void checkImplementationClassConstructor(
            Class<? extends XmlObjectNode> xmlObjectNodeImplClass)
            throws ImplementationClassConstructorException {
        Class<? extends ModelObject> modelObjectClass = null;
        Constructor<Class<? extends XmlObjectNode>> foundConstructor = null;
        for (Constructor<?> constructor : xmlObjectNodeImplClass
                .getDeclaredConstructors()) {
            Class<?>[] parameterTypes = constructor.getParameterTypes();
            if ((parameterTypes.length == 2)
                    && (XmlContext.class.isAssignableFrom(parameterTypes[0]))
                    && (ModelObject.class.isAssignableFrom(parameterTypes[1]))) {
                if (modelObjectClass != null) {
                    throw new ImplementationClassConstructorException(
                            xmlObjectNodeImplClass);
                }
                foundConstructor = (Constructor<Class<? extends XmlObjectNode>>) constructor;
                modelObjectClass = (Class<? extends ModelObject>) parameterTypes[1];
            } else {
                throw new ImplementationClassConstructorException(
                        xmlObjectNodeImplClass);
            }
        }
        if (modelObjectClass == null) {
            throw new ImplementationClassConstructorException(
                    xmlObjectNodeImplClass);
        }
        classMetadata.put(xmlObjectNodeImplClass,
                IMPLEMENTATION_CLASS_MODELOBJECT_CLASS, modelObjectClass);
        classMetadata.put(xmlObjectNodeImplClass,
                IMPLEMENTATION_CLASS_CONSTRUCTOR, foundConstructor);

    }

    private void checkConstantQName(
            Class<? extends XmlObjectNode> xmlObjectNodeImplClass) {
        try {
            Class<? extends XmlObjectNode> interfaceClass = classMetadata.get(
                    xmlObjectNodeImplClass,
                    IMPLEMENTATION_CLASS_INTERFACE_CLASS);
            Field qnameField = interfaceClass.getField("QNAME");
            QName constantQName = (QName) qnameField.get(interfaceClass);
            // TODO
            // if
            // (xmlContext.getModelObjectClassRegistry().isModelObjectClassBasedOnElement(modelObjectClass)
            // &&
            // (constantQName != null)) {
            // throw new XmlObjectMetaInfException(
            // String
            // .format(
            // "Interface class '%s' define a QName constant whereas its model is based on element.",
            // interfaceClass.getSimpleName()));
            // }

            classMetadata.put(interfaceClass, INTERFACE_CLASS_CONSTANT_QNAME,
                    constantQName);
            classMetadata.put(xmlObjectNodeImplClass,
                    IMPLEMENTATION_CLASS_CONSTANT_QNAME, constantQName);

        } catch (SecurityException se) {
        } catch (NoSuchFieldException nsfe) {
        } catch (IllegalArgumentException e) {
            throw new UncheckedException("Illegal argument.", e);
        } catch (IllegalAccessException e) {
            throw new UncheckedException("Illegal access.", e);
        }
    }

}
