package com.ebmwebsourcing.easybox.api;

import java.util.Map;

import javax.xml.namespace.QName;

import org.w3c.dom.Node;

/**
 * An {@link XmlObject} provides a base interface on a binding on an XML
 * structure.<br/>
 * <br/> {@link XmlObject} are organized as a tree structure. Every object knows its
 * parent. Every object can have children of two sorts :<br/>
 * <ul>
 * <li>natural children : those which directly reflect children in internal
 * model,</li>
 * <li>adopted children : those who reflect a convenience parent-child
 * relationship. They do not have any matching child in the internal model tree.
 * For instance, including an XML document could result in adding the included
 * {@link XmlObject} root as an adopted child of the originating including
 * {@link XmlObject}.
 * </ul>
 * 
 * It is assumed that a third-party library is used to create the binding object
 * itself. {@link XmlObject} can thus be seen as implementation independent
 * wrapping around the generated POJO (the "model").<br/>
 * 
 * @author mjambert
 * 
 */
public interface XmlObject extends XmlObjectNode {

	public static final XmlObject[] EMPTY_ARRAY = new XmlObject[0];

	/**
	 * Get {@link XmlContext} from which this {@link XmlObject} originated.
	 * 
	 * @return {@link XmlContext} which originated this object.
	 */
	XmlContext getXmlContext();

	
	/**
	 * Create a duplicate of this {@link XmlContext}.
	 * All descendants are duplicated as well ; parent hierarchy is NOT duplicated.
	 * 
	 * @return Duplicated instance of this {@link XmlContext};
	 */
	XmlObject duplicateXmlObject();
	
	<X extends XmlObject> X duplicateXmlObjectAs(Class<X> targetInterfaceClass);



	/**
	 * Set qname of this {@link XmlObject}, if allowed.
	 * 
	 * @param qname
	 *            New qname.
	 */
	void setXmlObjectQName(QName qname);

	/**
	 * Get all attributes defined on this {@link XmlObject}.
	 * 
	 * @return All attributes defined on this {@link XmlObject}, collected
	 *         inside a map.
	 */
	Map<QName, Object> getXmlObjectAttributes();

	/**
	 * Get list of all natural child {@link XmlObject}.
	 * 
	 * @return Properly ordered list of all natural child objects.
	 */
	XmlObjectNode[] getXmlObjectNaturalChildren();

	/**
	 * Get list of all adopted child {@link XmlObject}.
	 * 
	 * @return Properly ordered list of all adopted child objects.
	 */
	XmlObjectNode[] getXmlObjectAdoptedChildren();

	/**
	 * Get associated DOM node, if known.
	 * 
	 * @return Associated DOM node, or {@code null} if none.
	 */
	Node getXmlObjectDOMNode();

	/**
	 * Get all namespace bindings in scope of this {@link XmlObject}.
	 * 
	 * @return Map of prefix => namespace URI known on this {@link XmlObject} (ie declared either
	 *     on corresponding XML element, or on ancestors).
	 */
	Map<String, String> getXmlObjectInScopeNamespaces();

	
	
}
