/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easybox.api;

import java.net.URI;

import org.w3c.dom.Node;

/**
 * Empty interface flagging an object as being a model object.<br/>
 * A model object can be wrapped into an {@link XmlObject}.<br/>
 * A model object knows which unique {@link XmlObject} is attached to it.
 * Otherly said, as long as the {@link ModelObject} object stays in memory, it
 * remembers the {@link XmlObject} which wrapped it.
 * 
 * @author mjambert
 * 
 */
public interface ModelObject {

    
    /**
     * Duplicate this {@link ModelObject}.
     * 
     * @return Copy of this {@link ModelObject}.
     */
    ModelObject duplicate();
    
    
    /**
     * Duplicate this {@link ModelObject} data, but into a specific
     * target class.
     * 
     * @param <X>
     * @param targetClass
     * @return 
     */
    <X extends ModelObject> X duplicateAs(Class<X> targetClass);
    
    
    /**
     * Get unique {@link XmlObjectNode} wrapping this {@link ModelObject}.
     * 
     * @return {@link XmlObjectNode} wrapping this {@link ModelObject}.
     */
    XmlObjectNode getXmlObject();

    
    /**
     * Get natural parent object.
     * 
     * @return This object natural parent.
     */
    ModelObject getNaturalParent();

    /**
     * Get adoptive parent object.
     * 
     * @return This object adoptive parent.
     */
    ModelObject getAdoptiveParent();
    
    /**
     * Get natural or adoptive parent object.
     * 
     * @return This natural or adoptive object parent.
     */
    ModelObject getParent();
    
    
    /**
     * Get base URI of originating document.
     * 
     * If no base URI was set, this object will look
     * for return its parent base URI (recursively).
     *  
     * @return Originating document base URI.
     */
    URI getBaseURI();
    
    
    /**
     * Set base URI of originating document.
     * 
     * @param baseUri Originating base URI.
     */
    void setBaseURI(URI baseUri);
    

    Node getDOMNode();
}
