/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easybox.api;

import java.net.URI;

import javax.xml.namespace.QName;


/**
 * Convenience interface to be shared between {@link XmlObject}
 * {@link XmlObjectText}, and {@link XmlObjectAttribute}.<br/>
 * 
 * @author mjambert
 *
 */
public interface XmlObjectNode extends Comparable<XmlObjectNode> {

    public static final XmlObjectNode[] EMPTY_ARRAY = new XmlObjectNode[0];
	
	
	/**
	 * Get parent of this {@link XmlObjectNode}.
	 * 
	 * @return This {@link XmlObject} or {@code null} if none.
	 */
	XmlObject getXmlObjectParent();

	/**
	 * Get list of all child {@link XmlObjectNode}.
	 * 
	 * @return Properly ordered list of all children objects.
	 */
	XmlObjectNode[] getXmlObjectChildren();

    /**
     * Get text content.
     * 
     * @return Text content of this {@link XmlObjectNode} or {@code null} if none.
     */
    String getXmlObjectTextContent();
    
    Object getXmlObjectValue();
    
    /**
     * Get qualified name of XML node corresponding to this {@link XmlObjectNode}.
     * 
     * @return QName of this {@link XmlObjectNode} or {@code null} if not known,
     *  or not applicable.
     */
    QName getXmlObjectQName();
    
    
    /**
     * Get XPath leading to this {@link XmlObjectNode},
     * if executed from base root.
     * 
     * @return XPath leading to this {@link XmlObjectNode}.
     */
    String getXmlObjectBaseXPath();
    
    
    Integer[] getXmlObjectBaseIndexes();
    
    
    /**
     * Get index of this {@link XmlObjectNode}.
     * 
     * @return Position (starting from 0) of this {@link XmlObjectNode} in its parent
     *         children list.
     */
    int getXmlObjectIndex();

    
    /**
     * Get following sibling of this {@link XmlObjectNode}.
     * 
     * @return Following sibling of this {@link XmlObjectNode}, or {@code null} if none.
     */
    XmlObjectNode getXmlObjectFollowingSibling();
    
    /**
     * Get preceding sibling of this {@link XmlObjectNode}.
     * 
     * @return Preceding sibling of this {@link XmlObjectNode}, or {@code null} if none.
     */
    XmlObjectNode getXmlObjectPrecedingSibling();
    
    /**
     * Get {@link XmlObjectNode} ancestors.
     * 
     * @return Array of {@link XmlObjectNode} containing all ancestors of this object.
     */
    XmlObjectNode[] getXmlObjectAncestors();

    /**
     * Get followers of this {@link XmlObjectNode}.
     * 
     * @return Followers of this {@link XmlObjectNode}.
     */
    XmlObjectNode[] getXmlObjectFollowing();
    
    
    /**
     * Get following siblings of this {@link XmlObjectNode}.
     * 
     * @return Following siblings of this {@link XmlObjectNode}.
     */
    XmlObjectNode[] getXmlObjectFollowingSiblings();

    /**
     * Get preceding siblings of this {@link XmlObjectNode}.
     * 
     * @return Preceding siblings of this {@link XmlObjectNode}.
     */
    XmlObjectNode[] getXmlObjectPrecedingSiblings();

    /**
     * Get preceding {@link XmlObjectNode} of this {@link XmlObjectNode}.
     * 
     * @return Preceding {@link XmlObjectNode} of this {@link XmlObjectNode}.
     */
    XmlObjectNode[] getXmlObjectPreceding();
    
    /**
     * Get preceding or ancestor {@link XmlObjectNode} of this {@link XmlObjectNode}.
     * 
     * @return Preceding or ancestor {@link XmlObjectNode} of this {@link XmlObjectNode}.
     */
    XmlObjectNode[] getXmlObjectPrecedingOrAncestor();
  
    /**
     * Get root parent of this {@link XmlObjectNode} hierarchy.
     * 
     * @return Root parent of this {@link XmlObjectNode}.
     */
    XmlObject getXmlObjectRoot();

    /**
     * Get list of all descendants {@link XmlObjectNode}.
     * 
     * @return Properly ordered list of all descendant objects.
     */
    XmlObjectNode[] getXmlObjectDescendants();
    
    /**
     * Get base URI of document from where this {@link XmlObjectNode} originated.
     * 
     * @return Base URI of originating document, if known, {@code null}
     *         otherwise.
     */
    URI getXmlObjectBaseURI();

    /**
     * Get root of this {@link XmlObjectNode} base document
     * (the further ancestor which has same base URI).
     * 
     * @return Root of this {@link XmlObjectNode} base document.
     */
    XmlObject getXmlObjectBaseRoot();


    /**
     * Whether or not user data is set on this object.
     * 
     * @return {@code true} if user data is not null, {@code false} otherwise.
     */
    boolean hasUserData();
    
    /**
     * Get user data associated with this object.
     * 
     * @return User data associated with this object.
     */
    Object getUserData();
    
    /**
     * Associates user data with this object.
     * Anyone is free to store any object which might be convenient
     * on client side.
     * 
     * @param userData User data to associate.
     */
    void setUserData(Object userData);
    
}
