
package com.ebmwebsourcing.easybox.impl;

import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.w3c.dom.Node;

import com.ebmwebsourcing.easybox.api.ClassMetadataConstants;
import com.ebmwebsourcing.easybox.api.XmlContext;
import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectNode;
import com.ebmwebsourcing.easybox.api.with.WithName;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.easycommons.lang.reflect.ReflectionHelper;

public abstract class AbstractJaxbXmlObjectImpl<Model extends AbstractJaxbModelObject> extends
        AbstractXmlObjectImpl<Model> {

    protected static final QName ANY_QNAME = new QName("______anyQname______");

    private QName qname;

    protected AbstractJaxbXmlObjectImpl(XmlContext xmlContext, Model jaxbModel) {
        super(xmlContext, jaxbModel);
        this.qname = null;
    }

    @Override
    public QName getXmlObjectQName() {
        if (qname == null) {
            QName modelQName = getModelObject().getQName();
            if (modelQName != null) {
                qname = modelQName;
            } else if (getXmlContext().getClassMetadata().has(getClass(),
                    ClassMetadataConstants.IMPLEMENTATION_CLASS_CONSTANT_QNAME)) {
                qname = getXmlContext().getClassMetadata().get(getClass(),
                        ClassMetadataConstants.IMPLEMENTATION_CLASS_CONSTANT_QNAME);
            }
        }
        return qname;
    }

    @Override
    public void setXmlObjectQName(QName qname) {
        if ((getModelObject().getQName() != null)
                || (getXmlContext().getClassMetadata().has(getClass(),
                        ClassMetadataConstants.IMPLEMENTATION_CLASS_CONSTANT_QNAME))) {
            throw new UncheckedException(
                    String
                            .format(
                                    "Setting a QName on XML object of class '%s' is not allowed ; it has a well known QName equal to '%s'.",
                                    getClass().getSimpleName(), getXmlObjectQName()));
        }
        this.qname = qname;
    }

    protected Model createCompliantModel() {
        try {
            Class<? extends Model> compliantModelClass = getCompliantModelClass();
            assert !Modifier.isAbstract(compliantModelClass.getModifiers());
            Model model = ReflectionHelper.newInstance(compliantModelClass);
            return model;
        } catch (InvocationTargetException e) {
            throw new UncheckedException(
                    "Cannot create default compliant model (InvocationTargetException)", e);
        }
    }

    /**
     * Add a new implementation of {@link XmlObject} to underlying model
     * children list.
     * 
     * @param <J>
     *            Type of JAXB children.
     * @param jaxbChildrenList
     *            List of underlying JAXB children.
     * @param xmlObjectImplChild
     *            {@link XmlObject} child to add.
     */
    @SuppressWarnings("unchecked")
    protected final <J> void addToChildren(List<J> jaxbChildrenList, XmlObjectNode childXmlObject) {
        assert childXmlObject instanceof AbstractXmlObjectNodeImpl<?>;
        AbstractXmlObjectNodeImpl<?> axo = (AbstractXmlObjectNodeImpl<?>) childXmlObject;

        Field field = JaxbReflector.findFieldContainingChild(getModelObject(), jaxbChildrenList);

        Object modelObject = (Object) axo.getModelObject();
        
        if(modelObject instanceof TextModelObject){
            modelObject = ((TextModelObject)modelObject).getValue();
            assert modelObject != null;
            if (String.valueOf(modelObject).trim().isEmpty()) {
                // JAXB does not add empty String in children of an element with mixed content ;
                // for consistency sake, we should do nothing when adding empty text.
                return;
            }
        }
        else if (JaxbReflector.isFieldCollectionOfJaxbElement(field,(AbstractJaxbModelObject)modelObject)) {
            if (axo.getXmlObjectQName() != null && (!(modelObject instanceof JAXBElement))) {
                QName alternativeQname = JaxbReflector.guessQNameFromJaxbElementCollectionField(
                        (AbstractJaxbModelObject) modelObject, field);
                if (alternativeQname == null) {
                    modelObject = (Object) new JAXBElement(axo.getXmlObjectQName(), modelObject
                            .getClass(), modelObject);
                } else {
                    modelObject = (Object) new JAXBElement(alternativeQname,
                            modelObject.getClass(), field.getDeclaringClass(), modelObject);
                }
            }
        }
        
        jaxbChildrenList.add((J) modelObject);
        axo.setNaturalParent(this);
    }

    /**
     * Add a new implementation of {@link XmlObject} to any list.
     * 
     * @param <Object>
     *            Type of JAXB children.
     * @param jaxbChildrenList
     *            List of underlying JAXB children.
     * @param xmlObjectImplChild
     *            {@link XmlObject} child to add.
     */
    @SuppressWarnings("unchecked")
    protected final <Object> void addToAny(List<Object> jaxbChildrenList, XmlObject childXmlObject) {
        assert childXmlObject instanceof AbstractXmlObjectImpl<?>;
        AbstractXmlObjectImpl<?> axo = (AbstractXmlObjectImpl<?>) childXmlObject;

        Object anyElmt = (Object) axo.getModelObject();
        if (axo.getXmlObjectQName() != null && (!(anyElmt instanceof JAXBElement))) {
            anyElmt = (Object) new JAXBElement(axo.getXmlObjectQName(), anyElmt.getClass(), null,
                    anyElmt);
        }
        jaxbChildrenList.add(anyElmt);
        axo.setNaturalParent(this);
    }

    protected final void setChild(XmlObject child,
            Class<? extends XmlObject> xmlObjectInterfaceClass) {
        assert xmlObjectInterfaceClass != null;
        QName qname = getXmlContext().getClassMetadata().get(xmlObjectInterfaceClass,
                ClassMetadataConstants.INTERFACE_CLASS_CONSTANT_QNAME);

        assert qname != null : "Unexpected null QName, child should correspond to an api element, and thus have a constant QName.";
        if (child != null) {
            AbstractJaxbXmlObjectImpl<?> ajo = (AbstractJaxbXmlObjectImpl<?>) child;
            ajo.getModelObject().setNaturalParent(getModelObject());
            JaxbReflector.setJaxbChild(getModelObject(), ajo.getModelObject(), qname);
        } else {
            JaxbReflector.setJaxbChild(getModelObject(), null, qname);
        }
    }

    /**
     * Clear objects of a given class from underlying JAXB children list.
     * 
     * @param jaxbChildrenList
     *            List of underlying JAXB children.
     * @param filterClass
     *            Class of model objects to remove from list.
     * @param filterQName
     *            Filter QName, or {@code ANY_QNAME} if not applicable.
     */
    protected final void clearChildren(List<?> jaxbChildrenList, Class<?> filterClass,
            QName filterQName) {
        Iterator<?> it = jaxbChildrenList.iterator();
        while (it.hasNext()) {
            Object o = it.next();
            if (o instanceof JAXBElement<?>) {
                JAXBElement<?> je = (JAXBElement<?>) o;
                if ((!filterQName.equals(ANY_QNAME)) && (!filterQName.equals(je.getName())))
                    continue;
                o = je.getValue();
            }
            if (!filterClass.isInstance(o))
                continue;
            it.remove();
        }
    }

    /**
     * Get an {@link XmlObject} child by its name.
     * 
     * @param <X>
     *            Type of desired child.
     * @param children
     *            Array of {@link XmlObject} children.
     * @param name
     *            Searched name.
     * @return Found {@link XmlObject} child, or {@code null} if not found.
     */
    protected final <X extends WithName> X getChildByName(X[] children, String name) {
        assert children != null;
        assert name != null;
        for (X child : children) {
            if (name.equals(child.getName()))
                return child;
        }
        return null;
    }

    @Override
    public XmlObjectNode[] getXmlObjectAdoptedChildren() {
        return XmlObjectNode.EMPTY_ARRAY;
    }

    @Override
    public final XmlObjectNode[] getXmlObjectNaturalChildren() {
        Object[] jaxbChildren = JaxbReflector.collectJaxbChildren(this.getModelObject());
        XmlObjectNode[] children = new XmlObjectNode[jaxbChildren.length];
        for (int i = 0; i < jaxbChildren.length; ++i) {
            assert jaxbChildren[i] != null;
            children[i] = getXmlContext().getXmlObjectFactory().wrap(jaxbChildren[i]);
        }
        return children;
    }

    @Override
    public final Map<QName, Object> getXmlObjectAttributes() {
        return JaxbReflector.collectJaxbAttributes(this.getModelObject());
    }

    /**
     * Create {@link XmlObject} children array for a parent {@link XmlObject}
     * and an underlying JAXB filter class.
     * 
     * @param <X>
     *            Desired {@link XmlObject} interface type for created children.
     * @param jaxbChildrenList
     *            Model children list.
     * @param filterClass
     *            Filter class to only consider a JAXB specific class of
     *            children (it often happens that JAXB mixes children types in a
     *            single internal list).
     * @param filterQName
     *            Filter QName, or {@code ANY_QNAME} if not applicable.
     * @param xmlObjectInterfaceClass
     *            Desired {@link XmlObject} interface class for created
     *            children.
     * @param parent
     *            Parent {@link XmlObject} for created children.
     * @return Array containing properly filtered JAXB children, wrapped as
     *         {@link XmlObject}.
     */
    @SuppressWarnings("unchecked")
    protected final <X extends XmlObjectNode> X[] createChildrenArray(List<?> jaxbChildrenList,
            Class<?> filterClass, QName filterQName, Class<X> xmlObjectInterfaceClass) {
        List<X> filteredList = new ArrayList<X>();
        for (Object o : jaxbChildrenList) {
            if (o instanceof JAXBElement<?>) {
                JAXBElement<?> je = (JAXBElement<?>) o;
                if ((!filterQName.equals(ANY_QNAME)) && (!filterQName.equals(je.getName())))
                    continue;
                o = je.getValue();
            }
            if (!filterClass.isInstance(o))
                continue;

            filteredList.add(getXmlContext().getXmlObjectFactory().wrap(
                    ModelObjectFactory.createModelObject(getModelObject(), o),
                    xmlObjectInterfaceClass));
        }
        return filteredList.toArray((X[]) Array.newInstance(xmlObjectInterfaceClass, filteredList
                .size()));
    }

    protected final XmlObject[] createChildrenArray(List<?> jaxbChildrenList, Class<?> filterClass,
            QName filterQName) {
        List<XmlObject> filteredList = new ArrayList<XmlObject>();
        for (Object o : jaxbChildrenList) {
            if (o instanceof JAXBElement<?>) {
                JAXBElement<?> je = (JAXBElement<?>) o;
                if ((!filterQName.equals(ANY_QNAME)) && (!filterQName.equals(je.getName())))
                    continue;
                o = je.getValue();
            }
            if (!filterClass.isInstance(o))
                continue;
            XmlObjectNode xon = getXmlContext().getXmlObjectFactory().wrap(
                    filterClass.cast(ModelObjectFactory.createModelObject(getModelObject(), o)));
            if (!(xon instanceof XmlObject))
                continue;
            filteredList.add((XmlObject) xon);
        }
        return filteredList.toArray(new XmlObject[filteredList.size()]);
    }
    
    protected final void removeFromChildren(List<?> jaxbChildrenList, XmlObjectNode xmlObjectChild) {
        assert xmlObjectChild instanceof AbstractXmlObjectNodeImpl<?>;
        AbstractXmlObjectNodeImpl<?> axo = (AbstractXmlObjectNodeImpl<?>) xmlObjectChild;
        Iterator<?> it = jaxbChildrenList.iterator();
        while (it.hasNext()) {
            Object o = it.next();
            if (o instanceof JAXBElement<?>) {
                JAXBElement<?> je = (JAXBElement<?>) o;
                o = je.getValue();
            }
            if ((axo.getModelObject() instanceof AbstractJaxbModelObject)
                    && (!axo.getModelObject().equals(o)))
                continue;
            if ((axo.getModelObject() instanceof DomModelObject)
                    && (!((DomModelObject) axo.getModelObject()).getDOMNode().equals(o)))
                continue;
            if ((axo.getModelObject() instanceof TextModelObject)
                    && (!axo.getModelObject().equals(o))
                    && (!((TextModelObject) axo.getModelObject()).getValue().equals(o)))
                continue;
            it.remove();
            axo.setNaturalParent(null);
            break;
        }
    }

    @Override
    public Node getXmlObjectDOMNode() {
        return getModelObject().getBinder().getXMLNode(getModelObject());
    }

    @Override
    public final String toString() {
        return super.toString();
    }

}
