package com.ebmwebsourcing.easybox.impl;

import java.util.Deque;
import java.util.LinkedList;
import java.util.Map;

import javax.xml.namespace.QName;

import net.sf.saxon.Configuration;
import net.sf.saxon.om.Axis;
import net.sf.saxon.om.AxisIterator;
import net.sf.saxon.om.EmptyIterator;
import net.sf.saxon.om.NodeArrayIterator;
import net.sf.saxon.om.NodeInfo;
import net.sf.saxon.om.ReverseNodeArrayIterator;
import net.sf.saxon.om.SingleNodeIterator;
import net.sf.saxon.pattern.NodeTest;

import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectAttribute;
import com.ebmwebsourcing.easybox.api.XmlObjectNode;
import com.ebmwebsourcing.easybox.api.XmlObjectText;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;

final class NodeInfoHelper {

    public static final NodeArrayIterator EMPTY_AXIS_ITERATOR = new NodeArrayIterator(
            new NodeInfo[0]);

    
    static NodeInfo wrapXmlObjectAttributeAsNodeInfo(XmlObjectAttributeImpl xmlObjectAttribute,
            Configuration configuration) {
        assert xmlObjectAttribute != null;
        return new XmlObjectAttributeNodeInfo(xmlObjectAttribute, configuration);
    }

    
    static NodeInfo wrapXmlObjectAttributeAsNodeInfo(XmlObject xmlObject,
            QName key, Object value, Configuration configuration) {
        assert xmlObject != null;
        assert key != null;
        assert value != null;
        XmlObjectAttributeImpl xoa = new XmlObjectAttributeImpl(
                xmlObject.getXmlContext(), 
                new AttributeModelObject(null, key, value)); // TODO : attributes should be included in children!
        return wrapXmlObjectAttributeAsNodeInfo(xoa, configuration);
    }


    static XmlObjectNodeNodeInfo wrapXmlObjectAsNodeInfo(XmlObjectNode xmlObjectNode,
            Configuration configuration) {
        assert xmlObjectNode != null;
        if (xmlObjectNode instanceof XmlObject) {
            return new XmlObjectNodeInfo((XmlObject) xmlObjectNode, configuration);    
        } else if (xmlObjectNode instanceof XmlObjectAttribute) {
            return new XmlObjectAttributeNodeInfo((XmlObjectAttribute) xmlObjectNode, configuration);    
        } else if (xmlObjectNode instanceof XmlObjectText) {
            return new XmlObjectTextNodeInfo((XmlObjectText) xmlObjectNode, configuration);    
        } else {
            throw new UncheckedException("Unknown XmlObjectNode kind.");
        }
    }


    
    static Deque<NodeInfo> wrapXmlObjectsAsNodeInfos(XmlObjectNode[] xmlObjectNodes,
    		NodeTest nodeTest,
            Configuration configuration) {
        Deque<NodeInfo> nodeInfos = new LinkedList<NodeInfo>();
        for (int i = 0; i < xmlObjectNodes.length; ++i) {
            NodeInfo nodeInfo = null;
            if (xmlObjectNodes[i] instanceof XmlObject) {
                nodeInfo = wrapXmlObjectAsNodeInfo((XmlObject) xmlObjectNodes[i], configuration); 
            } else if (xmlObjectNodes[i] instanceof XmlObjectTextImpl) {
                nodeInfo = wrapXmlObjectTextAsNodeInfo((XmlObjectTextImpl) xmlObjectNodes[i], configuration);
            } else if (xmlObjectNodes[i] instanceof XmlObjectAttributeImpl) {
                nodeInfo = wrapXmlObjectAttributeAsNodeInfo((XmlObjectAttributeImpl) xmlObjectNodes[i], configuration);
            }
            if (nodeTest.matches(nodeInfo)) {
                nodeInfos.add(nodeInfo);
            }
        }
        return nodeInfos;
    }

    
    private static NodeInfo wrapXmlObjectTextAsNodeInfo(
			XmlObjectTextImpl xmlObjectText, Configuration configuration) {
    	assert xmlObjectText != null;
		return new XmlObjectTextNodeInfo(xmlObjectText, configuration);
	}


	static Deque<NodeInfo> wrapXmlObjectAttributesAsNodeInfos(
    		XmlObject xmlObject, Map<QName, Object> xmlObjectAttributes,
            Configuration configuration, NodeTest nodeTest) {
        Deque<NodeInfo> nodeInfos = new LinkedList<NodeInfo>();
        for (Map.Entry<QName, Object> entry : xmlObjectAttributes.entrySet()) {
            NodeInfo xoani = wrapXmlObjectAttributeAsNodeInfo(xmlObject, entry
                    .getKey(), entry.getValue(), configuration);
            if (nodeTest.matches(xoani)) {
                nodeInfos.add(xoani);
            }
        }
        return nodeInfos;
    }

    
    static AxisIterator createAncestorAxisIterator(XmlObjectNode xmlObjectNode, NodeTest nodeTest,
    		Configuration configuration) {
        Deque<NodeInfo> ancestors = wrapXmlObjectsAsNodeInfos(
                xmlObjectNode.getXmlObjectAncestors(), nodeTest, configuration);
        return new ReverseNodeArrayIterator(ancestors.toArray(new NodeInfo[ancestors.size()]), 0, ancestors.size());        
    }
    
    static AxisIterator createAncestorOrSelfAxisIterator(XmlObjectNode xmlObjectNode, NodeTest nodeTest,
    		Configuration configuration) {
        Deque<NodeInfo> ancestorsOrSelf = wrapXmlObjectsAsNodeInfos(
                xmlObjectNode.getXmlObjectAncestors(), nodeTest, configuration);
        ancestorsOrSelf.add(wrapXmlObjectAsNodeInfo(xmlObjectNode, configuration));
        return new ReverseNodeArrayIterator(ancestorsOrSelf.toArray(new NodeInfo[ancestorsOrSelf.size()]), 0, ancestorsOrSelf.size());
    }
    
    
    static AxisIterator createAttributeIterator(XmlObjectNode xmlObjectNode, 
    		NodeTest nodeTest, Configuration configuration) {
        if (!(xmlObjectNode instanceof XmlObject)) {
            return EMPTY_AXIS_ITERATOR;
        }
        Deque<NodeInfo> attributes = wrapXmlObjectAttributesAsNodeInfos((XmlObject) xmlObjectNode,
                ((XmlObject) xmlObjectNode).getXmlObjectAttributes(), configuration, nodeTest);
        return new NodeArrayIterator(attributes.toArray(new NodeInfo[attributes.size()]));
    }
    
    static AxisIterator createChildIterator(XmlObjectNode xmlObjectNode, NodeTest nodeTest,
    		Configuration configuration) {
        Deque<NodeInfo> children = wrapXmlObjectsAsNodeInfos(xmlObjectNode.getXmlObjectChildren(), 
        		nodeTest, configuration);
        return new NodeArrayIterator(children.toArray(new NodeInfo[children.size()]));
    }
    
    static AxisIterator createDescendantIterator(XmlObjectNode xmlObjectNode, NodeTest nodeTest, 
    		Configuration configuration) {
        Deque<NodeInfo> descendants = wrapXmlObjectsAsNodeInfos(
        		xmlObjectNode.getXmlObjectDescendants(), nodeTest, configuration);
        return new NodeArrayIterator(descendants.toArray(new NodeInfo[descendants.size()]));
    }
    
    static AxisIterator createDescendantOrSelfIterator(XmlObjectNode xmlObjectNode, 
    		NodeTest nodeTest, Configuration configuration) {
        Deque<NodeInfo> descendantsOrSelf = wrapXmlObjectsAsNodeInfos(
        		xmlObjectNode.getXmlObjectDescendants(), nodeTest, configuration);
        descendantsOrSelf.add(wrapXmlObjectAsNodeInfo(xmlObjectNode, configuration));
        return new NodeArrayIterator(descendantsOrSelf.toArray(new NodeInfo[descendantsOrSelf.size()]));        
    }
    
    static AxisIterator createFollowingIterator(XmlObjectNode xmlObjectNode, NodeTest nodeTest, 
    		Configuration configuration) {
        Deque<NodeInfo> following = wrapXmlObjectsAsNodeInfos(
        		xmlObjectNode.getXmlObjectFollowing(), nodeTest, configuration);
        return new NodeArrayIterator(following.toArray(new NodeInfo[following.size()]));        
    }
    

    static AxisIterator createFollowingSiblingAxisIterator(XmlObjectNode xmlObjectNode, 
    		NodeTest nodeTest, Configuration configuration) {
        Deque<NodeInfo> followingSiblings = wrapXmlObjectsAsNodeInfos(
        		xmlObjectNode.getXmlObjectFollowingSiblings(), nodeTest, configuration);
        return new NodeArrayIterator(followingSiblings.toArray(new NodeInfo[followingSiblings.size()]));        
    }

    static AxisIterator createEmptyAxisIterator() {
        return EmptyIterator.getInstance();
    }
    
    static AxisIterator createSingletonAxisIterator(NodeInfo nodeInfo) {
        return SingleNodeIterator.makeIterator(nodeInfo);
    }
    
    static AxisIterator createParentAxisIterator(XmlObjectNode xmlObjectNode, 
    		NodeTest nodeTest, Configuration configuration) {
        XmlObject parent = xmlObjectNode.getXmlObjectParent();
        if (parent == null) return createEmptyAxisIterator();
        XmlObjectNodeNodeInfo parentNodeInfo = wrapXmlObjectAsNodeInfo(parent, configuration);
        if (!nodeTest.matches(parentNodeInfo)) return createEmptyAxisIterator();
        return SingleNodeIterator.makeIterator(parentNodeInfo);
    }
     
    static AxisIterator createPrecedingAxisIterator(XmlObjectNode xmlObjectNode, 
    		NodeTest nodeTest, Configuration configuration) {
        Deque<NodeInfo> preceding = wrapXmlObjectsAsNodeInfos(
        		xmlObjectNode.getXmlObjectPreceding(), nodeTest, configuration);
        return new ReverseNodeArrayIterator(preceding.toArray(new NodeInfo[preceding.size()]), 0, preceding.size());        
    }
     
    static AxisIterator createPrecedingOrAncestorAxisIterator(XmlObjectNode xmlObjectNode, 
    		NodeTest nodeTest, Configuration configuration) {
        Deque<NodeInfo> precedingOrAncestor = wrapXmlObjectsAsNodeInfos(
        		xmlObjectNode.getXmlObjectPrecedingOrAncestor(), nodeTest, configuration);
        return new NodeArrayIterator(precedingOrAncestor.toArray(new NodeInfo[precedingOrAncestor.size()]));        
    }
     
    static AxisIterator createPrecedingSiblingAxisIterator(XmlObjectNode xmlObjectNode, 
    		NodeTest nodeTest, Configuration configuration) {
        Deque<NodeInfo> precedingSibling = wrapXmlObjectsAsNodeInfos(
        		xmlObjectNode.getXmlObjectPrecedingSiblings(), nodeTest, configuration);
        return new ReverseNodeArrayIterator(precedingSibling.toArray(new NodeInfo[precedingSibling.size()]), 0, 
        		precedingSibling.size());        
    }
     
    static AxisIterator createSelfAxisIterator(XmlObjectNode xmlObjectNode, 
    		NodeTest nodeTest, Configuration configuration) {
        return new NodeArrayIterator(new NodeInfo[] { wrapXmlObjectAsNodeInfo(xmlObjectNode, configuration) });
    }
     
    
    static AxisIterator createAxisIterator(byte axisNumber, XmlObjectNode xmlObjectNode, 
    		NodeTest nodeTest, Configuration configuration) {
        switch (axisNumber) {
        case Axis.ANCESTOR:
            return createAncestorAxisIterator(xmlObjectNode, nodeTest, configuration);
        case Axis.ANCESTOR_OR_SELF:
            return createAncestorOrSelfAxisIterator(xmlObjectNode, nodeTest, configuration);
        case Axis.ATTRIBUTE:
            return createAttributeIterator(xmlObjectNode, nodeTest, configuration);
        case Axis.CHILD:
            return createChildIterator(xmlObjectNode, nodeTest, configuration);
        case Axis.DESCENDANT:
            return createDescendantIterator(xmlObjectNode, nodeTest, configuration);
        case Axis.DESCENDANT_OR_SELF:
            return createDescendantOrSelfIterator(xmlObjectNode, nodeTest, configuration);
        case Axis.FOLLOWING:
            return createFollowingIterator(xmlObjectNode, nodeTest, configuration);
        case Axis.FOLLOWING_SIBLING:
            return createFollowingSiblingAxisIterator(xmlObjectNode, nodeTest, configuration);
        case Axis.NAMESPACE:
            throw new Error("Not implemented yet.");
        case Axis.PARENT:
            return createParentAxisIterator(xmlObjectNode, nodeTest, configuration);
        case Axis.PRECEDING:
            return createPrecedingAxisIterator(xmlObjectNode, nodeTest, configuration);
        case Axis.PRECEDING_OR_ANCESTOR:
            return createPrecedingOrAncestorAxisIterator(xmlObjectNode, nodeTest, configuration);
        case Axis.PRECEDING_SIBLING:
            return createPrecedingSiblingAxisIterator(xmlObjectNode, nodeTest, configuration);
        case Axis.SELF:
            return createSelfAxisIterator(xmlObjectNode, nodeTest, configuration);
        default:
            return null;
        }
        
    }

    
}
