/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easybox.impl;

import java.net.URI;
import java.util.Collections;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.XMLConstants;
import javax.xml.namespace.QName;

import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

import com.ebmwebsourcing.easybox.api.ClassMetadataConstants;
import com.ebmwebsourcing.easybox.api.ModelObject;
import com.ebmwebsourcing.easybox.api.XmlContext;
import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectNode;

public abstract class AbstractXmlObjectImpl<Model extends AbstractModelObject>
		extends AbstractXmlObjectNodeImpl<Model> implements XmlObject {

    private XmlObjectAdoptionMediator xmlObjectAdoptionMediator;
    
	protected AbstractXmlObjectImpl(XmlContext xmlContext, Model modelObject) {
	    super(xmlContext, modelObject);
        this.xmlObjectAdoptionMediator = null;
	}

	@SuppressWarnings("unchecked")
	@Override
	public final boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;

		AbstractXmlObjectImpl<Model> other = (AbstractXmlObjectImpl<Model>) obj;
		if (getModelObject() == null) {
			return other.getModelObject() == null;
		}
		return other.getModelObject().equals(getModelObject());
	}


	protected abstract Class<? extends Model> getCompliantModelClass();



	private void appendXmlObjectDescendants(XmlObjectNode xo,
			List<XmlObjectNode> result,
			IdentityHashMap<XmlObjectNode, Object> descendantsMap) {
		XmlObjectNode[] children = xo.getXmlObjectChildren();
		for (XmlObjectNode child : children) {
			if (descendantsMap.containsKey(child))
				continue;
			descendantsMap.put(child, null);
			result.add(child);
			appendXmlObjectDescendants(
					child, result, descendantsMap);
		}
	}

	
	@Override
	public XmlObjectNode[] getXmlObjectAdoptedChildren() {
		return XmlObjectNode.EMPTY_ARRAY;
	}

	
	@Override
	public String toString() {
		StringBuffer sb = new StringBuffer();
		QName qname = getXmlObjectQName();
		sb.append("\n+ ").append(qname);
		Map<QName, Object> attributes = getXmlObjectAttributes();
		StringBuffer pb = new StringBuffer();
		Iterator<Map.Entry<QName, Object>> itProperty = attributes.entrySet()
				.iterator();
		while (itProperty.hasNext()) {
			Map.Entry<QName, Object> property = itProperty.next();
			StringBuffer vb = new StringBuffer();
			pb.append("\n- ").append(property.getKey()).append(" = ");
			vb.append(String.valueOf(property.getValue()));
			pb.append(vb.toString().replaceAll("\n", "\n  "));
		}
		sb.append(pb.toString().replaceAll("\n", "\n  "));
		pb = new StringBuffer();
        // TODO : use descendants to prevent infinite recursion and to show adopted children.
		XmlObjectNode[] children = getXmlObjectNaturalChildren();
		for (Object child : children) {
			StringBuffer vb = new StringBuffer();
			vb.append(String.valueOf(child));
			pb.append(vb.toString().replaceAll("\n", "\n  "));
		}
		sb.append(pb.toString());
		return sb.toString();
	}


	@SuppressWarnings("unchecked")
	protected final void onNaturalParentChange(XmlObject parent) {
        ModelObject naturalParent = null; 
		if (parent != null) {
			naturalParent = ((AbstractXmlObjectImpl<Model>) parent).getModelObject();
		}
        getModelObject().setNaturalParent(naturalParent);
		adaptModelObjectIfNecessary();
		// reset parent after duplication
        getModelObject().setNaturalParent(naturalParent);
		
	}

    @SuppressWarnings("unchecked")
    protected final void onAdoptiveParentChange(XmlObject parent) {
        if (parent == null) {
            getModelObject().setAdoptiveParent(null);
        } else {
            getModelObject().setAdoptiveParent(
                    ((AbstractXmlObjectImpl<Model>) parent).getModelObject());
        }
        adaptModelObjectIfNecessary();
    }

	
	
	protected void adaptModelObjectIfNecessary() {
		// TODO : actually we could call this only when a marshalling is
		// requested on the object ;
		// this is the only moment when we *might* need to be really consistent
		// against jaxb model concrete
		// types.
		if (getModelObject() == null) {
			setModelObject(createCompliantModel());
		} else if (!getModelObject().getClass()
				.equals(getCompliantModelClass())) {
			// TODO !!!
			// System.err.println(String.format("/// compliant class = '%s' found class = '%s'",
			// getCompliantModelClass().getSimpleName(), jaxbModel.getClass()));
			// setModel(getJaxbTypeMorpher().morph(jaxbModel,
			// getCompliantModelClass()));
		}
	}


	@Override
	public final String getXmlObjectValue() {
	    return null;
	}
	
	
	@SuppressWarnings("unchecked")
    @Override
	public Map<QName, Object> getXmlObjectAttributes() {
	    return Collections.EMPTY_MAP;
	}
	

    public final XmlObject duplicateXmlObject() {
        @SuppressWarnings("unchecked")
        Model duplicateJaxbModel = (Model) getModelObject().duplicate();
        return getXmlContext().getXmlObjectFactory().wrap(duplicateJaxbModel,
                getClass());
    }


    @SuppressWarnings("unchecked")
    public final <X extends XmlObject> X duplicateXmlObjectAs(
            Class<X> targetInterfaceClass) {
        assert targetInterfaceClass != null;
        Class<? extends ModelObject> modelObjectClass = getXmlContext().getClassMetadata().get(getClass(), 
                ClassMetadataConstants.IMPLEMENTATION_CLASS_MODELOBJECT_CLASS);
        
        assert AbstractJaxbModelObject.class.isAssignableFrom(modelObjectClass);
        AbstractJaxbModelObject ajmo = getModelObject().duplicateAs(
                (Class<? extends AbstractJaxbModelObject>) modelObjectClass);
        return getXmlContext().getXmlObjectFactory().wrap(ajmo,
                targetInterfaceClass);
    }

    @Override
    public final URI getXmlObjectBaseURI() {
        return getModelObject().getBaseURI();
    }

    final void setXmlObjectBaseURI(URI baseURI) {
        getModelObject().setBaseURI(baseURI);
    }
    
    
    @Override
    public Map<String, String> getXmlObjectInScopeNamespaces() {
        // TODO : could be cached and flushed on parent change!
        Map<String, String> namespaces = new HashMap<String, String>();
        XmlObject current = this;
        while (true) {
            if (current == null) break;
            Node node = current.getXmlObjectDOMNode();
            assert node != null;
            NamedNodeMap nodeMap = node.getAttributes();
            for (int i = 0 ; i < nodeMap.getLength() ; ++i) {
                Node attrNode = nodeMap.item(i);
                if (XMLConstants.XMLNS_ATTRIBUTE.equals(attrNode.getPrefix())) {
                    if (namespaces.containsKey(attrNode.getLocalName())) continue;
                    namespaces.put(attrNode.getLocalName(), attrNode.getNodeValue());
                }
            }
            current = current.getXmlObjectParent();
        }
        return namespaces;
    }
    


    private final XmlObject createAdoptableChild(XmlObject parentAdopter, String publicId, String systemId) 
        {
        return doCreateAdoptableChild(parentAdopter, publicId, systemId, new IdentityHashMap<XmlObject, Object>());
    }
    
    
    protected XmlObject doCreateAdoptableChild(XmlObject parentAdopter, 
            String publicId, String systemId, IdentityHashMap<XmlObject, Object> alreadyProcessed) 
        {
        if (xmlObjectAdoptionMediator == null) {
            XmlObject parent = getXmlObjectParent();
            if ((parent instanceof AbstractJaxbXmlObjectImpl<?>) && !alreadyProcessed.containsKey(parent)) {
                // TODO createadoptablechild sould not be at this level ; this instanceof should not stay there.
                alreadyProcessed.put(parent, null);
                return ((AbstractJaxbXmlObjectImpl<?>) parent).doCreateAdoptableChild(parentAdopter, publicId, systemId, alreadyProcessed);
            } else {
                xmlObjectAdoptionMediator = new XmlObjectAdoptionMediator(parentAdopter);
            }
        }
        return xmlObjectAdoptionMediator.adoptChild(parentAdopter, publicId, systemId, null);
    }

    
    
    protected final XmlObject adoptChild(String publicId, String systemId,
            int adoptionIndex) {
        AbstractJaxbXmlObjectImpl<?> adoptedChild = (AbstractJaxbXmlObjectImpl<?>) createAdoptableChild(this, publicId, systemId);
        adoptedChild.setAdoptiveParent(this, adoptionIndex);
        return adoptedChild;
    }
    
    
    protected final XmlObject adoptChild(String systemId, int adoptionIndex) {
        return adoptChild(null, systemId, adoptionIndex);
    }    
}
