/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/
 
package com.ebmwebsourcing.easybpel.model.bpel.impl;

import java.util.Map;
import java.util.logging.Logger;

import org.oasisopen.sca.annotation.PolicySets;
import org.oasisopen.sca.annotation.Scope;

import com.ebmwebsourcing.easybpel.extended.service.activity.impl.ExtendedActivityServiceImpl;
import com.ebmwebsourcing.easybpel.model.bpel.api.BPELException;
import com.ebmwebsourcing.easybpel.model.bpel.api.BPELFactory;
import com.ebmwebsourcing.easybpel.model.bpel.api.inout.BPELReader;
import com.ebmwebsourcing.easybpel.model.bpel.api.inout.BPELReader.FeatureConstants;
import com.ebmwebsourcing.easybpel.model.bpel.api.inout.BPELWriter;
import com.ebmwebsourcing.easybpel.model.bpel.impl.compiler.BPELCompilerImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.compiler.validation.BPELStaticAnalysisImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.compiler.validation.BPELStaticAnalysisPrinter;
import com.ebmwebsourcing.easybpel.model.bpel.impl.inout.BPELReaderImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.inout.BPELWriterImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.registry.BPELMemoryRegistryImpl;
import com.ebmwebsourcing.easyviper.core.api.Core;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Engine;
import com.ebmwebsourcing.easyviper.core.api.engine.configuration.ConfigurationEngine;
import com.ebmwebsourcing.easyviper.core.api.env.ExternalEnvironment;
import com.ebmwebsourcing.easyviper.core.api.env.Receiver;
import com.ebmwebsourcing.easyviper.core.api.env.Sender;
import com.ebmwebsourcing.easyviper.core.api.factory.Factory;
import com.ebmwebsourcing.easyviper.core.api.model.Model;
import com.ebmwebsourcing.easyviper.core.api.model.compiler.Compiler;
import com.ebmwebsourcing.easyviper.core.api.model.compiler.validation.StaticAnalysis;
import com.ebmwebsourcing.easyviper.core.api.model.registry.definition.ProcessDefinitionRegistry;
import com.ebmwebsourcing.easyviper.core.impl.CoreImpl;
import com.ebmwebsourcing.easyviper.core.impl.engine.configuration.ConfigurationEngineImpl;
import com.ebmwebsourcing.easyviper.core.impl.model.ModelImpl;

/**
 * This class is a concrete implementation of the abstract class BPELFactory.
 * Some ideas used here have been shamelessly copied from the wonderful JAXP and
 * Xerces work.
 * 
 * @author Nicolas Salatge - EBM WebSourcing
 */

@Scope("COMPOSITE")
@org.oasisopen.sca.annotation.Service(value = Factory.class, names = "service")
@PolicySets("frascati:scaEasyPrimitive")
public class BPELFactoryImpl extends BPELFactory implements Factory {

    private static BPELFactory instance = null;

    private BPELFactoryImpl() {

    }

    public static BPELFactory getInstance() {
        if (instance == null) {
            instance = new BPELFactoryImpl();
        }
        return instance;
    }

    /**
     * Create a new instance of a BPELReaderImpl.
     * 
     * @throws BPELException
     */
    @Override
    public BPELReader newBPELReader() throws BPELException {
        final BPELReader reader = new BPELReaderImpl();
        return reader;
    }

    /**
     * Create a new instance of a BPELReaderImpl.
     * 
     * @throws BPELException
     */
    @Override
    public BPELReader newBPELReader(final Map<FeatureConstants, Object> features)
            throws BPELException {
        final BPELReader reader = this.newBPELReader();
        ((BPELReaderImpl) reader).setFeatures(features);
        return reader;

    }

    /**
     * Create a new instance of a BPELWriterImpl.
     * 
     * @throws BPELException
     */
    @Override
    public BPELWriter newBPELWriter() throws BPELException {
        final BPELWriter writer = new BPELWriterImpl();
        return writer;
    }


    @Override
    public Core newBPELEngine(final ConfigurationEngine conf, final int numberOfReceivers,
            final Class<? extends Receiver> receiverClass, final int numberOfSenders,
            final Class<? extends Sender> senderClass, final Logger logger) throws BPELException {
        Core core;
        try {
            core = this.createCore((ConfigurationEngineImpl) conf, numberOfReceivers,
                    receiverClass, numberOfSenders, senderClass, logger);
            core.getModel().getRegistry()
                    .setStaticAnalysis((StaticAnalysis) BPELStaticAnalysisImpl.getInstance());
            core.getModel()
                    .getRegistry()
                    .setStaticAnalysisPrinter(
                            (com.ebmwebsourcing.easyviper.core.api.model.compiler.validation.StaticAnalysisPrinter) BPELStaticAnalysisPrinter
                                    .getInstance());
        } catch (final CoreException e) {
            throw new BPELException(e);
        }
        return core;
    }

    public Core createCore(final int numberOfReceivers,
            final Class<? extends Receiver> receiverClass, final int numberOfSenders,
            final Class<? extends Sender> senderClass, final Logger logger) throws CoreException {
        return this.createCore(new ConfigurationEngineImpl(), numberOfReceivers, receiverClass,
                numberOfSenders, senderClass, logger);
    }

    public Core createCore(final ConfigurationEngine conf, final int numberOfReceivers,
            final Class<? extends Receiver> receiverClass, final int numberOfSenders,
            final Class<? extends Sender> senderClass, final Logger logger) throws CoreException {

        // create core
        final Core core = super.createCore(BPELFactory.DEFAULT_NAME, conf, CoreImpl.class);

        // set the logger
        if (logger != null) {
            core.setLog(logger);
        }

        // create worflow engine
        final Engine engine = core.createEngine(conf);

        // add extended service manager if not added
        if (core.getEngine().getServiceManager().getService(ExtendedActivityServiceImpl.class) == null) {
            core.getEngine().getServiceManager().addService(ExtendedActivityServiceImpl.class);
        }

        // create model
        final Model model = core.createModel("BPELModel", ModelImpl.class);

        // create compiler
        final Compiler compiler = model.createCompiler("BPELCompiler", BPELCompilerImpl.class);

        // create registry
        final ProcessDefinitionRegistry registry = model.createRegistry("BPELRegistry",
                BPELMemoryRegistryImpl.class);

        // create externalEnvironment
        final ExternalEnvironment externalEnvironment = core
                .createExternalEnvironment("BPELEnvironment");

        // add receivers
        for (int i = 0; i < numberOfReceivers; i++) {
            externalEnvironment.addReceiver(receiverClass);
        }

        // add senders
        for (int i = 0; i < numberOfSenders; i++) {
            externalEnvironment.addSender(senderClass);
        }

        return core;
    }

}
