/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/
 
package com.ebmwebsourcing.easybpel.model.bpel.impl.compiler.verification;

import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Logger;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import javax.xml.namespace.QName;

import org.jdom.Element;
import org.jdom.Namespace;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.ow2.easywsdl.schema.SchemaFactory;

import com.ebmwebsourcing.easybpel.model.bpel.api.BPELException;
import com.ebmwebsourcing.easybpel.model.bpel.api.BPELProcess;
import com.ebmwebsourcing.easybpel.model.bpel.api.compiler.verification.TypeVerifier;
import com.ebmwebsourcing.easycommons.xml.QNameHelper;


/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
public class TypeVerifierImpl implements TypeVerifier {

	private static Logger log = Logger.getLogger(TypeVerifierImpl.class.getName());


	private final BPELProcess bpelDefinition;



	public TypeVerifierImpl(final BPELProcess bpelDefinition) {
		this.bpelDefinition = bpelDefinition;
	}


	public Class<?> compare(final Element e1, final Element e2) throws BPELException {
		Class<?> res = null;
		if((this.isInteger(e1)&&(this.isInteger(e2)))) {
			res = Integer.class;
		} else if((this.isBoolean(e1)&&(this.isBoolean(e2)))) {
			res = Boolean.class;
		} else if((this.isDateTime(e1)&&(this.isDateTime(e2)))) {
			res = Date.class;
		} else if((this.isString(e1)&&(this.isString(e2)))) {
			res = String.class;
		} 
		return res;
	}

	public boolean isInteger(final Element e) throws BPELException {
		boolean res = false;
		String typeValue = e.getAttributeValue("type");
		if(typeValue == null) {
			typeValue = e.getAttributeValue("type", Namespace.getNamespace("http://www.w3.org/2001/XMLSchema-instance"));
		}

		QName type = null;
		if(typeValue != null) {
			final String prefix = QNameHelper.getPrefix(typeValue);
			type = new QName(typeValue);
			if(prefix != null) {
				final Namespace ns = e.getNamespace(prefix);
				if(ns != null) {
					type = new QName(ns.getURI(), QNameHelper.getLocalPartWithoutPrefix(typeValue));
				} 
			}

		} else {
			// find type in all wsdl using name of element
			final List<com.ebmwebsourcing.easyschema10.api.element.Element> elmts = 
			    this.bpelDefinition.getImports().findElementsInAllSchema(new QName(e.getNamespaceURI(),e.getName()));
			if(elmts.size() == 0) {
				if((e.getNamespaceURI() == null) || e.getNamespaceURI().equals("")) {
					log.finest("Element not found in schema. It's probably an unqualified element");
				} else {
					throw new BPELException("Element " + e + " not found in all schemas");
				}
			} 
			int typeTest = -1;
			if(elmts.size() > 1) {
				final Iterator<com.ebmwebsourcing.easyschema10.api.element.Element> it = elmts.iterator();
				com.ebmwebsourcing.easyschema10.api.element.Element elmt1 = it.next();
				com.ebmwebsourcing.easyschema10.api.element.Element elmt2 = null;
				// optimistic approach
				int id = 0;
				while(it.hasNext() && (typeTest==-1)) {
					elmt2 = it.next();
					if((elmt1.getType() != null) && elmt1.getType().equals(elmt2.getType())) {
						typeTest = id; 
					}
					elmt1 = elmt2;
					id++;
				}
			}

			if(typeTest >= 0) {
				type = elmts.get(typeTest).getType();
			} else {
				type = SchemaFactory.getDefaultSchema().getTypeString().getQName();
			}
		}

		if(type.equals(SchemaFactory.getDefaultSchema().getTypeInt().getQName()) || type.equals(SchemaFactory.getDefaultSchema().getTypeInteger().getQName())) {
			// analyse type
			log.finest("type: " + new XMLOutputter(Format.getPrettyFormat()).outputString(e) + " is an integer");
			res = true;
		} 
		return res;
	}

	public boolean isDateTime(final Element e) throws BPELException {
		boolean res = false;
		String typeValue = e.getAttributeValue("type");
		if(typeValue == null) {
			typeValue = e.getAttributeValue("type", Namespace.getNamespace("http://www.w3.org/2001/XMLSchema-instance"));
		}

		QName type = null;
		if(typeValue != null) {
			final String prefix = QNameHelper.getPrefix(typeValue);
			type = new QName(typeValue);
			if(prefix != null) {
				final Namespace ns = e.getNamespace(prefix);
				if(ns != null) {
					type = new QName(ns.getURI(), QNameHelper.getLocalPartWithoutPrefix(typeValue));
				} 
			}

		} else {
			// find type in all wsdl using name of element
			final List<com.ebmwebsourcing.easyschema10.api.element.Element> elmts = this.bpelDefinition.getImports().findElementsInAllSchema(new QName(e.getNamespaceURI(),e.getName()));
			if(elmts.size() == 0) {
				if((e.getNamespaceURI() == null) || e.getNamespaceURI().equals("")) {
					log.finest("Element not found in schema. It's probably an unqualified element");
				} else {
					throw new BPELException("Element " + e + " not found in all schemas");
				}
			} 

			int typeTest = -1;
			if(elmts.size() > 1) {
				final Iterator<com.ebmwebsourcing.easyschema10.api.element.Element> it = elmts.iterator();
				com.ebmwebsourcing.easyschema10.api.element.Element elmt1 = it.next();
				com.ebmwebsourcing.easyschema10.api.element.Element elmt2 = null;

				// optimistic approach
				int id = 0;
				while(it.hasNext() && (typeTest==-1)) {
					elmt2 = it.next();
					if((elmt1.getType() != null) && elmt1.getType().equals(elmt2.getType())) {
						typeTest = id; 
					}
					elmt1 = elmt2;
					id++;
				}

			}

			if(typeTest >= 0) {
				type = elmts.get(typeTest).getType();
			} else {
				type = SchemaFactory.getDefaultSchema().getTypeString().getQName();
			}
		}

		if(type.equals(SchemaFactory.getDefaultSchema().getTypeDateTime().getQName())) {
			// analyse type
			log.finest("type: " + new XMLOutputter(Format.getPrettyFormat()).outputString(e) + " is an datetime");
			res = true;
		} 
		return res;
	}

	public boolean isString(final Element e) throws BPELException {
		boolean res = false;
		String typeValue = e.getAttributeValue("type");
		if(typeValue == null) {
			typeValue = e.getAttributeValue("type", Namespace.getNamespace("http://www.w3.org/2001/XMLSchema-instance"));
		}

		QName type = null;
		if(typeValue != null) {
			final String prefix = QNameHelper.getPrefix(typeValue);
			type = new QName(typeValue);
			if(prefix != null) {
				final Namespace ns = e.getNamespace(prefix);
				if(ns != null) {
					type = new QName(ns.getURI(), QNameHelper.getLocalPartWithoutPrefix(typeValue));
				} 
			}

		} else {
			// find type in all wsdl using name of element
			final List<com.ebmwebsourcing.easyschema10.api.element.Element> elmts = 
			    this.bpelDefinition.getImports().findElementsInAllSchema(new QName(e.getNamespaceURI(),e.getName()));
			if(elmts.size() == 0) {
				if((e.getNamespaceURI() == null) || e.getNamespaceURI().equals("")) {
					log.finest("Element not found in schema. It's probably an unqualified element");
				} else {
					throw new BPELException("Element " + e + " not found in all schemas");
				}
			}

			int typeTest = -1;
			if(elmts.size() > 1) {
				final Iterator<com.ebmwebsourcing.easyschema10.api.element.Element> it = elmts.iterator();
				com.ebmwebsourcing.easyschema10.api.element.Element elmt1 = it.next();
				com.ebmwebsourcing.easyschema10.api.element.Element elmt2 = null;
				// optimistic approach
				int id = 0;
				while(it.hasNext() && (typeTest==-1)) {
					elmt2 = it.next();
					if((elmt1.getType() != null) && elmt1.getType().equals(elmt2.getType())) {
						typeTest = id; 
					}
					elmt1 = elmt2;
				}
				id++;
			}

			if(typeTest >= 0) {
				type = elmts.get(typeTest).getType();
			} else {
				type = SchemaFactory.getDefaultSchema().getTypeString().getQName();
			}
		}

		if(type.equals(SchemaFactory.getDefaultSchema().getTypeString().getQName())) {
			// analyse type
			log.finest("type: " + new XMLOutputter(Format.getPrettyFormat()).outputString(e) + " is a string");
			res = true;
		} 
		return res;
	}

	public boolean isBoolean(final Element e) throws BPELException {
		boolean res = false;
		String typeValue = e.getAttributeValue("type");
		if(typeValue == null) {
			typeValue = e.getAttributeValue("type", Namespace.getNamespace("http://www.w3.org/2001/XMLSchema-instance"));
		}

		QName type = null;
		if(typeValue != null) {
			final String prefix = QNameHelper.getPrefix(typeValue);
			type = new QName(typeValue);
			if(prefix != null) {
				final Namespace ns = e.getNamespace(prefix);
				if(ns != null) {
					type = new QName(ns.getURI(), QNameHelper.getLocalPartWithoutPrefix(typeValue));
				} 
			}
		} else {
			// find type in all wsdl using name of element
			final List<com.ebmwebsourcing.easyschema10.api.element.Element> elmts = 
			    this.bpelDefinition.getImports().findElementsInAllSchema(new QName(e.getNamespaceURI(),e.getName()));
			if(elmts.size() == 0) {
				if((e.getNamespaceURI() == null) || e.getNamespaceURI().equals("")) {
					log.finest("Element not found in schema. It's probably an unqualified element");
				} else {
					throw new BPELException("Element " + e + " not found in all schemas");
				}
			}

			int typeTest = -1;
			if(elmts.size() > 1) {
				final Iterator<com.ebmwebsourcing.easyschema10.api.element.Element> it = elmts.iterator();
				com.ebmwebsourcing.easyschema10.api.element.Element elmt1 = it.next();
				com.ebmwebsourcing.easyschema10.api.element.Element elmt2 = null;
				// optimistic approach
				int id = 0;
				while(it.hasNext() && (typeTest==-1)) {
					elmt2 = it.next();
					if((elmt1.getType() != null) && elmt1.getType().equals(elmt2.getType())) {
						typeTest = id; 
					}
					elmt1 = elmt2;
				}

				id++;
			}

			if(typeTest >= 0) {
				type = elmts.get(typeTest).getType();
			} else {
				type = SchemaFactory.getDefaultSchema().getTypeString().getQName();
			}
		}

		if((type != null)&&(type.equals(SchemaFactory.getDefaultSchema().getTypeString().getQName()))) {
			if((e.getText() != null) && (e.getText().trim().equals("true") || e.getText().trim().equals("false"))) {
				// analyse type
				log.finest("type: " + new XMLOutputter(Format.getPrettyFormat()).outputString(e) + " is a boolean");
				res = true;
			}
		} else if((type != null)&&(type.equals(SchemaFactory.getDefaultSchema().getTypeBoolean().getQName()))) {
			// analyse type
			log.finest("type: " + new XMLOutputter(Format.getPrettyFormat()).outputString(e) + " is a boolean");
			res = true;
		} 

		return res;
	}


	public Integer getIntValue(final Element e) {
		Integer res = null;
		if((e.getText() != null)&&(e.getText().trim().length() > 0)) {
			res = Integer.parseInt(e.getText());
		}
		return res;
	}


	public String getStringValue(final Element e) {
		return e.getText();
	}


	public Boolean getBooleanValue(final Element e) {
		Boolean res = null;
		if((e.getText() != null)&&(e.getText().trim().length() > 0)) {
			res = Boolean.parseBoolean(e.getText());
		}
		return res;
	}

	public Date getDateTimeValue(final Element e) throws BPELException {
		Date res = null;
		if((e.getText() != null)&&(e.getText().trim().length() > 0)) {
			try {
				final XMLGregorianCalendar xmlCalendar = DatatypeFactory.newInstance().newXMLGregorianCalendar(e.getText());
				res = xmlCalendar.toGregorianCalendar().getTime();
			} catch (final DatatypeConfigurationException e1) {
				throw new BPELException("Impossible to convert this date: " + e.getText() + " in element: " + e.getName());
			} catch (final IllegalArgumentException e1) {
				throw new BPELException("Impossible to convert this date: " + e.getText() + " in element: " + e.getName());
			}
		}
		return res;
	}

	public boolean isNullValue(final Element e) {
		boolean res = false;
		if((e != null)&&(e.getText() == null)) {
			res = true;
		}
		return res;
	}

}
