/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.logger.handler;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;
import java.util.logging.ErrorManager;
import java.util.logging.LogManager;
import java.util.logging.LogRecord;
import java.util.logging.StreamHandler;

import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.easycommons.properties.PropertiesException;
import com.ebmwebsourcing.easycommons.properties.PropertiesHelper;
import com.ebmwebsourcing.easycommons.thread.ExecutionContext;

/**
 * Contextual file logging <tt>Handler</tt>.
 * <p>
 * The <tt>ContextualFileHandler</tt> writes the log records in different log
 * files depending on the contextual information.
 * <p>
 * By default the <tt>SimpleFormatter</tt> class is used for formatting.
 * <p>
 * <b>Configuration:</b> By default each <tt>ContextualFileHandler</tt> is
 * initialized using the following <tt>LogManager</tt> configuration properties.
 * If properties are not defined (or have invalid values) then the specified
 * default values are used.
 * <ul>
 * <li>com.ebmwebsourcing.easycommons.logger.handler.ContextualFileHandler.level
 * specifies the default level for the <tt>Handler</tt> (defaults to
 * <tt>Level.ALL</tt>).</li>
 * <li>com.ebmwebsourcing.easycommons.logger.handler.ContextualFileHandler.
 * formatter specifies the name of a <tt>Formatter</tt> class to use (defaults
 * to <tt>java.util.logging.SimpleFormatter</tt>)</li>
 * <li>
 * com.ebmwebsourcing.easycommons.logger.handler.ContextualFileHandler.basedir
 * specifies the path of the base directory for all the log files (defaults to
 * the working directory)</li>
 * <li>
 * com.ebmwebsourcing.easycommons.logger.handler.ContextualFileHandler.subdir
 * specifies the relative path from the base directory to a relative
 * subdirectory containing log file, it may contain placeholders resolved
 * against the current contextual information</li>
 * <li>com.ebmwebsourcing.easycommons.logger.handler.ContextualFileHandler.
 * logFilename specifies the simple filename (without any path information)
 * containing logs, it may contain placeholders resolved against the current
 * contextual information (defaults to the file "default.log")</li>
 * </ul>
 * <p>
 */
public class ContextualFileHandler extends StreamHandler {

    private static final String PROPERTY_NAME_PREFIX = ContextualFileHandler.class.getName();

    private static final String BASEDIR_PROPERTY_NAME = PROPERTY_NAME_PREFIX + ".basedir";

    private static final String SUBDIR_PROPERTY_NAME = PROPERTY_NAME_PREFIX + ".subdir";

    private static final String LOGFILENAME_PROPERTY_NAME = PROPERTY_NAME_PREFIX + ".logFilename";

    static final String DEFAULT_LOG_FILE_NAME = "default.log";

    private final String basedir;

    private final String subdir;

    private final String logFilename;

    protected ContextualFileHandler(String basedir, String subdir, String logFilename,
            Properties systemProperties) {
        super();

        assert basedir != null;
        assert subdir != null;
        assert logFilename != null;
        assert systemProperties != null;

        if (logFilename.isEmpty()) {
            throw new UncheckedException("The log filename cannot be empty.");
        }
        if (logFilename.contains(File.separator)) {
            throw new UncheckedException("The log filename cannot contain a file separator.");
        }

        try {
            this.basedir = PropertiesHelper.resolveString(basedir, systemProperties);
        } catch (PropertiesException pe) {
            throw new UncheckedException(pe);
        }
        this.subdir = subdir;
        this.logFilename = logFilename;
    }

    /**
     * Construct a default <tt>ContextualFileHandler</tt>. This will be
     * configured entirely from <tt>LogManager</tt> properties (or their default
     * values).
     * <p>
     * 
     * @throws IOException
     *             If the base directory is not set and a the temporary
     *             directory could not be created to write the logs
     * @throws PropertiesException
     *             If an error occurs when resolving a place holder in the base
     *             directory properties
     */
    public ContextualFileHandler() throws IOException, PropertiesException {
        this(getStringProperty(BASEDIR_PROPERTY_NAME, System.getProperty("user.dir")),
                getStringProperty(SUBDIR_PROPERTY_NAME, ""), getStringProperty(
                        LOGFILENAME_PROPERTY_NAME, DEFAULT_LOG_FILE_NAME), System.getProperties());
    }

    /**
     * This method overrides the method publish() of <StreamHandler>. It gets
     * the file to publish the log record by resolving the place holders present
     * in the path pattern according the execution context before calling the
     * method publish() of <StreamHandler>.
     */
    @Override
    public synchronized void publish(final LogRecord record) {
        File logFile = null;

        try {
            logFile = this.getLogFile();
        } catch (PropertiesException pe) {
            reportError(pe.getMessage(), pe, ErrorManager.GENERIC_FAILURE);
        }

        /*
         * FileOutputStream can generate java.io.IOException: Bad file
         * descriptor if this FOS has been closed too early. So, the publish
         * method must keep the file locked until the publishing process is
         * completed, in order to close() the opening logging location file
         * before another thread open new FOS base on a bad file descriptor
         * (because logging location file descriptor is already used).
         */
        if (logFile != null) {
            FileOutputStream fos = null;
            try {
                // open the File Output Stream in append mode (FOS) to the log
                // file
                fos = new FileOutputStream(logFile, true);

                // Set the FOS in order to allow the StreamHandler to write
                // in this FOS.
                super.setOutputStream(fos);

                // publish the log record.
                super.publish(record);
                super.close();
            } catch (final FileNotFoundException e) {
                /*
                 * We can not throw an exception here, so report the exception
                 * to any registered ErrorManager.
                 */
                reportError(null, e, ErrorManager.OPEN_FAILURE);
            } finally {
                if (fos != null) {
                    try {
                        fos.close();
                    } catch (final IOException e) {
                        /*
                         * We can not throw an exception here, so report the
                         * exception to any registered ErrorManager.
                         */
                        reportError(null, e, ErrorManager.CLOSE_FAILURE);
                    }
                }
            }
        }
    }

    /**
     * Get the log file to write the log record.
     * 
     * @return The <File> where the log record must be written.
     * @throws PropertiesException
     *             Thrown if a property cannot be resolved.
     */
    private File getLogFile() throws PropertiesException {
        Properties contextualProperties = PropertiesHelper.flattenProperties(ExecutionContext
                .getProperties());
        String resolvedSubdir = null;
        String resolvedLogFilename = null;

        resolvedSubdir = PropertiesHelper.resolveString(this.subdir, contextualProperties);
        resolvedLogFilename = PropertiesHelper
                .resolveString(this.logFilename, contextualProperties);

        assert resolvedSubdir != null;
        assert resolvedLogFilename != null;

        File logFileDir = new File(this.basedir, resolvedSubdir);
        logFileDir.mkdirs();

        File logFile = new File(logFileDir, resolvedLogFilename);
        return logFile;
    }

    protected static final String getStringProperty(String name, String defaultValue) {
        String val = LogManager.getLogManager().getProperty(name);
        if (val == null) {
            return defaultValue;
        }

        return val.trim();
    }

    final String getBasedir() {
        return basedir;
    }

    final String getSubdir() {
        return subdir;
    }

    final String getLogFilename() {
        return logFilename;
    }

}
