
package com.ebmwebsourcing.easycommons.stream;

import java.io.ByteArrayOutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;

public abstract class DumbExecutor {

    private final ByteArrayOutputStream stdOutCapture;

    private final ByteArrayOutputStream stdErrCapture;

    private Throwable capturedThrowable;

    public DumbExecutor() {
        stdOutCapture = new ByteArrayOutputStream();
        stdErrCapture = new ByteArrayOutputStream();
    }

    protected abstract void doRun() throws Throwable;

    public final void run() {
        // body of this method is statically synchronized because we modify
        // System.err and System.out which are static attributes.
        synchronized (DumbExecutor.class) {
            PrintStream originalStdOut = System.out;
            PrintStream originalStdErr = System.err;
            PrintStream stdOutCapturePrintStream = new PrintStream(
                    new ThreadSpecificCapturingStream(originalStdOut, stdOutCapture));
            PrintStream stdErrCapturePrintStream = new PrintStream(
                    new ThreadSpecificCapturingStream(originalStdErr, stdErrCapture));

            System.setOut(stdOutCapturePrintStream);
            System.setErr(stdErrCapturePrintStream);
            try {
                doRun();
            } catch (Throwable t) {
                capturedThrowable = t;
            }
            System.setOut(originalStdOut);
            System.setErr(originalStdErr);
        }
    }

    public final String getCapturedStdOut() {
        return stdOutCapture.toString();
    }

    public String getCapturedStdErr() {
        return stdErrCapture.toString();
    }

    public Throwable getCapturedThrowable() {
        return capturedThrowable;
    }

    private static final class ThreadSpecificCapturingStream extends FilterOutputStream {

        private final Thread capturingThread;

        private final OutputStream capturingOutputStream;

        public ThreadSpecificCapturingStream(OutputStream capturedOutputStream,
                OutputStream capturingOutputStream) {
            super(capturedOutputStream);
            this.capturingThread = Thread.currentThread();
            this.capturingOutputStream = capturingOutputStream;
        }

        @Override
        public void write(int b) throws IOException {
            if (Thread.currentThread() != capturingThread) {
                super.write(b);
            } else {
                capturingOutputStream.write(b);
            }
        }

    }

}
