/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.stream;

import java.io.ByteArrayOutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;

public abstract class DumbExecutor {

    private final ByteArrayOutputStream stdOutCapture;

    private final ByteArrayOutputStream stdErrCapture;

    private Throwable capturedThrowable;

    public DumbExecutor() {
        stdOutCapture = new ByteArrayOutputStream();
        stdErrCapture = new ByteArrayOutputStream();
    }

    protected abstract void doRun() throws Throwable;

    public final void run() {
        // body of this method is statically synchronized because we modify
        // System.err and System.out which are static attributes.
        synchronized (DumbExecutor.class) {
            PrintStream originalStdOut = System.out;
            PrintStream originalStdErr = System.err;
            PrintStream stdOutCapturePrintStream = new PrintStream(
                    new ThreadSpecificCapturingStream(originalStdOut, stdOutCapture));
            PrintStream stdErrCapturePrintStream = new PrintStream(
                    new ThreadSpecificCapturingStream(originalStdErr, stdErrCapture));

            System.setOut(stdOutCapturePrintStream);
            System.setErr(stdErrCapturePrintStream);
            try {
                doRun();
            } catch (Throwable t) {
                capturedThrowable = t;
            }
            System.setOut(originalStdOut);
            System.setErr(originalStdErr);
        }
    }

    public final String getCapturedStdOut() {
        return stdOutCapture.toString();
    }

    public String getCapturedStdErr() {
        return stdErrCapture.toString();
    }

    public Throwable getCapturedThrowable() {
        return capturedThrowable;
    }

    private static final class ThreadSpecificCapturingStream extends FilterOutputStream {

        private final Thread capturingThread;

        private final OutputStream capturingOutputStream;

        public ThreadSpecificCapturingStream(OutputStream capturedOutputStream,
                OutputStream capturingOutputStream) {
            super(capturedOutputStream);
            this.capturingThread = Thread.currentThread();
            this.capturingOutputStream = capturingOutputStream;
        }

        @Override
        public void write(int b) throws IOException {
            if (Thread.currentThread() != capturingThread) {
                super.write(b);
            } else {
                capturingOutputStream.write(b);
            }
        }

    }

}
