/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.stream;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;

import com.ebmwebsourcing.easycommons.lang.UncheckedException;

/**
 * @author aruffie This forker class allow to create on demand one copy of
 * original <InputStream> provided. Several copies can be recovered by
 * many calls to its, but take in account that original <InputStream>
 * provided will be consumed.
 * @see {@link InputStreamForker#fork()} function.
 */
public final class InputStreamForker {
    
    private static final int BUFFER_SIZE = 8192;
    
    private final byte[] bytes;
    
    public InputStreamForker(final InputStream originalInputStream) {
        bytes = readBytes(new BufferedInputStream(originalInputStream));
    }

    
    private static final byte[] readBytes(InputStream originalInputStream) {
        int n;
        final byte[] buffer = new byte[BUFFER_SIZE];
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            while ((n = originalInputStream.read(buffer)) != -1) {
                baos.write(buffer, 0, n);
            }
        } catch (IOException e) {
            throw new UncheckedException(e);
        }
        return baos.toByteArray();
    }
    
    /**
     * This function allow to recover a new <InputStream> which contains one
     * copy of the original <InputStream>
     * 
     * @return Return one copy of the original <InputStream>
     * @throws <IOException>
     */
    public final InputStream fork() throws IOException {
        return new ByteArrayInputStream(bytes);
    }

    
    
    
}
