/**
 * 
 */

package com.ebmwebsourcing.easycommons.registry;

import java.util.HashMap;
import java.util.Map;

/**
 * This is a generic implementation of a resource registry. All methods that
 * alter registry state are thread safe.
 * 
 * @author ofabre
 * 
 */
public class ResourceRegistry<T> {

    private Map<String, T> registry = new HashMap<String, T>();

    /**
     * 
     * @param key
     *            must not be null
     * @param resource
     *            must not be null
     * @throws KeyAlreadyBoundException
     *             if the provided key is already bound to a registered object
     */
    public synchronized void register(String key, T resource) {
        assert key != null;
        assert resource != null;

        T previouslyMappedRessource = registry.get(key);
        if (previouslyMappedRessource != null) {
            throw new KeyAlreadyBoundException("The provided key is already bound: " + key);
        } else {
            registry.put(key, resource);
        }
    }

    /**
     * 
     * @param key
     *            must not be null
     * @throws KeyNotFoundException
     *             if the provided key isn't found in registry
     */
    public synchronized void unregister(String key) {
        assert key != null;

        T previouslyMappedRessource = registry.get(key);
        if (previouslyMappedRessource == null) {
            throw new KeyNotFoundException("The given key isn't found in registry: " + key);
        } else {
            registry.remove(key);
        }
    }

    /**
     * 
     * @param key
     *            must not be null
     * @return the resource bound to this key or null if not found
     */
    public T lookup(String key) {
        assert key != null;

        return registry.get(key);
    }

}
