/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.registry;

import java.util.HashMap;
import java.util.Map;

/**
 * This is a generic implementation of a resource registry. All methods that
 * alter registry state are thread safe.
 * 
 * @author ofabre
 * 
 */
public class ResourceRegistry<T> {

    private Map<String, T> registry = new HashMap<String, T>();

    /**
     * 
     * @param key
     *            must not be null
     * @param resource
     *            must not be null
     * @throws KeyAlreadyBoundException
     *             if the provided key is already bound to a registered object
     */
    public synchronized void register(String key, T resource) {
        assert key != null;
        assert resource != null;

        T previouslyMappedRessource = registry.get(key);
        if (previouslyMappedRessource != null) {
            throw new KeyAlreadyBoundException("The provided key is already bound: " + key);
        } else {
            registry.put(key, resource);
        }
    }

    /**
     * 
     * @param key
     *            must not be null
     * @throws KeyNotFoundException
     *             if the provided key isn't found in registry
     */
    public synchronized void unregister(String key) {
        assert key != null;

        T previouslyMappedRessource = registry.get(key);
        if (previouslyMappedRessource == null) {
            throw new KeyNotFoundException("The given key isn't found in registry: " + key);
        } else {
            registry.remove(key);
        }
    }

    /**
     * 
     * @param key
     *            must not be null
     * @return the resource bound to this key or null if not found
     */
    public T lookup(String key) {
        assert key != null;

        return registry.get(key);
    }

}
