/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.soap;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import javax.xml.XMLConstants;
import javax.xml.soap.SOAPConstants;

import org.custommonkey.xmlunit.DetailedDiff;
import org.custommonkey.xmlunit.Diff;
import org.custommonkey.xmlunit.XMLUnit;
import org.w3c.dom.Attr;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.easycommons.xml.IgnoreIrrelevantNodesDifferenceListener;



public final class SoapComparator {

    private SoapComparator() {
    }
    
    public static boolean isEquivalent(String s1, String s2) {
        return isEquivalent(new ByteArrayInputStream(s1.getBytes()), new ByteArrayInputStream(s2.getBytes()));
    }

    public static boolean isEquivalent(InputStream is1, InputStream is2) {
        try {
            XMLUnit.setIgnoreAttributeOrder(true);
            XMLUnit.setIgnoreWhitespace(true);
            
            Diff diff = XMLUnit.compareXML(new InputStreamReader(is1), new InputStreamReader(is2));
            diff.overrideDifferenceListener(new IgnoreIrrelevantNodesDifferenceListener() {
                
                @Override
                protected boolean isIrrelevantChildNode(Node node) {
                    assert node != null;
                    if (SOAPConstants.URI_NS_SOAP_1_1_ENVELOPE.equals(node.getNamespaceURI()) &&
                            "Header".equals(node.getLocalName()) &&
                            !node.hasChildNodes()) return true;
                    return false;
                }

                @Override
                protected boolean isIrrelevantAttribute(Attr att) {
                    assert att != null;
                    return (XMLConstants.XMLNS_ATTRIBUTE_NS_URI.equals(att.getNamespaceURI())) ||
                        (XMLConstants.W3C_XML_SCHEMA_INSTANCE_NS_URI.equals(att.getNamespaceURI()));
                }
            });
            if (!diff.identical()) {
                System.err.println(new DetailedDiff(diff));
            }
            
            return diff.identical();
        } catch (SAXException se) {
            throw new UncheckedException(se);
        } catch (IOException ioe) {
            throw new UncheckedException(ioe);
        }
    }
    

    
    
}
