/**
 * PETALS - PETALS Services Platform.
 * Copyright (c) 2007 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.ebmwebsourcing.easycommons.uuid;

import com.eaio.uuid.UUID;

/**
 * <p>
 * A simple and fast qualified UUID generator.<br/>
 * </p>
 * <p>
 * The identifier is composed of two parts:
 * <ol>
 * <li>a domain namespace,</li>
 * <li>an UUID created according to the DCE Universal Token Identifier
 * specification</li>
 * </ol>
 * </p>
 * <p>
 * Here is a generated UUID sample, based on a domain namespace called
 * 'petalslink': <b>petalslink:00000000-0000-002a-0000-00000000002a<b>.
 * <p>
 * 
 * @author ofabre
 * 
 */
public final class QualifiedUUIDGenerator extends SimpleUUIDGenerator {

    private static final String SEPARATOR = ":";

    /**
     * The max size of the UUID string.
     */
    public final static int UUID_GENERATED_PART_SIZE = new UUID().toString().length();

    private int uuidMaxLength;

    private String domain;

    /**
     * 
     * @param domain
     *            The domain namespace of the unique identifier.
     */
    public QualifiedUUIDGenerator(final String domain) {
        assert domain != null : "UUID domain must not be null. " +
        		"See SimpleUUIDGenerator for non qualified UUID";
        this.domain = domain;
        this.uuidMaxLength = domain.length() + SEPARATOR.length() + UUID_GENERATED_PART_SIZE;
    }

    /**
     * <p>
     * Generates an unique identifier. An identifier value has never been
     * returned (existed) before its calculation.
     * </p>
     * 
     * <p>
     * The identifier is composed of two parts:
     * <ol>
     * <li>a domain namespace,</li>
     * <li>an UUID created according to the DCE Universal Token Identifier
     * specification</li>
     * </ol>
     * </p>
     * <p>
     * This method is threadsafe.
     * </p>
     * 
     * @param prefixNS
     *            prefix or not the id
     * @return A unique identifier
     * 
     */
    public String getNewID() {
        final StringBuilder builder = new StringBuilder(uuidMaxLength);
        builder.append(domain).append(SEPARATOR);
        return builder.append(super.getNewID()).toString();
    }

    public String getDomain() {
        return domain;
    }

    public int getUuidMaxLength() {
        return uuidMaxLength;
    }

    public static String getSeparator() {
        return SEPARATOR;
    }

}
