/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.uuid;

import com.eaio.uuid.UUID;

/**
 * <p>
 * A simple and fast qualified UUID generator.<br/>
 * </p>
 * <p>
 * The identifier is composed of two parts:
 * <ol>
 * <li>a domain namespace,</li>
 * <li>an UUID created according to the DCE Universal Token Identifier
 * specification</li>
 * </ol>
 * </p>
 * <p>
 * Here is a generated UUID sample, based on a domain namespace called
 * 'petalslink': <b>petalslink:00000000-0000-002a-0000-00000000002a<b>.
 * <p>
 * 
 * @author ofabre
 * 
 */
public final class QualifiedUUIDGenerator extends SimpleUUIDGenerator {

    private static final String SEPARATOR = ":";

    /**
     * The max size of the UUID string.
     */
    public final static int UUID_GENERATED_PART_SIZE = new UUID().toString().length();

    private int uuidMaxLength;

    private String domain;

    /**
     * 
     * @param domain
     *            The domain namespace of the unique identifier.
     */
    public QualifiedUUIDGenerator(final String domain) {
        assert domain != null : "UUID domain must not be null. " +
        		"See SimpleUUIDGenerator for non qualified UUID";
        this.domain = domain;
        this.uuidMaxLength = domain.length() + SEPARATOR.length() + UUID_GENERATED_PART_SIZE;
    }

    /**
     * <p>
     * Generates an unique identifier. An identifier value has never been
     * returned (existed) before its calculation.
     * </p>
     * 
     * <p>
     * The identifier is composed of two parts:
     * <ol>
     * <li>a domain namespace,</li>
     * <li>an UUID created according to the DCE Universal Token Identifier
     * specification</li>
     * </ol>
     * </p>
     * <p>
     * This method is threadsafe.
     * </p>
     * 
     * @param prefixNS
     *            prefix or not the id
     * @return A unique identifier
     * 
     */
    public String getNewID() {
        final StringBuilder builder = new StringBuilder(uuidMaxLength);
        builder.append(domain).append(SEPARATOR);
        return builder.append(super.getNewID()).toString();
    }

    public String getDomain() {
        return domain;
    }

    public int getUuidMaxLength() {
        return uuidMaxLength;
    }

    public static String getSeparator() {
        return SEPARATOR;
    }

}
