/*******************************************************************************
 * Copyright (c) 2011 EBM Websourcing.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v2.1
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * 
 * Contributors:
 *     EBM Websourcing - initial API and implementation
 ******************************************************************************/
package com.ebmwebsourcing.easyesb.soa.impl.endpoint;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Logger;

import javax.xml.namespace.QName;

import org.oasisopen.sca.annotation.PolicySets;
import org.oasisopen.sca.annotation.Scope;
import org.osoa.sca.annotations.Property;
import org.petalslink.abslayer.Factory;
import org.petalslink.abslayer.service.api.Description;

import com.ebmwebsourcing.easycommons.research.util.SOAException;
import com.ebmwebsourcing.easycommons.research.util.easybox.SOAUtil;
import com.ebmwebsourcing.easycommons.research.util.easybox.resolver.URIMemoryMultipleResolvers;
import com.ebmwebsourcing.easycommons.research.util.esb.ESBUtil;
import com.ebmwebsourcing.easycommons.research.util.esb.EndpointAddress;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAComponentImpl;
import com.ebmwebsourcing.easycommons.xml.resolver.ClasspathURIResolver;
import com.ebmwebsourcing.easycommons.xml.resolver.DefaultURIResolver;
import com.ebmwebsourcing.easyesb.constant.EasyESBFramework;
import com.ebmwebsourcing.easyesb.exchange10.api.element.Exchange;
import com.ebmwebsourcing.easyesb.soa.api.ESBException;
import com.ebmwebsourcing.easyesb.soa.api.SOAElement;
import com.ebmwebsourcing.easyesb.soa.api.component.Component;
import com.ebmwebsourcing.easyesb.soa.api.config.Configuration;
import com.ebmwebsourcing.easyesb.soa.api.endpoint.Endpoint;
import com.ebmwebsourcing.easyesb.soa.api.endpoint.ProviderEndpoint;
import com.ebmwebsourcing.easyesb.soa.api.endpoint.behaviour.EndpointBehaviour;
import com.ebmwebsourcing.easyesb.soa.api.endpoint.behaviour.specific.ProviderProxyBehaviour;
import com.ebmwebsourcing.easyesb.soa.api.interceptors.EndpointInitializationInterceptor;
import com.ebmwebsourcing.easyesb.soa.api.node.Node;
import com.ebmwebsourcing.easyesb.soa.api.node.NodeBehaviour;
import com.ebmwebsourcing.easyesb.soa.api.service.Service;
import com.ebmwebsourcing.easyesb.soa.api.transport.Skeleton;
import com.ebmwebsourcing.easyesb.soa.api.transport.TransportersManager;
import com.ebmwebsourcing.easyesb.soa.api.transport.listener.ListenersManager;
import com.ebmwebsourcing.easyesb.soa.impl.endpoint.resolver.EasyESBInternalURIResolver;
import com.ebmwebsourcing.easyesb.soa.impl.transport.SkeletonImpl;
import com.ebmwebsourcing.easyesb.soa10.api.type.EndpointType;
import com.ebmwebsourcing.easyesb.soa10.api.type.NodeType;
import com.ebmwebsourcing.easyesb.soa10.api.type.ProviderEndpointType;
import com.ebmwebsourcing.easyesb.transporter.api.transport.TransportException;
import com.ebmwebsourcing.easywsdl11.api.element.Definitions;
import com.ebmwebsourcing.easywsdl11.api.element.Port;
import com.ebmwebsourcing.soapbinding11.api.SoapBindingHelper;
import com.ebmwebsourcing.soapbinding11.api.element.Address;

@Scope("COMPOSITE")
@org.oasisopen.sca.annotation.Service(value=Endpoint.class, names="service")
@PolicySets("frascati:scaEasyPrimitive")
public abstract class AbstractEndpointImpl<M extends EndpointType> extends SCAComponentImpl implements Endpoint<M> {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;


	private static Logger log = Logger.getLogger(AbstractEndpointImpl.class.getName());

	private Node<? extends NodeType> node;

	protected Description description;

	private ListenersManager listenersManager = null;

	@Property(name="model", required=true)
	protected M model = null;

	@Property(name="behaviours", required=true)
	private List<EndpointBehaviour> behaviours = null;

	@Property(name="parent", required=true)
	protected SOAElement<?> parent = null;

	private Skeleton skeleton = null;

	protected List<EndpointInitializationInterceptor> initializationsInterceptor = new ArrayList<EndpointInitializationInterceptor>();

	// only used by frascati
	public AbstractEndpointImpl() throws ESBException {
		this.skeleton = new SkeletonImpl(this);
	}

	public AbstractEndpointImpl(M model, SOAElement<?> parent) throws ESBException {
		assert model != null;
		this.skeleton = new SkeletonImpl(this);
		this.model = model;

		if(model == null) {
			throw new ESBException("the model cannot be null!!!");
		}
		if(this.model.getReference() == null) {
			throw new ESBException("the endpoint name cannot be null!!!");
		}

		this.parent = parent;

	}

	public SOAElement<?> getParent() {
		return parent;
	}


	@SuppressWarnings("unchecked")
	public void init() throws ESBException {
		try {
			this.model.setType(this.getClass().getName());

			// init behaviours
			this.behaviours = new ArrayList<EndpointBehaviour>();
			for(Class<? extends EndpointBehaviour> behaviourClass: this.getBehaviourClasses()) {
				EndpointBehaviour behaviour = createBehaviourFormClass(behaviourClass);
				this.behaviours.add(behaviour);
			}

			// create description
			this.description = createDescriptionFromBehaviours();

			createEndpointInitializationInterceptorFromContext();
			createEndpointInitializationInterceptorFromConfiguration();

			// realize processing of the creation of interceptor
			for(EndpointInitializationInterceptor eii: this.initializationsInterceptor) {
				eii.processingCreation();
			}
		} catch (ClassNotFoundException e) {
			throw new ESBException(e);
		}
	}

	private void createEndpointInitializationInterceptorFromConfiguration() throws ESBException {
		Configuration conf = null;

		if(this.getNode() != null) {
			conf = ((NodeBehaviour)this.getNode().findBehaviour(NodeBehaviour.class)).getConfiguration();
		}

		if(conf != null) {
			// create initiatorInterceptor
			List<String> interceptorClassNames = conf.getEndpointInitializationInterceptorClassNames();
			if(interceptorClassNames != null) {
				for(String interceptorClassName: interceptorClassNames) {
					try {
						Class<?> interceptorClass = Thread.currentThread().getContextClassLoader().loadClass(interceptorClassName);
						log.fine("create interceptor: " + interceptorClass);
						Constructor<?> constructor = interceptorClass.getConstructors()[0];

						EndpointInitializationInterceptor eii = (EndpointInitializationInterceptor) constructor.newInstance(this);

						this.initializationsInterceptor.add(eii);

					} catch (ClassNotFoundException e) {
						log.severe("ERROR : " + e.getMessage());
						e.printStackTrace();
						throw new ESBException(e);
					} catch (IllegalArgumentException e) {
						log.severe("ERROR : " + e.getMessage());
						e.printStackTrace();
						throw new ESBException(e);
					} catch (InstantiationException e) {
						log.severe("ERROR : " + e.getMessage());
						e.printStackTrace();
						throw new ESBException(e);
					} catch (IllegalAccessException e) {
						log.severe("ERROR : " + e.getMessage());
						e.printStackTrace();
						throw new ESBException(e);
					} catch (InvocationTargetException e) {
						log.severe("ERROR : " + e.getMessage());
						e.printStackTrace();
						throw new ESBException(e);
					} catch (SecurityException e) {
						log.severe("ERROR : " + e.getMessage());
						e.printStackTrace();
						throw new ESBException(e);
					} 
				}
			}
		}
	}


	private void createEndpointInitializationInterceptorFromContext()
			throws ESBException {
		// create initiatorInterceptor
		List<String> interceptorClassNames = null;
		if(this.model.getEndpointInitialContext() != null && this.model.getEndpointInitialContext().getInitializationInterceptors() != null) {
			interceptorClassNames = Arrays.asList(this.model.getEndpointInitialContext().getInitializationInterceptors().getInterceptorClassName());
		}
		if(interceptorClassNames != null) {
			for(String interceptorClassName: interceptorClassNames) {
				try {
					Class<?> interceptorClass = Thread.currentThread().getContextClassLoader().loadClass(interceptorClassName);
					log.fine("create interceptor: " + interceptorClass);
					Constructor<?> constructor = interceptorClass.getConstructors()[0];

					EndpointInitializationInterceptor eii = (EndpointInitializationInterceptor) constructor.newInstance(this);

					this.initializationsInterceptor.add(eii);

				} catch (ClassNotFoundException e) {
					log.severe("ERROR : " + e.getMessage());
					e.printStackTrace();
					throw new ESBException(e);
				} catch (IllegalArgumentException e) {
					log.severe("ERROR : " + e.getMessage());
					e.printStackTrace();
					throw new ESBException(e);
				} catch (InstantiationException e) {
					log.severe("ERROR : " + e.getMessage());
					e.printStackTrace();
					throw new ESBException(e);
				} catch (IllegalAccessException e) {
					log.severe("ERROR : " + e.getMessage());
					e.printStackTrace();
					throw new ESBException(e);
				} catch (InvocationTargetException e) {
					log.severe("ERROR : " + e.getMessage());
					e.printStackTrace();
					throw new ESBException(e);
				} catch (SecurityException e) {
					log.severe("ERROR : " + e.getMessage());
					e.printStackTrace();
					throw new ESBException(e);
				} 
			}
		}
	}

	public URI getReference() {
		return this.model.getReference();
	}

	public Skeleton getSkeleton() {
		return this.skeleton;
	}

	public Node<? extends NodeType> getNode() {
		return this.node;
	}

	public void accept(Exchange exchange) throws TransportException {
		this.skeleton.accept(exchange);
	}

	public void sendResponseToClient(Exchange exchange)
			throws TransportException {
		this.skeleton.sendResponseToClient(exchange);
	}

	public boolean getTakeToSendResponseInCharge() {
		return this.skeleton.getTakeToSendResponseInCharge();
	}

	public void setTakeToSendResponseInCharge(boolean takeToSendResponseInCharge) {
		this.skeleton.setTakeToSendResponseInCharge(takeToSendResponseInCharge);
	}

	public TransportersManager getTransportersManager() throws ESBException {
		return this.getNode().getTransportersManager();
	}

	public void setNode(Node<? extends NodeType> node) throws ESBException {
		try {
			this.node = node;
			if(model != null) {
				EndpointAddress epr = ESBUtil.analyzeURI(node.getReference());

				this.model.setNode(new QName(epr.getNamespace(), epr.getServicename()));
				this.model.setBasicNodeInformations(node.getModel().getBasicNodeInformations());
			}
		} catch (SOAException e) {
			throw new ESBException(e);
		}
	}



	public ListenersManager getListenersManager() {
		return this.listenersManager;
	}

	public void setListenersManager(ListenersManager listenersManager) {
		this.listenersManager = listenersManager;
	}

	public M getModel() {
		return this.model;
	}

	public void addEndpointInitializationInterceptor(
			EndpointInitializationInterceptor i) {
		this.initializationsInterceptor.add(i);
	}

	public EndpointInitializationInterceptor removeEndpointInitializationInterceptor(
			EndpointInitializationInterceptor i) {
		return (this.initializationsInterceptor.remove(i) ? i:null);
	}

	public List<EndpointInitializationInterceptor> getEndpointInitializationInterceptors() {
		return this.initializationsInterceptor;
	}


	@Override
	public Description getDescription() {
		return description;
	}

	public void setDescription(Description desc) {
		this.description = desc;
	}

	@Override
	public Class<? extends EndpointBehaviour>[] getBehaviourClasses() throws ClassNotFoundException {
		List<Class<? extends EndpointBehaviour>> res = new ArrayList<Class<? extends EndpointBehaviour>>();
		for(String className: this.getModel().getBehavioursList().getBehaviours()) {
			res.add((Class<? extends EndpointBehaviour>) Class.forName(className));
		}
		return res.toArray(new Class[res.size()]);
	}

	@Override
	public void addBehaviourClass(Class<? extends EndpointBehaviour> behaviourClass) throws ESBException {
		if(behaviourClass != null) {
			if(this.model.getBehavioursList().getBehaviourByName(behaviourClass.getName()) == null) {
				this.model.getBehavioursList().addBehaviour(behaviourClass.getName());
				this.behaviours.add(this.createBehaviourFormClass(behaviourClass));

				// re-create description
				this.description = createDescriptionFromBehaviours();
			}
		} 
	}

	@Override
	public void removeBehaviourClass(
			Class<? extends EndpointBehaviour> behaviourClass) throws ESBException {
		this.model.getBehavioursList().removeBehaviour(behaviourClass.getName());
		EndpointBehaviour behaviourToDelete = null;
		for(EndpointBehaviour eb: this.getBehaviours()) {
			if(behaviourClass.isAssignableFrom(eb.getClass())) {
				behaviourToDelete = eb;
			}
		}
		if(behaviourToDelete != null) {
			this.behaviours.remove(behaviourToDelete);

			// re-create description
			this.description = createDescriptionFromBehaviours();
		}
	}

	@Override
	public List<EndpointBehaviour> getBehaviours() {
		return this.behaviours;
	}


	protected EndpointBehaviour createBehaviourFormClass(
			Class<? extends EndpointBehaviour> behaviourClass)
					throws ESBException {
		EndpointBehaviour res = null;
		if(behaviourClass != null) {
			if(behaviourClass.isInterface()) {
				throw new ESBException("Impossible to add an interface, give the implementation");
			}
			log.fine("create behaviour: " + behaviourClass);
			System.out.println("create behaviour: " + behaviourClass);
			try {
				Constructor<EndpointBehaviour> constructor = (Constructor<EndpointBehaviour>) behaviourClass.getConstructors()[0];
				res = constructor.newInstance(this);
			} catch (IllegalArgumentException e) {
				log.severe("ERROR : " + e.getMessage());
				e.printStackTrace();
				throw new ESBException(e);
			} catch (InstantiationException e) {
				log.severe("ERROR : " + e.getMessage());
				e.printStackTrace();
				throw new ESBException(e);
			} catch (IllegalAccessException e) {
				log.severe("ERROR : " + e.getMessage());
				e.printStackTrace();
				throw new ESBException(e);
			} catch (InvocationTargetException e) {
				log.severe("ERROR : " + e.getMessage());
				e.printStackTrace();
				throw new ESBException(e);
			} catch (SecurityException e) {
				log.severe("ERROR : " + e.getMessage());
				e.printStackTrace();
				throw new ESBException(e);
			} 
		}
		return res;
	}

	@Override
	public <B extends EndpointBehaviour> B findBehaviour(Class<B> behaviour) {
		B res = null;
		if(this.getBehaviours() != null) {
			for(EndpointBehaviour b: this.getBehaviours()) {
				if(behaviour.isAssignableFrom(b.getClass())) {
					res = (B) b;
					break;
				}
			}
		}
		return res;
	}

	public void refreshDescription() throws ESBException {
		this.description = createDescriptionFromBehaviours();
	}

	private Description createDescriptionFromBehaviours() throws ESBException {
		Description desc = null;
		try {
			if(this instanceof ProviderEndpoint) {
				Definitions def = SOAUtil.getInstance().getXmlContext(EasyESBFramework.getInstance()).getXmlObjectFactory().create(Definitions.class);

				// create import
				EndpointAddress epr = ESBUtil.analyzeURI(this.getReference());
				QName serviceQName = ((ProviderEndpointType)this.getModel()).getServiceName();


				if(serviceQName == null) {
					throw new ESBException("serviceQName cannot be null!!!");
				}

				EasyESBInternalURIResolver resolver = new EasyESBInternalURIResolver();
				SOAUtil.getInstance().getXmlContext(EasyESBFramework.getInstance()).setURIResolver(new URIMemoryMultipleResolvers(new DefaultURIResolver(), new ClasspathURIResolver(), new EasyESBInternalURIResolver()));

				def.setTargetNamespace(serviceQName.getNamespaceURI());
				def.setName(serviceQName.getLocalPart() + "_Description");
				for(EndpointBehaviour eb: this.getBehaviours()) {
					if((!(eb instanceof ProviderProxyBehaviour))&&(eb.getBinding() != null)) {
						com.ebmwebsourcing.easywsdl11.api.element.Import impt = SOAUtil.getInstance().getXmlContext(EasyESBFramework.getInstance()).getXmlObjectFactory().create(com.ebmwebsourcing.easywsdl11.api.element.Import.class);
						impt.setNamespace(eb.getBinding().getQName().getNamespaceURI());
						if(eb.getBinding().getDescription().getDocumentBaseURI() != null) {
							URI imptUri = eb.getBinding().getDescription().getDocumentBaseURI();

							impt.setLocation(imptUri.toString());
							resolver.addImportedURI(imptUri);
						}
						def.addImport(impt);
					}
				}

				// create service
				com.ebmwebsourcing.easywsdl11.api.element.Service service = SOAUtil.getInstance().getXmlContext(EasyESBFramework.getInstance()).getXmlObjectFactory().create(com.ebmwebsourcing.easywsdl11.api.element.Service.class);
				service.setName(serviceQName.getLocalPart());
				def.addService(service);

				// create endpoints
				for(EndpointBehaviour eb: this.getBehaviours()) {
					if((!(eb instanceof ProviderProxyBehaviour))&&(eb.getBinding() != null)) {
						Port ep = SOAUtil.getInstance().getXmlContext(EasyESBFramework.getInstance()).getXmlObjectFactory().create(com.ebmwebsourcing.easywsdl11.api.element.Port.class);
						ep.setBinding(eb.getBinding().getQName());
						ep.setName(((ProviderEndpointType)this.getModel()).getName());
						Address addr = SOAUtil.getInstance().getXmlContext(EasyESBFramework.getInstance()).getXmlObjectFactory().create(Address.class);
						addr.setLocation(this.getReference().toString());
						SoapBindingHelper.setAddress(ep, addr);
						service.addPort(ep);
					}
				}

				desc = (Description) Factory.getInstance().wrap(def);
			} 
		} catch (SOAException e) {
			throw new ESBException(e);
		}
		return desc;
	}



}
