/**
 * geasy-diagram-editor - A project for editing diagrams based on OMG Diagram Definition standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.geasytools.diagrameditor;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;

import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramElementGraphicState;
import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramElementView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramView;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.diagramcommon.layout.IPoint;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagram;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagramElement;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IEdge;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IShape;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.loader.ElementLoadedEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.loader.LoadErrorEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.loader.LoadSuccessEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.loader.LoadingStateChangeEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnectable;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnector;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnectorPoint;
import com.ebmwebsourcing.geasytools.geasyui.api.core.IContainer;
import com.ebmwebsourcing.geasytools.geasyui.impl.connectable.ConnectorPoint;
import com.ebmwebsourcing.geasytools.geasyui.impl.core.Point;
import com.ebmwebsourcing.geasytools.geasyui.impl.core.Util;



/**
 * Creates and places {@link IDiagramElementView} into a {@link IDiagramView}
 * @author nfleury
 */
public class DiagramViewLoader {
	
	private DiagramController controller;
	
	private boolean loadWithRelativeCoordinates;
	private IDiagramView diagramView;
	private IDiagram diagramModel;
	
	private HashSet<IDiagramElement> treatedElements;
	
	private HashSet<IDiagramElementView> elementViews;
	
	private int totalElements;

	private int finished;

	private int percentage;

	private String loadingStatus;

	
	public DiagramViewLoader(DiagramController controller) {
		this.controller = controller;
		this.treatedElements = new HashSet<IDiagramElement>();
	}
	
	
	public void loadDiagram(IDiagramView diagramView,
			IDiagram diagramModel,boolean loadWithRelativeCoordinates){
		
		
		this.diagramView 	= diagramView;
		this.diagramModel 	= diagramModel;
		this.loadWithRelativeCoordinates = loadWithRelativeCoordinates;
		this.elementViews   = new HashSet<IDiagramElementView>();
		
		//Assign root element's model
		this.diagramView.getDiagram().getRootElement().setModelElement(diagramModel.getRootElement().getModelElement());
		
		//initialize diagram editor model with it syntax model
		this.diagramView.getSyntaxModelBuilder().initializeEditorModel(diagramView.getEditorModel(), diagramModel.getRootElement().getModelElement(),controller.getMainModelElementWithoutValidation());
		
		ArrayList<IDiagramElementView> elementsView = new ArrayList<IDiagramElementView>();


		this.totalElements = diagramModel.getRootElement().getOwnedElements().size();
		this.finished	 = 0;
		this.percentage  = 0;

		this.loadingStatus = "Loading diagram properties";
		controller.fireEvent(new LoadingStateChangeEvent(loadingStatus, percentage));
		
		////////////Set Diagram properties
		this.diagramView.getDiagram().setName(diagramModel.getName());
		this.diagramView.getDiagram().setDocument(diagramModel.getDocument());
		////////////Load diagram elements
		
		this.loadingStatus = "Loading diagram shapes";
		controller.fireEvent(new LoadingStateChangeEvent(loadingStatus, percentage));
		

		//first parse every diagram elements shapes that are on the root Plane
		for(IDiagramElement diElement:diagramModel.getRootElement().getOwnedElements()){
			

			if (treatedElements.contains(diElement)==false && diElement instanceof IShape){

				IDiagramElementView view = loadDiagramElementView(diElement,null);			
				elementsView.add(view);

			}
		
		
		}

		//TODO: change loading state
		this.loadingStatus = "Loading diagram edges";
		controller.fireEvent(new LoadingStateChangeEvent(loadingStatus, percentage));
		
		//then deal with edges
		for(IDiagramElement diElement:diagramModel.getRootElement().getOwnedElements()){
			

			if (treatedElements.contains(diElement)==false && diElement instanceof IEdge){
				IDiagramElementView view = loadDiagramElementView(diElement,null);			
				elementsView.add(view);
				
			}
		
		
		}
		
		this.loadingStatus = "Processing diagram size";
		
		controller.fireEvent(new LoadingStateChangeEvent(loadingStatus, percentage));
		
		//process diagram size
		Point nwPoint = Util.getInstance().getNWPointFromElements(elementsView);
		Point sePoint = Util.getInstance().getSEPointFromElements(elementsView);
		
		
		int width = (int) (sePoint.getX() - nwPoint.getX());
		int height= (int) (sePoint.getY() - nwPoint.getY());
		
		
//		diagramView.setWidth(width);
//		diagramView.setHeight(height);
		
		
		
		//Set the graphical state of previously loaded elements
		for(IDiagramElementView elView:elementViews){
			
		
			for(IDiagramElementGraphicState state:elView.getStates()){
				
					if (state.condition(elView.getDiagramElement(), elView.getEditorModel())){
						
						state.changeGraphicState(elView);
						
					}
				
			}
	
		}
		
		
		treatedElements.clear();
		elementViews.clear();

		controller.fireEvent(new LoadSuccessEvent());


	}
	
	private IDiagramElementView loadDiagramElementView(IDiagramElement diagramElement,IContainer parentElementView){
		
		if (diagramElement.getModelElement()==null){
			throw new IllegalStateException("IDiagramElement "+diagramElement+" doesn't reference any IModelElement");
		}

		
		//first create the corresponding graphic element based on Model Element instance
		IDiagramElementView elView = diagramView.getElementFactory()
												.getElementByDiagramElementModel(
														diagramElement);		
			
		
			//if couldn't create the element view
			// ... can't go further dude !
			if (elView==null){
				controller.fireEvent(new LoadErrorEvent("Couldn't load an element"));
				throw new IllegalStateException("Please, make sure the {@link IDiagramElementGraphicFactory} factory has been correctly implemented. Loader couldn't create a {@link IDiagramElementView} from type "+diagramElement
						.getModelElement()
						.getClass());
			}
		
		
			diagramView.addUIElement(elView);
		
			if (parentElementView!=null && (diagramElement instanceof IEdge==false)){
				elView.setContainer(parentElementView);
				parentElementView.addUIElement(elView);
				elView.refresh();
			}
		
			if (diagramElement instanceof IShape){
				
				IShape shape = (IShape) diagramElement;
				
				float relativeX = (float) shape.getBounds().getX();
				float relativeY = (float) shape.getBounds().getY();
				
				//means that the provided values are absolute relatively
				//to the Plane element. So we need to calculate actual
				//element position relatively to its container
				if (loadWithRelativeCoordinates==true && parentElementView!=null){
					
					relativeX 		= relativeX - (parentElementView.getAbsoluteLeft() - diagramView.getAbsoluteLeft());
					relativeY 	 	= relativeY - (parentElementView.getAbsoluteTop() - diagramView.getAbsoluteTop()); 
				
				}
				
				elView.setRelativeX(relativeX);
				elView.setRelativeY(relativeY);
				
				
				
				//bind actual DI shape bound to view DI bounds
				IShape elViewShape = (IShape) elView.getDiagramElement();
				//System.out.println(elView.getName()+" loading x:"+shape.getBounds().getX()+" y:"+shape.getBounds().getY());
				elViewShape.getBounds().setWidth(shape.getBounds().getWidth());
				elViewShape.getBounds().setHeight(shape.getBounds().getHeight());
				elViewShape.getBounds().setX(shape.getBounds().getX());
				elViewShape.getBounds().setY(shape.getBounds().getY());
				
				
//				Shape must have a bound !!
//				if (shape.getBounds().getWidth()==0){
//					throw new IllegalStateException("Shape with id:"+shape.getId()+" has a with of 0. Make sure actual shape has a width supperior to 0");	
//				}
//				
//				if (shape.getBounds().getWidth()==0){	
//					throw new IllegalStateException("Shape with id"+shape.getId()+" has a height of 0. Make sure actual shape has a height supperior to 0");
//				}				
				//Shape should have bound otherwise take default value
				//TODO:check that elView has a default width a height
				if (shape.getBounds().getWidth() > 0) elView.setWidth((float) shape.getBounds().getWidth());
				if (shape.getBounds().getHeight() > 0 ) elView.setHeight((float) shape.getBounds().getHeight());
			
			}
			
			
			if (diagramElement instanceof IEdge && parentElementView==null){
				
				if (elView instanceof IConnector){
					
					IEdge edge = (IEdge) diagramElement;
					IConnector connector = (IConnector) elView;
					
					//set wayPoints
					IConnectorPoint previousPoint = connector.getConnectorStartPoint();
					IConnectorPoint nextPoint = connector.getConnectorEndPoint();
					
					if (edge.getWayPoints().size()>2){
						
						ArrayList<IPoint> edgeWayPoints = new ArrayList<IPoint>();
						
						Iterator<IPoint> it = edge.getWayPoints().iterator();
						
						while(it.hasNext()){
							
							IPoint p = it.next();
							
							edgeWayPoints.add(p);
							
						}
						
						for(int i=1;i<edge.getWayPoints().size()-1;i++){
							
							IPoint actualWayPoint = edgeWayPoints.get(i);
							
							IConnectorPoint pointToAdd = connector.createConnectorPoint((int)actualWayPoint.getX(), (int)actualWayPoint.getY());
							
							connector.addIntermediateConnectorPoint(pointToAdd, previousPoint, nextPoint);
							
							previousPoint = pointToAdd;
						}
						
						
					}
					
					//TODO: bind DI edge to elView edge
					
					
					//get source and target views
					IDiagramElementView sourceElement = diagramView.getUIElementById(edge.getSource().getId());
					IDiagramElementView targetElement = diagramView.getUIElementById(edge.getTarget().getId());
					
					//if we didn't found one of the previous element can't go further
					if (sourceElement==null){
						throw new IllegalStateException("Couldn't find sourceElement with id:"+edge.getSource().getId()+" for edge with id: "+edge.getId());
					}
					
					if (targetElement==null){
						throw new IllegalStateException("Couldn't find targetElement with id:"+edge.getSource().getId()+" for edge with id: "+edge.getId());
					}
					
					//then check if both previous elements are connectable
					if (sourceElement instanceof IConnectable && targetElement instanceof IConnectable){
						
						IConnectable connectableSourceElement = (IConnectable) sourceElement;
						IConnectable connectableTargetElement = (IConnectable) targetElement;
						
						//ok => connect elements
						connector.connect(connectableSourceElement, connectableTargetElement);						

						
					}else{
						
						if (sourceElement instanceof IConnectable ==false){
							throw new IllegalStateException(sourceElement+ "must implement " + IConnectable.class + " in order to be connected with other elements");
						}

						if (targetElement instanceof IConnectable ==false){
							throw new IllegalStateException(targetElement+ "must implement " + IConnectable.class + " in order to be connected with other elements");
						}
						
					}

					
					
				}else{
					
					throw new IllegalStateException("Please, make sure the {@link IDiagramElementGraphicFactory} factory has been correctly implemented. Loader was expecting a {@link IConnector} actual is:"+elView);
				
				}
				
				
				
			}
			
			
			//if created element is a container
			//and actual diagram contains other elements
			if (elView instanceof IContainer && diagramElement.getOwnedElements().size()>0){
				
				IContainer elViewContainer = (IContainer) elView;
				
				//load each one of them and set 
				//actual elView as current container of the created child
				for(IDiagramElement childModel:diagramElement.getOwnedElements()){
					
					//load only if it is not an edge
					if (childModel instanceof IShape) loadDiagramElementView(childModel, elViewContainer);
					
				}
	
			}
			

			
			//refresh the elView
			elView.refresh();
			
			
			this.elementViews.add(elView);
			
			//add diagramElement to treated list
			this.treatedElements.add(diagramElement);
			
			finished++;

			percentage = ((finished*100)/totalElements);
			controller.fireEvent(new LoadingStateChangeEvent(loadingStatus, percentage));
			controller.fireEvent(new ElementLoadedEvent(elView));
			
			return elView;
	}



}
