/**
 * model-manager - Handles models on client side for stuffs like undo/redo, methods observers, uibinding ... - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;




@SuppressWarnings("unchecked")
public class HistoryManager<T> {
		
	private	HashMap<ModelProxy,ModelHistory> modelsHistory;
	
	private HashSet<ModelProxy> undoredoSuscribers;
	
	private HashMap<ModelProxy,List<IHistorizationHandler>> historizationHandlers;
	
	private HashMap<ModelProxy,List<ICopyHandler>> copyHandlers;

	private static HistoryManager instance;
	
	static{
		instance = new HistoryManager();
	}
	
	private HistoryManager() {
		this.modelsHistory 			= new HashMap<ModelProxy, ModelHistory>();
		this.undoredoSuscribers 	= new HashSet<ModelProxy>();
		this.historizationHandlers	= new HashMap<ModelProxy, List<IHistorizationHandler>>();
		this.copyHandlers			= new HashMap<ModelProxy, List<ICopyHandler>>();
	}
	
	/**
	 * TODO: Really necessary ?
	 * @param undoredo
	 */
	public void addUndoRedoSuscriber(ModelProxy undoredo){
		this.undoredoSuscribers.add(undoredo);
	}
	
	
	public static <T> HistoryManager getInstance(){
		return instance;
	}
	
	public void addModelToHistory(ModelProxy model,T stateXModel){

		//get actual history for this model
		ModelHistory history = this.modelsHistory.get(model);
		
		if (history==null){
			
			history = new ModelHistory(model);
			
		}
		
		history.addModelToHistory((HasModelManager) stateXModel);
		
		
		this.modelsHistory.put(model, history);
		
		List<IHistorizationHandler> handlers = historizationHandlers.get(model);

		if (handlers!=null){
			for(IHistorizationHandler h:handlers){
				h.onHistorize(model);
			}
		}
	
	}
	
	
	public void addHistorizationHandler(ModelProxy model,IHistorizationHandler handler){
			
		List<IHistorizationHandler> handlers = historizationHandlers.get(model);
		
		if (handlers==null){
			handlers = new ArrayList<IHistorizationHandler>();
		}
		
		handlers.add(handler);
		
		historizationHandlers.put(model, handlers);
		
	}
	
	public void addCopyHandler(ModelProxy model,ICopyHandler handler){
		
		List<ICopyHandler> handlers = copyHandlers.get(model);
		
		if (handlers==null){
			handlers = new ArrayList<ICopyHandler>();
		}
		
		handlers.add(handler);
		
		copyHandlers.put(model, handlers);
		
	}
	
	public List<ICopyHandler> getCopyHandlers(ModelProxy modelProxy){
		return copyHandlers.get(modelProxy);
	}
	
	
	public boolean hasMoreUndo(T model){
		
		
		ModelHistory history = modelsHistory.get(model);  
		
		if (history==null){
			throw new IllegalStateException("Object "+model+" doesn't have any history yet");
		}
		
		return history.hasMoreUndo();
		
	}
	
	public boolean hasMoreRedo(T model){
		
		ModelHistory history = modelsHistory.get(model);  
		
		if (history==null){
			throw new IllegalStateException("Object "+model+" doesn't have any history yet");
		}
		
		
		return history.hasMoreRedo();
	}
	
	public HashSet<ModelProxy> getUndoredoSuscribers() {
		return undoredoSuscribers;
	}
	
	
	public void undo(T model){

		if (!this.hasMoreUndo(model)){
			throw new IllegalStateException("Object " +model+" doesn't have any history anymore");
		}
		
		
		ModelHistory history = modelsHistory.get(model);  
		
		history.undo();
		
	}
	
	public void redo(T model){
		
		if (!this.hasMoreRedo(model)){
			throw new IllegalStateException("Object " +model+" doesn't have any history anymore");
		}
		
		ModelHistory history = modelsHistory.get(model);  
		
		history.redo();
		
	}


	public interface IHistorizationHandler{
		
		void onHistorize(ModelProxy modelProxy);
		
	}
	
	public interface ICopyHandler{
		
		void onCopy(ModelProxy modelProxy);
		
	}

	public class ModelHistory{
		
		private ModelProxy refModel;
		private List<HasModelManager> history;
		
		//pointer is always on actual state
		private int actualStatePointer;
		
			
		public ModelHistory(ModelProxy model){
			
			this.refModel 	= model;
			this.history 	= new ArrayList<HasModelManager>();
			this.actualStatePointer	= -1;
		}
		
		public void addModelToHistory(HasModelManager model){
			
			this.actualStatePointer = actualStatePointer + 1;
		
			this.history.add(actualStatePointer, model);
		
			if (history.size()-1>actualStatePointer){
				//The last model added reflects the actual model state
				//so we have to remove every p > p+1 in history 
				for(int i = actualStatePointer;i<=history.size()-1;i++){
		
					this.history.remove(i);
					
				}
			}
		}
		
		public void undo(){
			
			if (hasMoreUndo()){
				
				this.actualStatePointer--;
				
				//copy previous model state to actual model ref 
				HasModelManager previousStateModel = history.get(actualStatePointer);
				
				this.refModel.copy(previousStateModel);
				//System.out.println("undo statePointer=>"+actualStatePointer);
				
			}
			
		}
		
		public void redo(){
			
			if (hasMoreRedo()){
				
				this.actualStatePointer++;
				
				//copy next state model to actual model ref
				HasModelManager nextStateModel = history.get(actualStatePointer);
				
				this.refModel.copy(nextStateModel);
				///System.out.println("redo statePointer=>"+actualStatePointer);
			}
			
		}
		
		public boolean hasMoreUndo(){
			
			if (actualStatePointer-1>=0){
				return true;
			}
			
			return false;
		}
		
		public boolean hasMoreRedo(){
			
			if (actualStatePointer+1<=history.size()-1){
				return true;
			}
			
			return false;
		}
		
		
	}
	
}
