/**
 * webeditor-impl - Petalslink is an open source editor providing tools for Service Oriented Architecture. - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.geasytools.webeditor.impl.client.core.handler.manager.component;

import java.util.List;

import com.ebmwebsourcing.geasytools.webeditor.api.components.IContentPanelComponent;
import com.ebmwebsourcing.geasytools.webeditor.api.components.events.IComponentAfterDispatchEvent;
import com.ebmwebsourcing.geasytools.webeditor.api.components.events.IComponentHandler;
import com.ebmwebsourcing.geasytools.webeditor.api.components.events.IComponentLoadedEvent;
import com.ebmwebsourcing.geasytools.webeditor.api.components.events.IComponentUnloadEvent;
import com.ebmwebsourcing.geasytools.webeditor.api.components.events.IComponentUpdateEvent;
import com.ebmwebsourcing.geasytools.webeditor.api.core.DefaultErrorLevel;
import com.ebmwebsourcing.geasytools.webeditor.api.core.handler.manager.component.IDefaultProjectConfigurationHandler;
import com.ebmwebsourcing.geasytools.webeditor.api.core.manager.IComponentManager;
import com.ebmwebsourcing.geasytools.webeditor.api.plugin.IProjectPlugin;
import com.ebmwebsourcing.geasytools.webeditor.api.project.configuration.view.IEditConfigurationPanel;
import com.ebmwebsourcing.geasytools.webeditor.api.project.configuration.view.events.IEditProjectConfigurationCompleteEvent;
import com.ebmwebsourcing.geasytools.webeditor.api.project.configuration.view.events.IExportProjectConfigurationCompleteEvent;
import com.ebmwebsourcing.geasytools.webeditor.api.project.configuration.view.events.IImportProjectConfigurationCompleteEvent;
import com.ebmwebsourcing.geasytools.webeditor.api.project.configuration.view.events.INewProjectConfigurationCompleteEvent;
import com.ebmwebsourcing.geasytools.webeditor.api.project.configuration.view.events.IOpenProjectConfigurationCompleteEvent;
import com.ebmwebsourcing.geasytools.webeditor.api.project.configuration.view.events.IProjectConfigurationCompleteEvent;
import com.ebmwebsourcing.geasytools.webeditor.impl.client.core.events.DisableLoadingStateEvent;
import com.ebmwebsourcing.geasytools.webeditor.impl.client.core.events.EditorErrorEvent;
import com.ebmwebsourcing.geasytools.webeditor.impl.client.core.events.EnableLoadingStateEvent;
import com.ebmwebsourcing.geasytools.webeditor.impl.client.service.project.IProjectServiceAsync;
import com.ebmwebsourcing.webeditor.api.domain.project.IDefaultProjectConfigurationData;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectConfigurationData;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectFile;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstance;
import com.ebmwebsourcing.webeditor.api.domain.project.ProjectValidationException;
import com.google.gwt.user.client.rpc.AsyncCallback;

public class DefaultProjectConfigurationHandler implements IDefaultProjectConfigurationHandler{
	
	private IComponentManager componentManager;
	private IProjectServiceAsync projectService;
	
	
	public DefaultProjectConfigurationHandler(IComponentManager componentManager) {
	
		this.componentManager = componentManager;
	
		this.projectService   = (IProjectServiceAsync) componentManager.getServiceFactory().getProjectService(); 
	}
	
	@Override
	public IComponentManager getComponentManager() {
		return componentManager;
	}

	@Override
	public void onConfigurationComplete(IProjectConfigurationCompleteEvent event) {
		
		
		//New Project
		if (event instanceof INewProjectConfigurationCompleteEvent){
		
			INewProjectConfigurationCompleteEvent npcce = (INewProjectConfigurationCompleteEvent) event;	
			
			componentManager.getEventBus().fireEvent(new EnableLoadingStateEvent("Loading "+npcce.getProjectInstance().getProjectType().getName()+" , please wait..."));
			
			IProjectInstance	pi = npcce.getProjectInstance();
			
			dispatchPluginComponent(pi,true);
			
			componentManager.getEventBus().fireEvent(new DisableLoadingStateEvent());
			
			//TODO: Fire NewProjectCreatedEvent 
			
		//Open Project	
		}else if (event instanceof IOpenProjectConfigurationCompleteEvent){
			
			IOpenProjectConfigurationCompleteEvent opcce = (IOpenProjectConfigurationCompleteEvent) event;

			if (opcce.getProjectInstance()==null){
				
				componentManager.getEventBus().fireEvent(new EnableLoadingStateEvent("Loading "+opcce.getProjectType().getName()+" , please wait..."));
				
				projectService.loadProjectInstances(opcce.getProjectFiles(), new AsyncCallback<List<IProjectInstance>>() {
					
					@Override
					public void onSuccess(List<IProjectInstance> result) {
						
	
						dispatchPluginComponent(result,false);
						
						componentManager.getEventBus().fireEvent(new DisableLoadingStateEvent());
						//TODO: Fire ProjectOpenedEvent
					}
					
					@Override
					public void onFailure(Throwable caught) {
						
						System.out.println("RPC error "+caught.getLocalizedMessage());
						componentManager.getEventBus().fireEvent(new EditorErrorEvent("Project Service Error",caught.getMessage(),DefaultErrorLevel.ERROR));
						
					}
				});
			
			}else{
				
				dispatchPluginComponent(opcce.getProjectInstance(), false);
				
			}

			
		//Import	
		}else if (event instanceof IImportProjectConfigurationCompleteEvent){
			
			IImportProjectConfigurationCompleteEvent ipcce = (IImportProjectConfigurationCompleteEvent) event;
			
			componentManager.getEventBus().fireEvent(new EnableLoadingStateEvent("Importing "+ipcce.getProjectType().getName()+" , please wait..."));
			
			projectService.importProjectInstance(ipcce.getUploadRef(), ipcce.getProjectType(), ipcce.getFormat(), new AsyncCallback<List<IProjectInstance>>() {
				
				@Override
				public void onSuccess(List<IProjectInstance> result) {
					
					dispatchPluginComponent(result,true);
					
					//TODO: Fire ProjectImportedEvent
					componentManager.getEventBus().fireEvent(new DisableLoadingStateEvent());
				}
				
				@Override
				public void onFailure(Throwable caught) {
					
					componentManager.getEventBus().fireEvent(new EditorErrorEvent("Project Service Error",caught.getLocalizedMessage(),DefaultErrorLevel.ERROR));
				}
			});
			
		//Export	
		}else if (event instanceof IExportProjectConfigurationCompleteEvent){
			
			final IExportProjectConfigurationCompleteEvent epcc = (IExportProjectConfigurationCompleteEvent) event;
			
			componentManager.getEventBus().fireEvent(new EnableLoadingStateEvent("Exporting "+epcc.getProjectInstance().getName()+", project will download in a few seconds. Please wait..."));
			
			if(epcc.isCancelled()) { return; }
			
			////////EXPORT PROJECT INSTANCE ONLY IF VALIDATION DOESNT FAIL
			try {
			
				componentManager.getFrontController().getProjectManager().getActualContentPanel().validate();

				projectService.saveProjectInstance(epcc.getProjectInstance(), new AsyncCallback<IProjectFile>() {
					@Override
					public void onSuccess(IProjectFile projectFile) {
						epcc.getProjectInstance().setFile(projectFile);
						
						projectService.exportProjectInstance(epcc.getProjectInstance(), epcc.getFormat(), new AsyncCallback<String>() {
							@Override
							public void onSuccess(String result) {
								
								if (epcc.isCancelled()==false) componentManager.getFrontController().getDownloadManager().downloadFile(result);
								
								componentManager.getEventBus().fireEvent(new DisableLoadingStateEvent());
								//TODO: Fire ProjectExportedEvent
							}
							@Override
							public void onFailure(Throwable caught) {


								componentManager.getEventBus().fireEvent(new EditorErrorEvent("Project Service Error",caught.getMessage(),DefaultErrorLevel.ERROR));
							}
						});
					}
					@Override
					public void onFailure(Throwable caught) {

						componentManager.getEventBus().fireEvent(new EditorErrorEvent("Project Service Error",caught.getMessage(),DefaultErrorLevel.ERROR));
					}
				});
				
				
				
			} catch (ProjectValidationException e) {

				//notify user of validation failure
				componentManager.getEventBus().fireEvent(new EditorErrorEvent("Cannot export project", "Cannot save actual project because of the following errors:<br>"+e.getMessage(), DefaultErrorLevel.ERROR));
			}
			
			

		
		//EDIT
		}else if (event instanceof IEditProjectConfigurationCompleteEvent){
			IEditProjectConfigurationCompleteEvent editEvent = (IEditProjectConfigurationCompleteEvent) event;
			//when editing project configuration is complete

			//close edit window
			componentManager.getFrontController().getView().getEditProjectConfigurationView().close();
			
			//if a project instance is set and the list isnt null => commit data
			if (editEvent.getProjectInstance()!=null && editEvent.getProjectConfigurationDatas()!=null){
			
				//bind actual project instance with configuration data
				//TODO: bind other types of project configuration data
				for(IProjectConfigurationData data: editEvent.getProjectConfigurationDatas()){
					
					editEvent.getProjectInstance().bind(data);
					
				}
				
				//change contentPanel title if project name changed
				componentManager.getFrontController().getProjectManager().getActualContentPanel().setTitle(editEvent.getProjectInstance().getName());
				
			}
		}
	
	}
	
	
	private void dispatchPluginComponent(final IProjectInstance instance,boolean mustSave){
		
		IProjectPlugin<IProjectInstance> plugin = componentManager.getFrontController().getRegistry().getProjectPlugin(instance.getProjectType());
		
		//set components title to project instance name
		//once component have been dispatched
		final IContentPanelComponent component  = plugin.getContentPanelComponent(instance);
		
		componentManager.getComponentDispatcher().dispatch(component);
		component.setTitle(instance.getName());


		//mark content panel as saved or unsaved
		if (mustSave){
			componentManager.getFrontController().getContentManager().markContentPanelNotSaved(component);
		}else{
			componentManager.getFrontController().getContentManager().markContentPanelSaved(component);
		}
		
	}
	
	private void dispatchPluginComponent(List<IProjectInstance> instances,boolean mustSave){
		
		
		for(int i=0;i<instances.size();i++){
			
			dispatchPluginComponent(instances.get(i),mustSave);
			
		}
		
	}

}
